﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/os.h>

#include <nn/psm/detail/psm_Log.h>
#include <nn/usb/usb_Pm.h>
#include <nn/usb/pd/usb_Pd.h>

#include "psm_BatteryState.h" // TODO: 本来は BatteryState ではない値に格納されるべき情報を BatteryState に含めている。
#include "psm_ChargeArbiter.h"
#include "psm_IChargerDriver.h"
#include "psm_ErrorReporter.h"
#include "psm_OverlayNotificationSender.h"
#include "psm_SessionManager.h"
#include "psm_SettingsHolder-spec.nx.h"
#include "psm_SupplyRouteManager.h"
#include "psm_UsbPowerManager.h"

namespace nn { namespace psm { namespace driver { namespace detail {

namespace {

const ::nn::TimeSpan InputCurrentLimitWaitTime = ::nn::TimeSpan::FromMilliSeconds(420);

const auto BoostModeCurrentLimitDefault = 500;
const auto BoostModeCurrentLimitHigh    = 1300;

int GetUpdRdoCurrentMilliAmpere(const ::nn::usb::UsbPowerState& state) NN_NOEXCEPT
{
    auto current = state.rdo.Get<::nn::usb::pd::Rdo::MaximumOperatingCurrent>();
    return static_cast<int>(current * ::nn::usb::pd::RdoMilliAmpereUnit);
}

int GetUpdPdoCurrentMilliAmpere(const ::nn::usb::UsbPowerState& state) NN_NOEXCEPT
{
    auto current = state.pdo.Get<::nn::usb::pd::Pdo::MaximumCurrent>();
    return static_cast<int>(current * ::nn::usb::pd::PdoMilliAmpereUnit);
}

int GetUpdVoltageMilliVolt(const ::nn::usb::UsbPowerState& state) NN_NOEXCEPT
{
    auto voltage = state.pdo.Get<::nn::usb::pd::Pdo::Voltage>();
    return static_cast<int>(voltage * ::nn::usb::pd::PdoMilliVoltUnit);
}

int GetCurrentMilliAmpere(const ::nn::usb::UsbPowerState& state) NN_NOEXCEPT
{
    if (state.charger == ::nn::usb::UsbChargerType_Pd)
    {
        // PD ネゴシエーション成立
        return GetUpdRdoCurrentMilliAmpere(state);
    }
    else if ((state.charger == ::nn::usb::UsbChargerType_TypeC15)
        || (state.charger == ::nn::usb::UsbChargerType_TypeC30))
    {
        // PD ネゴシエーション非成立かつ Type-C
        return GetUpdPdoCurrentMilliAmpere(state);
    }
    else
    {
        // PD ネゴシエーション非成立かつ非 Type-C
        return state.current;
    }
}

int GetVoltageMilliVolt(const ::nn::usb::UsbPowerState& state) NN_NOEXCEPT
{
    if ( state.charger == ::nn::usb::UsbChargerType_Pd )
    {
        return GetUpdVoltageMilliVolt(state);
    }
    else
    {
        return state.voltage;
    }
}

bool IsSufficientVoltageToSupplyToVdd50B(const ::nn::usb::UsbPowerState& state) NN_NOEXCEPT
{
    const auto Vdd50BThresholdMilliVolt = 9000;

    return GetVoltageMilliVolt(state) >= Vdd50BThresholdMilliVolt;
}

int CalculateInputCurrentLimitMilliAmpere(const ::nn::usb::UsbPowerState& state, bool isVdd50BEnabled) NN_NOEXCEPT
{
    const auto VoltageThresholdLow  = 13000;
    const auto VoltageThresholdHigh = 15000;

    // PD ネゴシエーションで確定しなかった場合の電圧、電流値
    const auto VoltageDefault = 5000;

    const auto InputCurrentLimitDefault   = 500;
    const auto InputCurrentLimitLowMax    = 2000;
    const auto InputCurrentLimitMediumMax = 1500;
    const auto InputCurrentLimitHighMax   = 1200;

    // USB 未装着時と外部への給電時はデフォルト値
    if (state.role != ::nn::usb::UsbPowerRole_Sink)
    {
        return InputCurrentLimitDefault;
    }

    if (state.charger == ::nn::usb::UsbChargerType_Unknown)
    {
        return 0;
    }
    // Type-C
    else if (state.charger == ::nn::usb::UsbChargerType_Pd)
    {
        const auto MarginForVdd50BMilliWatt = 4500;

        auto voltage = GetUpdVoltageMilliVolt(state);
        auto current = GetUpdRdoCurrentMilliAmpere(state);

        // SIGLO-73946: VDD50B への供給時は Ilim は RDO 上の値から 4.5 W 分引いた値にします。
        // SIGLO-74631: 静的に VDD50B が無効化されている場合は本処理を通らないようにします
        if ( IsSufficientVoltageToSupplyToVdd50B(state) && isVdd50BEnabled )
        {
            current -= (MarginForVdd50BMilliWatt * 1000 / voltage);
            current = std::max(current, 0);
        }

        // PD ネゴシエーションで確定かつ 13V 未満
        if (voltage < VoltageThresholdLow)
        {
            return ::std::min(current, InputCurrentLimitLowMax);
        }
        // PD ネゴシエーションで確定かつ 15V 未満
        else if (voltage < VoltageThresholdHigh)
        {
            return ::std::min(current, InputCurrentLimitMediumMax);
        }
        // PD ネゴシエーションで確定かつ 15V 以上
        else
        {
            return ::std::min(current, InputCurrentLimitHighMax);
        }
    }
    else if (state.charger == ::nn::usb::UsbChargerType_TypeC15)
    {
        auto voltage = GetUpdVoltageMilliVolt(state);
        auto current = GetUpdPdoCurrentMilliAmpere(state);
        if (voltage == VoltageDefault && current == 1500)
        {
            // InputCurrentLimitLowMax でクランプ。
            return ::std::min(current, InputCurrentLimitLowMax);
        }
        else
        {
            // 異常系。Charger と Pdo の値に不整合ありとして InputCurrentLimitDefault を返す。
            NN_DETAIL_PSM_WARN("%d mA and %d mV don't match UsbChargerType_TypeC15.\n", current, voltage);
            return InputCurrentLimitDefault;
        }
    }
    else if (state.charger == ::nn::usb::UsbChargerType_TypeC30)
    {
        auto voltage = GetUpdVoltageMilliVolt(state);
        auto current = GetUpdPdoCurrentMilliAmpere(state);
        if (voltage == VoltageDefault && current == 3000)
        {
            // InputCurrentLimitLowMax でクランプ。
            return ::std::min(current, InputCurrentLimitLowMax);
        }
        else
        {
            // 異常系。Charger と Pdo の値に不整合ありとして InputCurrentLimitDefault を返す。
            NN_DETAIL_PSM_WARN("%d mA and %d mV don't match UsbChargerType_TypeC30.\n", current, voltage);
            return InputCurrentLimitDefault;
        }
    }
    // 非 Type-C
    else
    {
        return state.current;
    }
}

ChargerType DeriveChargerType(
    const ::nn::usb::UsbPowerState& pmState,
    const ::nn::usb::pd::Status& pdState) NN_NOEXCEPT
{
    auto error = pdState.GetError();
    if (!(error == ::nn::usb::pd::StatusError_None || error == ::nn::usb::pd::StatusError_CradleUsbHubUndetected))
    {
        return ChargerType_NotSupported;
    }
    if (pmState.role != ::nn::usb::UsbPowerRole_Sink)
    {
        return ChargerType_Unconnected;
    }
    if (pmState.charger == ::nn::usb::UsbChargerType_Unknown)
    {
        return ChargerType_NotSupported;
    }
    if (pmState.charger != ::nn::usb::UsbChargerType_Pd)
    {
        return ChargerType_LowPower;
    }

    const auto EnoughPowerMilliWatThreshold = 39000;
    auto voltage = GetUpdVoltageMilliVolt(pmState);
    auto current = GetUpdRdoCurrentMilliAmpere(pmState);
    if (voltage * current / 1000 >= EnoughPowerMilliWatThreshold)
    {
        return ChargerType_EnoughPower;
    }

    return ChargerType_LowPower;
}

} // namespace

UsbPowerManager::UsbPowerManager() NN_NOEXCEPT
    : m_pChargeArbiter(nullptr)
    , m_pSupplyRouteManager(nullptr)
    , m_pBatteryState(nullptr)
    , m_pChargerDriver(nullptr)
    , m_pNotificationSender(nullptr)
    , m_pSessionManager(nullptr)
    , m_UpdSession()
    , m_UpdEvent()
    , m_UpmClient()
    , m_IsOtgAllowed(false)
    , m_IsPowerRequestEnabled(false)
    , m_IsEnoughPowerChargeEmulationEnabled(false)
    , m_EnoughPowerChargeEmulationEvent()
    , m_SufficientVoltageToSupplyToVdd50B(false)
    , m_CorrectDirectionToSupplyToVdd50B(true)
    , m_pSettingsHolder(nullptr)
{
}

UsbPowerManager::~UsbPowerManager() NN_NOEXCEPT
{
    // 何もしない。
}

void UsbPowerManager::Initialize(
    ChargeArbiter* pChargeArbiter,
    SupplyRouteManager* pSupplyRouteManager,
    BatteryState* pBatteryState,
    IChargerDriver* pChargerDriver,
    SessionManager* pSessionManager,
    OverlayNotificationSender* pNotificationSender,
    const SettingsHolder* pSettingsHolder) NN_NOEXCEPT
{
    m_pChargeArbiter = pChargeArbiter;
    m_pSupplyRouteManager = pSupplyRouteManager;
    m_pBatteryState = pBatteryState;
    m_pChargerDriver = pChargerDriver;
    m_pSessionManager = pSessionManager;
    m_pNotificationSender = pNotificationSender;
    m_pSettingsHolder = pSettingsHolder;

    ::nn::usb::pd::Initialize();
    ::nn::usb::pd::OpenSession(&m_UpdSession);
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::usb::pd::BindNoticeEvent(&m_UpdEvent, &m_UpdSession));

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_UpmClient.Initialize());

    m_IsOtgAllowed = false;
    m_IsPowerRequestEnabled = false;
    m_IsEnoughPowerChargeEmulationEnabled = false;

    ::nn::os::CreateSystemEvent(&m_EnoughPowerChargeEmulationEvent, ::nn::os::EventClearMode_ManualClear, false);

    // チャージャと VDD50A, B の状態を更新します。
    UpdateChargerAndVdd50();
}

void UsbPowerManager::Finalize() NN_NOEXCEPT
{
    ::nn::os::DestroySystemEvent(&m_EnoughPowerChargeEmulationEvent);

    m_UpmClient.Finalize();

    ::nn::usb::pd::UnbindNoticeEvent(&m_UpdSession);
    ::nn::usb::pd::CloseSession(&m_UpdSession);
    ::nn::usb::pd::Finalize();
}

::nn::Result UsbPowerManager::AllowOtg() NN_NOEXCEPT
{
    m_IsOtgAllowed = true;
    NN_RESULT_DO(UpdatePowerRequest());

    NN_RESULT_SUCCESS;
}

::nn::Result UsbPowerManager::DisallowOtg() NN_NOEXCEPT
{
    m_IsOtgAllowed = false;
    NN_RESULT_DO(UpdatePowerRequest());

    NN_RESULT_SUCCESS;
}

::nn::Result UsbPowerManager::EnableEnoughPowerChargeEmulation() NN_NOEXCEPT
{
    m_IsEnoughPowerChargeEmulationEnabled = true;
    ::nn::os::SignalSystemEvent(&m_EnoughPowerChargeEmulationEvent);

    NN_RESULT_SUCCESS;
}

::nn::Result UsbPowerManager::DisableEnoughPowerChargeEmulation() NN_NOEXCEPT
{
    m_IsEnoughPowerChargeEmulationEnabled = false;
    ::nn::os::SignalSystemEvent(&m_EnoughPowerChargeEmulationEvent);

    NN_RESULT_SUCCESS;
}

::nn::os::SystemEventType* UsbPowerManager::GetPowerEvent() NN_NOEXCEPT
{
    return m_UpmClient.GetPowerEvent();
}

::nn::os::SystemEventType* UsbPowerManager::GetPowerDeliveryNoticeEvent() NN_NOEXCEPT
{
    return &m_UpdEvent;
}

::nn::os::SystemEventType* UsbPowerManager::GetEnoughPowerChargeEmulationEvent() NN_NOEXCEPT
{
    return &m_EnoughPowerChargeEmulationEvent;
}

::nn::Result UsbPowerManager::Update() NN_NOEXCEPT
{
    // TODO: UpdateVdd50() が Vdd50SwitchingTime の期間ブロックするので
    //       問題になるようであればスレッドの分離を検討

    UpdateChargerAndVdd50();

    NN_RESULT_SUCCESS;
}

// 他の Process 関数とは別のスレッドに呼ばれる想定
::nn::Result UsbPowerManager::ProcessPowerDeliveryNoticeEvent() NN_NOEXCEPT
{
    ::nn::usb::pd::Notice notice;
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::usb::pd::GetNotice(&notice, &m_UpdSession));
    ::nn::usb::pd::Status status;
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::usb::pd::GetStatus(&status, &m_UpdSession));

    // m_IsPowerRequestEnabled は EnoughPowerChargeEmulation イベントのハンドリング等別スレッドから更新される
    if ( m_IsPowerRequestEnabled && notice.IsRequestNotice() )
    {
        bool isOtgChanged = false;

        switch (status.GetRequest())
        {
        case ::nn::usb::pd::StatusRequest_PowerOutput:
            GetErrorReporter().SetOtgRequested(true);
            isOtgChanged = true;
            // OTG 開始時に突入電流による boost fault の発生を防ぐため
            // 一時的に出力電流の制限を緩める（SIGLO-39784）
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                m_pChargerDriver->SetBoostModeCurrentLimit(
                    BoostModeCurrentLimitHigh));
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_pChargeArbiter->StartOtgRequest());
            m_CorrectDirectionToSupplyToVdd50B = false;
            NN_DETAIL_PSM_TRACE("VBUS power output.\n");
            break;
        case ::nn::usb::pd::StatusRequest_PowerInput:
            GetErrorReporter().SetOtgRequested(false);
            isOtgChanged = true;
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_pChargeArbiter->StopOtgRequest());
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                m_pChargerDriver->SetBoostModeCurrentLimit(
                    BoostModeCurrentLimitDefault));
            m_CorrectDirectionToSupplyToVdd50B = true;
            NN_DETAIL_PSM_TRACE("VBUS power input.\n");
            break;
        default:
            break;
        }

        if ( isOtgChanged )
        {
            m_pSupplyRouteManager->SetVdd50BAvailableWithoutUpdate(
                m_SufficientVoltageToSupplyToVdd50B && m_CorrectDirectionToSupplyToVdd50B);

            NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::usb::pd::ReplyPowerRequest(&m_UpdSession, true));
        }
    }

    if ( notice.IsActiveNotice() && status.IsActive() && m_pChargeArbiter->IsOtgRequested() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            m_pChargerDriver->SetBoostModeCurrentLimit(BoostModeCurrentLimitDefault));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result UsbPowerManager::ProcessEnoughPowerChargeEmulationEvent() NN_NOEXCEPT
{
    if ( m_IsEnoughPowerChargeEmulationEnabled )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_pSupplyRouteManager->EnableForceVdd50A());
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_pSupplyRouteManager->DisableForceVdd50A());
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        m_pChargerDriver->SetEnoughPowerChargeEmulation(m_IsEnoughPowerChargeEmulationEnabled));

    NN_ABORT_UNLESS_RESULT_SUCCESS(UpdatePowerRequest());

    UpdateChargerAndVdd50();

    NN_RESULT_SUCCESS;
}

void UsbPowerManager::UpdateChargerAndVdd50() NN_NOEXCEPT
{
    ::nn::usb::UsbPowerState pmState;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_UpmClient.GetPowerState(&pmState));

    ::nn::usb::pd::Status pdState;
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::usb::pd::GetStatus(&pdState, &m_UpdSession));

    // 入力電流の設定には USB 装着から 420 ミリ秒以上待機する必要があるが
    // VDD50B に切り替わった場合には既に 500 ミリ秒待機しているので待機せずに
    // 据置モードの遷移時間や他のイベントの処理までの時間を延ばさないようにする（SIGLO-38044）

    // VDD50B への切り替えで判定せずに SystemTick で判定する。（SIGLO-41779）
    auto begin = ::nn::os::GetSystemTick();

    UpdateChargerType(pmState, pdState);

    m_SufficientVoltageToSupplyToVdd50B = IsSufficientVoltageToSupplyToVdd50B(pmState);
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_pSupplyRouteManager->SetVdd50BAvailable(
        m_SufficientVoltageToSupplyToVdd50B && m_CorrectDirectionToSupplyToVdd50B));

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    auto chargerType = m_pBatteryState->GetChargerType();
    if ( chargerType != ChargerType_Unconnected && chargerType != ChargerType_NotSupported )
    {
        auto actualDuration = (::nn::os::GetSystemTick() - begin).ToTimeSpan();
        if ( InputCurrentLimitWaitTime > actualDuration )
        {
            ::nn::os::SleepThread(InputCurrentLimitWaitTime - actualDuration);
        }
    }
#endif

    UpdateInputVoltageLimit(pmState);
    UpdateInputCurrentLimit(pmState);
    UpdatePowerSupply();

    GetErrorReporter().SetPowerRole(pmState.role);
    GetErrorReporter().SetPowerSupplyType(pmState.charger);
    GetErrorReporter().SetPowerSupplyVoltage(GetVoltageMilliVolt(pmState));
    GetErrorReporter().SetPowerSupplyCurrent(GetCurrentMilliAmpere(pmState));
}

void UsbPowerManager::UpdateChargerType(const ::nn::usb::UsbPowerState& pmState, const ::nn::usb::pd::Status& pdState) NN_NOEXCEPT
{
    ChargerType chargerType;
    // 給電エミュレーションが有効の場合には強制的に EnoughPower
    if (m_IsEnoughPowerChargeEmulationEnabled)
    {
        chargerType = ChargerType_EnoughPower;
    }
    else
    {
        chargerType = DeriveChargerType(pmState, pdState);
    }

    // 以前と同じであれば更新しない
    if (m_pBatteryState->GetChargerType() == chargerType)
    {
        return;
    }

    m_pBatteryState->SetChargerType(chargerType);
    m_pNotificationSender->NotifyChargerStateChange();
    m_pSessionManager->SignalStateChangeEvents(
        SessionManager::EventType_ChargerTypeChange);
}

void UsbPowerManager::UpdateInputVoltageLimit(const ::nn::usb::UsbPowerState& usbPowerState) NN_NOEXCEPT
{
    auto inputVoltageLimitMilliVolt = 4360;

    // 以下の理由から USB PD 9V 以上の場合のみ VINDPM を引き下げる
    // - 9V 以上の電圧には VINDPM は効果がない
    // - スロットリングの応答速度を維持したい

    if ( (usbPowerState.charger == ::nn::usb::UsbChargerType_Pd)
        && (GetUpdVoltageMilliVolt(usbPowerState) >= 9000) )
    {
        inputVoltageLimitMilliVolt = 3880;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_pChargerDriver->SetInputVoltageLimitMilliVolt(inputVoltageLimitMilliVolt));
    NN_DETAIL_PSM_TRACE("ChargerInputVoltageLimit = %d mV.\n", inputVoltageLimitMilliVolt);
}

void UsbPowerManager::UpdateInputCurrentLimit(const ::nn::usb::UsbPowerState& state) NN_NOEXCEPT
{
    int inputCurrentLimit;
    // 給電エミュレーションが有効の場合には強制的に最大入力電流 1.2 A
    if (m_IsEnoughPowerChargeEmulationEnabled)
    {
        inputCurrentLimit = 1200;
    }
    else
    {
        inputCurrentLimit = CalculateInputCurrentLimitMilliAmpere(state, m_pSettingsHolder->IsVdd50BEnabled());
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        m_pChargerDriver->SetInputCurrentLimit(inputCurrentLimit));

    NN_DETAIL_PSM_TRACE("ChargerInputCurrentLimit = %d mA.\n", inputCurrentLimit);
}

void UsbPowerManager::UpdatePowerSupply() NN_NOEXCEPT
{
    bool isEnoughPowerSupplied =
        (m_pBatteryState->GetChargerType() == ChargerType_EnoughPower);

    // 以前と同じであれば更新しない
    if (m_pBatteryState->IsEnoughPowerSupplied() == isEnoughPowerSupplied)
    {
        return;
    }

    m_pBatteryState->SetEnoughPowerSupplied(isEnoughPowerSupplied);
    m_pSessionManager->SignalStateChangeEvents(
        SessionManager::EventType_PowerSupplyChange);
}

::nn::Result UsbPowerManager::UpdatePowerRequest() NN_NOEXCEPT
{
    auto isEnabled = m_IsOtgAllowed && !m_IsEnoughPowerChargeEmulationEnabled;
    if ( isEnabled == m_IsPowerRequestEnabled )
    {
        NN_RESULT_SUCCESS;
    }

    // PowerStateManager による AllowOtg, DisallowOtg 失敗時に PowerStateManager が psc に対して result を返せるように NN_RESULT_DO でのハンドリングを維持します。
    m_IsPowerRequestEnabled = isEnabled;
    if ( isEnabled )
    {
        NN_RESULT_DO(::nn::usb::pd::EnablePowerRequestNotice(&m_UpdSession));
    }
    else
    {
        NN_RESULT_DO(::nn::usb::pd::DisablePowerRequestNotice(&m_UpdSession));
        NN_RESULT_DO(m_pChargeArbiter->StopOtgRequest());

        m_CorrectDirectionToSupplyToVdd50B = true;
        m_pSupplyRouteManager->SetVdd50BAvailableWithoutUpdate(
            m_SufficientVoltageToSupplyToVdd50B && m_CorrectDirectionToSupplyToVdd50B);
    }

    NN_RESULT_SUCCESS;
}

}}}} // namespace nn::psm::driver::detail
