﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include "psm_ISupplyRouteDriver.h"
#include "psm_SettingsHolder-spec.nx.h"

namespace nn { namespace psm { namespace driver { namespace detail {

class SupplyRouteManager
{
    NN_DISALLOW_COPY(SupplyRouteManager);
    NN_DISALLOW_MOVE(SupplyRouteManager);

public:
    SupplyRouteManager() NN_NOEXCEPT;

    ~SupplyRouteManager() NN_NOEXCEPT;

    ::nn::Result Initialize(ISupplyRouteDriver* pSupplyRouteDriver, const SettingsHolder* pSettingsHolder) NN_NOEXCEPT;

    void Finalize() NN_NOEXCEPT;

    //! VDD50B が利用可能か設定します。
    ::nn::Result SetVdd50BAvailable(bool isAvailable) NN_NOEXCEPT;

    //! VDD50B が利用可能か設定します。内部で VDD50 の状態は更新されません。
    void SetVdd50BAvailableWithoutUpdate(bool isAvailable) NN_NOEXCEPT;

    //! VDD50 を有効化します。
    ::nn::Result EnableVdd50() NN_NOEXCEPT;

    //! VDD50 を無効化します。
    ::nn::Result DisableVdd50() NN_NOEXCEPT;

    //! VDD50A を強制有効します。
    ::nn::Result EnableForceVdd50A() NN_NOEXCEPT;

    //! VDD50A の強制有効を解除します。
    ::nn::Result DisableForceVdd50A() NN_NOEXCEPT;

    //! VDD50 の制御を行う状態にします。
    void EnableSupplyRouteControl() NN_NOEXCEPT;

    //! VDD50 の制御を行わない状態にします。
    void DisableSupplyRouteControl() NN_NOEXCEPT;

private:
    //! 各公開関数処理後に実行され SupplyRouteDriver を更新します。
    ::nn::Result Update() NN_NOEXCEPT;

    //! 引数の中から適切な SupplyRoute を選択して返します。
    SupplyRoute GetAppropriateSupplyRoute(SupplyRoute primaryRoute, SupplyRoute secondaryRoute) NN_NOEXCEPT;

    //! 引数の SupplyRoute に設定可能かを確認します。
    bool CanUpdateSupplyRoute(SupplyRoute supplyRoute) NN_NOEXCEPT;

private:
    //! 給電経路ドライバへのポインタ
    ISupplyRouteDriver* m_pSupplyRouteDriver;

    //! デバッグ・プラットフォーム設定
    const SettingsHolder* m_pSettingsHolder;

    //! データアクセスの排他
    ::nn::os::Mutex m_DataMutex;

    //! VDD50B が利用可能かどうか
    bool m_IsVdd50BAvailable;

    //! VDD50 が有効化されているかどうか
    bool m_IsVdd50Enabled;

    //! VDD50A が強制有効されているかどうか
    bool m_IsForceVdd50AEnabled;

    //! VDD50 の制御を行うかどうか
    bool m_SupplyRouteControlEnabled;
};

}}}} // namespace nn::psm::driver::detail
