﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>

#include <nn/psc/psc_PmModule.h>
#include <nn/psm/psm_ManufactureTypes.h>
#include <nn/psm/psm_SystemTypes.h>
#include <nn/psm/psm_SystemProcessTypes.h>
#include <nn/psm/driver/psm_Types.h>
#include <nn/util/util_TypedStorage.h>

#include "psm_EventMonitor.h"
#include "psm_ChargerDriver.h"
#include "psm_ChargerDriverStub.h"
#include "psm_ChargeArbiter.h"
#include "psm_FuelGaugeDriver.h"
#include "psm_FuelGaugeDriverStub.h"
#include "psm_IChargerDriver.h"
#include "psm_IFuelGaugeDriver.h"
#include "psm_ISupplyRouteDriver.h"
#include "psm_ModuleHolder.h"
#include "psm_SessionManager.h"
#include "psm_SupplyRouteDriver.h"
#include "psm_SupplyRouteDriverStub.h"

namespace nn { namespace psm { namespace driver { namespace detail {

class PowerSupplyManager final
{
    NN_DISALLOW_COPY(PowerSupplyManager);
    NN_DISALLOW_MOVE(PowerSupplyManager);

private:
    union ChargerDriverStorage
    {
        ::nn::util::TypedStorage<ChargerDriver, sizeof(ChargerDriver), NN_ALIGNOF(ChargerDriver)> driver;
        ::nn::util::TypedStorage<ChargerDriverStub, sizeof(ChargerDriverStub), NN_ALIGNOF(ChargerDriverStub)> stub;
    };

    union FuelGaugeDriverStorage
    {
        ::nn::util::TypedStorage<FuelGaugeDriver, sizeof(FuelGaugeDriver), NN_ALIGNOF(FuelGaugeDriver)> driver;
        ::nn::util::TypedStorage<FuelGaugeDriverStub, sizeof(FuelGaugeDriverStub), NN_ALIGNOF(FuelGaugeDriverStub)> stub;
    };

    union SupplyRouteDriverStorage
    {
        ::nn::util::TypedStorage<SupplyRouteDriver, sizeof(SupplyRouteDriver), NN_ALIGNOF(SupplyRouteDriver)> driver;
        ::nn::util::TypedStorage<SupplyRouteDriverStub, sizeof(SupplyRouteDriverStub), NN_ALIGNOF(SupplyRouteDriverStub)> stub;
    };

public:
    PowerSupplyManager() NN_NOEXCEPT;

    ~PowerSupplyManager() NN_NOEXCEPT;

    //! チャージャを初期化します。
    void Initialize() NN_NOEXCEPT;

    //! チャージャの終了処理をします。
    void Finalize() NN_NOEXCEPT;

    //! 電池残量を取得します。
    ::nn::Result GetBatteryChargePercentage(int* pOutValue) NN_NOEXCEPT;

    //! 未加工の電池残量を取得します。
    ::nn::Result GetRawBatteryChargePercentage(double* pOutValue) NN_NOEXCEPT;

    //! 電池劣化量を取得します。
    ::nn::Result GetBatteryAgePercentage(double* pOutValue) NN_NOEXCEPT;

    //! 本体に接続された充電器の種別を取得します。
    ::nn::Result GetChargerType(ChargerType* pOutValue) NN_NOEXCEPT;

    //! 電池電圧状態を取得します。
    ::nn::Result GetBatteryVoltageState(BatteryVoltageState* pOutValue) NN_NOEXCEPT;

    //! 充電を有効化します。
    ::nn::Result EnableBatteryCharging() NN_NOEXCEPT;

    //! 充電を無効化します。
    ::nn::Result DisableBatteryCharging() NN_NOEXCEPT;

    //! 充電が有効化されているかどうかを取得します。
    ::nn::Result IsBatteryChargingEnabled(bool* pOutValue) NN_NOEXCEPT;

    //! コントローラへの給電の使用権を取得します。
    ::nn::Result AcquireControllerPowerSupply() NN_NOEXCEPT;

    //! コントローラへの給電の使用権を解放します。
    ::nn::Result ReleaseControllerPowerSupply() NN_NOEXCEPT;

    //! セッションをオープンします。
    ::nn::Result OpenSession(Session* pOutSession) NN_NOEXCEPT;

    //! セッションをクローズします。
    ::nn::Result CloseSession(const Session& session) NN_NOEXCEPT;

    //! 本体に接続された充電器や電池電圧レベルの状態の変化とイベントを紐付けます。
    ::nn::Result BindStateChangeEvent(
        ::nn::os::SystemEventType* pEvent,
        const Session& session) NN_NOEXCEPT;

    //! 本体に接続された充電器や電池電圧レベルの状態の変化とイベントの紐付けを解除します。
    ::nn::Result UnbindStateChangeEvent(const Session& session) NN_NOEXCEPT;

    //! 充電器の変化によるイベントのシグナルを有効化するかどうかを設定します。
    ::nn::Result SetChargerTypeChangeEventEnabled(
         const Session& session,
         bool isEnabled) NN_NOEXCEPT;

    //! 給電能力の変化によるイベントのシグナルを有効化するかどうかを設定します。
    ::nn::Result SetPowerSupplyChangeEventEnabled(
         const Session& session,
         bool isEnabled) NN_NOEXCEPT;

    //! 電池電圧の状態の変化によるイベントのシグナルを有効化するかどうかを設定します。
    ::nn::Result SetBatteryVoltageStateChangeEventEnabled(
         const Session& session,
         bool isEnabled) NN_NOEXCEPT;

    //! 給電エミュレーションを有効にします。
    ::nn::Result EnableEnoughPowerChargeEmulation() NN_NOEXCEPT;

    //! 給電エミュレーションを無効にします。
    ::nn::Result DisableEnoughPowerChargeEmulation() NN_NOEXCEPT;

    //! 高速充電を有効にします。
    ::nn::Result EnableFastBatteryCharging() NN_NOEXCEPT;

    //! 高速充電を無効にします。
    ::nn::Result DisableFastBatteryCharging() NN_NOEXCEPT;

    //! 十分な給電がされているかどうかを取得します。
    ::nn::Result IsEnoughPowerSupplied(bool* pOutValue) NN_NOEXCEPT;

    //! 電池の残量ずれを検出して補正を行ったことを伝えるシステムイベントを取得します。
    ::nn::Result GetBatteryChargeCalibratedEvent(::nn::os::SystemEventType** pOutSystemEventPointer) NN_NOEXCEPT;

    //! テスト用途で PmModule へのポインタを返します。
    ::nn::Result GetPmModuleForTest(::nn::psc::PmModule** pOutPmModulePointer) NN_NOEXCEPT;

    //! テスト用途で PSC のハンドリング完了イベントへのポインタを返します。
    ::nn::Result GetPscEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT;

    //! テスト用途で電池残量 IC のハンドリング完了イベントへのポインタを返します。
    ::nn::Result GetFuelGaugeEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT;

    //! テスト用途で電池残量計定期的データ受信のハンドリング完了イベントへのポインタを返します。
    ::nn::Result GetFuelGaugeTimerEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT;

    //! テスト用途で USB PD のハンドリング完了イベントへのポインタを返します。
    ::nn::Result GetUsbPdEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT;

    //! テスト用途で USB PM のハンドリング完了イベントへのポインタを返します。
    ::nn::Result GetUsbPmEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT;

    //! テスト用途で給電エミュレーションのハンドリング完了イベントへのポインタを返します。
    ::nn::Result GetEnoughPowerChargeEmulationEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT;

#if defined(PSM_DETAIL_MANUFACTURE_BUILD)

    //! 給電経路制御機能を有効にします。
    void EnableSupplyRouteControl() NN_NOEXCEPT;

    //! 給電経路制御機能を無効にします。
    void DisableSupplyRouteControl() NN_NOEXCEPT;

    //! 給電経路を設定します。
    void SetSupplyRoute(::nn::psm::SupplyRoute supplyRoute) NN_NOEXCEPT;
#endif

private:
    //! チャージャドライバ
    ChargerDriverStorage m_ChargerDriverStorage;
    IChargerDriver* m_pChargerDriver;

    //! 電池残量計ドライバ
    FuelGaugeDriverStorage m_FuelGaugeDriverStorage;
    IFuelGaugeDriver* m_pFuelGaugeDriver;

    //! 給電経路ドライバ
    SupplyRouteDriverStorage m_SupplyRouteDriverStorage;
    ISupplyRouteDriver* m_pSupplyRouteDriver;

    //! 充電調停用アービタ
    ChargeArbiter m_ChargeArbiter;

    //! セッション管理
    SessionManager m_SessionManager;

    //! イベント監視スレッド
    EventMonitor m_EventMonitor;

    //! イベント監視スレッド
    EventMonitor m_EventMonitorForUsbPowerDelivery;

    //! プラットフォームへの依存の強いモジュールおよびハンドラを保持するオブジェクト
    ModuleHolder m_ModuleHolder;

    //! デバッグ・プラットフォーム設定
    SettingsHolder m_SettingsHolder;

    //! イベント監視スレッドのスタック領域のサイズ
    static const int EventMonitorStackSize = ::nn::os::ThreadStackAlignment;

    //! イベント監視スレッドのスタック領域
    static NN_OS_ALIGNAS_THREAD_STACK
        uint8_t s_EventMonitorStack[EventMonitorStackSize];

    //! イベント監視スレッドのスタック領域
    static NN_OS_ALIGNAS_THREAD_STACK
        uint8_t s_EventMonitorForUsbPowerDeliveryStack[EventMonitorStackSize];
};

PowerSupplyManager& GetPowerSupplyManager() NN_NOEXCEPT;

}}}} // namespace nn::psm::driver::detail
