﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/psc/psc_PmModule.h>

#include "Handlers/psm_EventHandler.h"
#include "psm_BatteryChargeManager.h"
#include "psm_BatteryState.h"
#include "psm_ChargeArbiter.h"
#include "psm_IChargerDriver.h"
#include "psm_IFuelGaugeDriver.h"
#include "psm_IFuelGaugeParameterManager.h"
#include "psm_PowerState.h"
#include "psm_SettingsHolder-spec.nx.h"
#include "psm_SupplyRouteManager.h"

namespace nn { namespace psm { namespace driver { namespace detail {

// 循環参照解決用の宣言
class ModuleHolder;

class PowerStateManager final
{
    NN_DISALLOW_COPY(PowerStateManager);
    NN_DISALLOW_MOVE(PowerStateManager);

public:
    PowerStateManager() NN_NOEXCEPT
    {
        // 何もしない
    }

    ~PowerStateManager() NN_NOEXCEPT
    {
        // 何もしない
    }

    //! 初期化します。
    ::nn::Result Initialize(
        IChargerDriver* pChargerDriver,
        IFuelGaugeDriver* pFuelGaugeDriver,
        ChargeArbiter* pChargeArbiter,
        BatteryChargeManager* pBatteryChargeManager,
        IBatteryState* pBatteryState,
        SupplyRouteManager* pSupplyRouteManager,
        IFuelGaugeParameterManager* pFuelGaugeParameterManager,
        ModuleHolder* pModuleHolder,
        const SettingsHolder* pSettingsHolder) NN_NOEXCEPT;

    //! 終了処理をします。
    void Finalize() NN_NOEXCEPT;

    //! コントローラへの給電の使用権を取得します。
    ::nn::Result AcquireControllerPowerSupply() NN_NOEXCEPT;

    //! コントローラへの給電の使用権を解放します。
    ::nn::Result ReleaseControllerPowerSupply() NN_NOEXCEPT;

    //! イベントがシグナル状態の場合に適切な処理をした上でイベントをクリアします。
    ::nn::Result Update(::nn::psc::PmState state, ::nn::psc::PmFlagSet flags) NN_NOEXCEPT;

private:
    //! FullAwake ステートに遷移する準備をします。
    ::nn::Result HandleFullAwake(::nn::psc::PmFlagSet flags) NN_NOEXCEPT;

    //! MinimumAwake ステートに遷移する準備をします。
    ::nn::Result HandleMinimumAwake(::nn::psc::PmFlagSet flags) NN_NOEXCEPT;

    //! SleepReady ステートに遷移する準備をします。
    ::nn::Result HandleSleepReady(::nn::psc::PmFlagSet flags) NN_NOEXCEPT;

    //! ShutdownReady ステートに遷移する準備をします。
    ::nn::Result HandleShutdownReady(::nn::psc::PmFlagSet flags) NN_NOEXCEPT;

    //! 内部状態を更新します。
    ::nn::Result UpdateState() NN_NOEXCEPT;

    //! 条件を満たした場合に内部状態を SleepDischarge に遷移します。
    ::nn::Result TryEnterSleepDischargeState() NN_NOEXCEPT;

    //! 条件を満たした場合に内部状態を SleepCharge に遷移します。
    ::nn::Result TryEnterSleepChargeState() NN_NOEXCEPT;

    //! 内部状態を SleepDischarge に遷移します。
    ::nn::Result EnterSleepDischargeState() NN_NOEXCEPT;

    //! 内部状態を SleepCharge に遷移します。
    ::nn::Result EnterSleepChargeState() NN_NOEXCEPT;

    //! 内部状態を Awake に遷移します。
    ::nn::Result EnterAwakeState() NN_NOEXCEPT;

    //! 現在の内部状態にもとづいて割り込みと VDD50 の設定をします。
    ::nn::Result SetupInterruptAndVdd50() NN_NOEXCEPT;

    //! HiZ モードの割り込み設定を有効にします。
    ::nn::Result EnableHiZModeInterrupt() NN_NOEXCEPT;

    //! HiZ モードの割り込み設定を無効にします。
    ::nn::Result DisableHiZModeInterrupt() NN_NOEXCEPT;

    //! 電池残量低下の割り込み設定を有効にします。
    ::nn::Result EnableLowBatteryInterrupt() NN_NOEXCEPT;

    //! 電池残量低下の割り込み設定を無効にします。
    ::nn::Result DisableLowBatteryInterrupt() NN_NOEXCEPT;

    //! VDD50 の有効化を設定します。
    ::nn::Result SetVdd50Enabled(bool isVdd50Enabled) NN_NOEXCEPT;

private:
    //! チャージャドライバ
    IChargerDriver* m_pChargerDriver;

    //! 電池残量計ドライバ
    IFuelGaugeDriver* m_pFuelGaugeDriver;

    //! 充電調停用アービタ
    ChargeArbiter* m_pChargeArbiter;

    //! 電池状態
    IBatteryState* m_pBatteryState;

    //! VDD50 管理
    SupplyRouteManager* m_pSupplyRouteManager;

    //! 電池残量管理
    BatteryChargeManager* m_pBatteryChargeManager;

    //! 電源ステート
    PowerState m_PowerState;

    //! コントローラへの電源供給が必要かどうか
    ::std::atomic<bool> m_IsControllerPowerSupplyNeeded;

    //! イベント監視の有無を設定する為の親オブジェクトへのポインタ
    ModuleHolder* m_pModuleHolder;

    //! 電池残量計パラメータ管理
    IFuelGaugeParameterManager* m_pFuelGaugeParameterManager;

    //! デバッグ・プラットフォーム設定
    const SettingsHolder* m_pSettingsHolder;
};

}}}} // namespace nn::psm::driver::detail
