﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>

#include <nn/nn_Common.h>

#include <nn/psc/psc_PmModule.h>
#include <nn/psm/psm_SystemTypes.h>
#include <nn/psm/psm_SystemProcessTypes.h>

#include "Handlers/psm_ChargerEventHandler.h"
#include "Handlers/psm_FuelGaugeEventHandler.h"
#include "Handlers/psm_PeriodicCustomParameterSaver.h"
#include "Handlers/psm_PeriodicStateUpdater.h"
#include "Handlers/psm_PeriodicSystemReportSaver.h"
#include "Handlers/psm_SystemPowerStateEventHandler.h"
#include "Handlers/psm_SystemPowerStateEventHandlerLow.h"
#include "Handlers/psm_UsbPowerDeliveryEventHandler.h"
#include "Handlers/psm_UsbPowerEventHandler.h"
#include "psm_BatteryChargeManager.h"
#include "psm_EventMonitor.h"
#include "psm_BatteryTemperatureManager.h"
#include "psm_BatteryVoltageLevelForCharge.h"
#include "psm_ChargeArbiter.h"
#include "psm_FuelGaugeParameterManager.h"
#include "psm_FuelGaugeParameterManagerStub.h"
#include "psm_IChargerDriver.h"
#include "psm_IFuelGaugeDriver.h"
#include "psm_IFuelGaugeParameterManager.h"
#include "psm_ISupplyRouteDriver.h"
#include "psm_OverlayNotificationSender.h"
#include "psm_IBatteryState.h"
#include "psm_IUsbPowerManager.h"
#include "psm_PowerStateManager.h"
#include "psm_SessionManager.h"
#include "psm_SettingsHolder-spec.nx.h"
#include "psm_SupplyRouteManager.h"
#include "psm_UsbPowerManager.h"
#include "psm_UsbPowerManagerStub.h"

namespace nn { namespace psm { namespace driver { namespace detail {

//! プラットフォームへの依存の強いモジュールおよびハンドラを保持するクラス
class ModuleHolder final
{
    NN_DISALLOW_COPY(ModuleHolder);
    NN_DISALLOW_MOVE(ModuleHolder);

private:
    union UsbPowerManagerStorage
    {
        ::nn::util::TypedStorage<UsbPowerManager, sizeof(UsbPowerManager), NN_ALIGNOF(UsbPowerManager)> driver;
        ::nn::util::TypedStorage<UsbPowerManagerStub, sizeof(UsbPowerManagerStub), NN_ALIGNOF(UsbPowerManagerStub)> stub;
    };

    union FuelGaugeParameterManagerStorage
    {
        ::nn::util::TypedStorage<FuelGaugeParameterManager, sizeof(FuelGaugeParameterManager), NN_ALIGNOF(FuelGaugeParameterManager)> driver;
        ::nn::util::TypedStorage<FuelGaugeParameterManagerStub, sizeof(FuelGaugeParameterManagerStub), NN_ALIGNOF(FuelGaugeParameterManagerStub)> stub;
    };

public:
    ModuleHolder() NN_NOEXCEPT;

    ~ModuleHolder() NN_NOEXCEPT;

    //! 初期化します。
    void Initialize(
        IChargerDriver* pChargerDriver,
        IFuelGaugeDriver* pFuelGaugeDriver,
        ISupplyRouteDriver* pSupplyRouteDriver,
        ChargeArbiter* pChargeArbiter,
        SessionManager* pSessionManager,
        EventMonitor* pBatteryEventMonitor,
        EventMonitor* pEventMonitorForUsbPowerDelivery,
        const SettingsHolder* pSettingsHolder) NN_NOEXCEPT;

    //! 終了処理をします。
    void Finalize() NN_NOEXCEPT;

    //! 保持するハンドラをイベントモニタにリンクしてイベント監視します。
    void Activate() NN_NOEXCEPT;

    //! 保持するハンドラをイベントモニタからアンリンクしてイベント監視を終了します。
    void Deactivate() NN_NOEXCEPT;

    //! 電池残量を取得します。
    ::nn::Result GetBatteryChargePercentage(int* pOutValue) NN_NOEXCEPT;

    //! 未加工の電池残量を取得します。
    ::nn::Result GetRawBatteryChargePercentage(double* pOutValue) NN_NOEXCEPT;

    //! 電池劣化量を取得します。
    ::nn::Result GetBatteryAgePercentage(double* pOutValue) NN_NOEXCEPT;

    //! 本体に接続された充電器の種別を取得します。
    ::nn::Result GetChargerType(ChargerType* pOutValue) NN_NOEXCEPT;

    //! 電池電圧状態を取得します。
    ::nn::Result GetBatteryVoltageState(BatteryVoltageState* pOutValue) NN_NOEXCEPT;

    //! コントローラへの給電の使用権を取得します。
    ::nn::Result AcquireControllerPowerSupply() NN_NOEXCEPT;

    //! コントローラへの給電の使用権を解放します。
    ::nn::Result ReleaseControllerPowerSupply() NN_NOEXCEPT;

    //! OTG を許可します。
    ::nn::Result AllowOtg() NN_NOEXCEPT;

    //! OTG を禁止します。
    ::nn::Result DisallowOtg() NN_NOEXCEPT;

    //! 給電エミュレーションを有効にします。
    ::nn::Result EnableEnoughPowerChargeEmulation() NN_NOEXCEPT;

    //! 給電エミュレーションを無効にします。
    ::nn::Result DisableEnoughPowerChargeEmulation() NN_NOEXCEPT;

    //! 高速充電を有効にします。
    ::nn::Result EnableFastBatteryCharging() NN_NOEXCEPT;

    //! 高速充電を無効にします。
    ::nn::Result DisableFastBatteryCharging() NN_NOEXCEPT;

    //! 十分な給電がされているかどうかを取得します。
    ::nn::Result IsEnoughPowerSupplied(bool* pOutValue) NN_NOEXCEPT;

    //! 電池の残量ずれを検出して補正を行ったことを伝えるシステムイベントを取得します。
    ::nn::Result GetBatteryChargeCalibratedEvent(::nn::os::SystemEventType** pOutEventPointer) NN_NOEXCEPT;

#if defined(PSM_DETAIL_MANUFACTURE_BUILD)
    //! 給電経路制御機能を有効にします。
    void EnableSupplyRouteControl() NN_NOEXCEPT;

    //! 給電経路制御機能を無効にします。
    void DisableSupplyRouteControl() NN_NOEXCEPT;
#endif

    //! テスト用途で PmModule へのポインタを返します。
    ::nn::Result GetPmModuleForTest(::nn::psc::PmModule** pOutPmModulePointer) NN_NOEXCEPT;

    //! テスト用途で PSC のハンドリング完了イベントへのポインタを返します。
    ::nn::Result GetPscEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT;

    //! テスト用途で電池残量 IC のハンドリング完了イベントへのポインタを返します。
    ::nn::Result GetFuelGaugeEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT;

    //! テスト用途で電池残量計定期的データ受信のハンドリング完了イベントへのポインタを返します。
    ::nn::Result GetFuelGaugeTimerEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT;

    //! テスト用途で USB PD のハンドリング完了イベントへのポインタを返します。
    ::nn::Result GetUsbPdEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT;

    //! テスト用途で USB PM のハンドリング完了イベントへのポインタを返します。
    ::nn::Result GetUsbPmEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT;

    //! テスト用途で給電エミュレーションのハンドリング完了イベントへのポインタを返します。
    ::nn::Result GetEnoughPowerChargeEmulationEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT;

private:
    //! イベント監視がアクティブかどうか
    bool m_IsActive;

    //! デバッグ・プラットフォーム設定
    const SettingsHolder* m_pSettingsHolder;

    //! チャージャのイベントハンドラ
    ChargerEventHandler m_ChargerEventHandler;

    //! 電池残量計のイベントハンドラ
    FuelGaugeEventHandler m_FuelGaugeEventHandler;

    //! 定期的に電池残量系のカスタムパラメータの保存するハンドラ
    PeriodicCustomParameterSaver m_PeriodicCustomParameterSaver;

    //! 定期的な状態更新のイベントハンドラ
    PeriodicStateUpdater m_PeriodicStateUpdater;

    //! 定期的にシステムレポートを送信するハンドラ
    PeriodicSystemReportSaver m_PeriodicSystemReportSaver;

    //! USB Power Delivery のイベントハンドラ
    UsbPowerDeliveryEventHandler m_UsbPowerDeliveryEventHandler;

    //! USB の電源関連のイベントハンドラ
    UsbPowerEventHandler m_UsbPowerEventHandler;

    //! システム電源状態のイベントハンドラ
    SystemPowerStateEventHandler m_SystemPowerStateEventHandler;

    //! 起床時により早い段階で復旧しなければいけないサービスに関する、システム電源状態のイベントハンドラ
    SystemPowerStateEventHandlerLow m_SystemPowerStateEventHandlerLow;

    //! 電源状態管理
    PowerStateManager m_PowerStateManager;

    //! 電池温度管理
    BatteryTemperatureManager m_TemperatureManager;

    //! 電池残量管理
    BatteryChargeManager m_BatteryChargeManager;

    //! 電池残量計パラメータ管理
    IFuelGaugeParameterManager* m_pFuelGaugeParameterManager;
    FuelGaugeParameterManagerStorage m_FuelGaugeParameterManagerStorage;

    //! USB 電源関連管理
    IUsbPowerManager* m_pUsbPowerManager;
    UsbPowerManagerStorage m_UsbPowerManagerStorage;

    //! 電池状態
    BatteryState m_BatteryState;

    //! 給電経路ルール
    SupplyRouteManager m_SupplyRouteManager;

    //! オーバーレイ通知管理
    OverlayNotificationSender m_NotificationSender;

    //! Link, Unlink 対象
    EventMonitor* m_pBatteryEventMonitor;

    //! Link, Unlink 対象
    EventMonitor* m_pEventMonitorForUsbPowerDelivery;

    //! 電池残量計ドライバへのポインタ
    IFuelGaugeDriver* m_pFuelGaugeDriver;
};

//ModuleHolder* GetModuleHolder() NN_NOEXCEPT;

}}}} // namespace nn::psm::driver::detail
