﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <nn/nn_SdkAssert.h>
#include <nn/psm/detail/psm_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/factory/settings_BatteryLot.h>
#include <nn/settings/factory/settings_Result.h>
#include <nn/settings/system/settings_Ptm.h>
#include <nn/TargetConfigs/build_Base.h>

// TODO: FuelGaugeDriver の定数参照の為に用意しています。可能ならば参照を切ります。
#include "psm_FuelGaugeDriver.h"
#include "psm_FuelGaugeParameterManager.h"

namespace nn { namespace psm { namespace driver { namespace detail {

namespace {

bool IsValidBatteryLot(
    ::nn::settings::factory::BatteryLot lot) NN_NOEXCEPT
{
    for (auto i = 0; i < sizeof(lot.string); ++i)
    {
        if (lot.string[i] != 0)
        {
            return true;
        }
    }

    // ゼロ埋めされている場合は無効として扱う（SIGLO-39573）
    return false;
}

bool IsSameBatteryLot(
    ::nn::settings::factory::BatteryLot lot1,
    ::nn::settings::factory::BatteryLot lot2) NN_NOEXCEPT
{
    if (::std::strncmp(lot1.string, lot2.string, sizeof(lot1.string)) == 0)
    {
        return true;
    }
    else
    {
        return false;
    }
}

void ConvertParameter(
    IFuelGaugeDriver::Parameter* pOutValue,
    const ::nn::settings::system::PtmFuelGaugeParameter& parameter) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    pOutValue->rcomp0 = parameter.rcomp0;
    pOutValue->tempCo = parameter.tempCo;
    pOutValue->fullCap = parameter.fullCap;
    pOutValue->fullCapNom = parameter.fullCapNom;
    pOutValue->iAvgEmpty = parameter.iavg_empty;
    pOutValue->qrTable00 = parameter.qrTable00;
    pOutValue->qrTable10 = parameter.qrTable10;
    pOutValue->qrTable20 = parameter.qrTable20;
    pOutValue->qrTable30 = parameter.qrTable30;
    pOutValue->cycles = static_cast<uint16_t>(parameter.cycles);
}

void ConvertParameter(
    ::nn::settings::system::PtmFuelGaugeParameter* pOutValue,
    const IFuelGaugeDriver::Parameter& parameter) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    pOutValue->rcomp0 = parameter.rcomp0;
    pOutValue->tempCo = parameter.tempCo;
    pOutValue->fullCap = parameter.fullCap;
    pOutValue->fullCapNom = parameter.fullCapNom;
    pOutValue->iavg_empty = parameter.iAvgEmpty;
    pOutValue->qrTable00 = parameter.qrTable00;
    pOutValue->qrTable10 = parameter.qrTable10;
    pOutValue->qrTable20 = parameter.qrTable20;
    pOutValue->qrTable30 = parameter.qrTable30;
    pOutValue->cycles = parameter.cycles;
}

} // namespace

::nn::Result FuelGaugeParameterManager::Initialize(IFuelGaugeDriver* pFuelGaugeDriver) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pFuelGaugeDriver);

    m_pFuelGaugeDriver = pFuelGaugeDriver;

    ::nn::settings::factory::BatteryLot factoryLot;
    ::nn::settings::factory::BatteryLot savedLot;

    // SDEV や生産工程の EDEV では較正情報に電池 LOT が書き込まれておらず CRC エラーになる
    // 電池 LOT の読み出しに失敗した場合は CRC エラーのときだけはエラーとせず読み書きもしない
    auto result = ::nn::settings::factory::GetBatteryLot(&factoryLot);

    // ハードウェア NX 以外で較正情報の取得に失敗した場合、エラーを返さず処理を継続する。
#if (defined NN_BUILD_CONFIG_HARDWARE_NX)
    if (result.IsFailure() &&
        !nn::settings::factory::ResultCalibrationDataCrcError::Includes(result))
    {
        return result;
    }
#endif

    if (result.IsSuccess() && IsValidBatteryLot(factoryLot))
    {
        m_IsUpdateEnabled = true;

        ::nn::settings::system::GetPtmBatteryLot(&savedLot);

        if (IsSameBatteryLot(savedLot, factoryLot))
        {
            // SIGLO-66059: パラメータの復旧が必要か確認する。
            bool needToRestoreParameters = true;
            m_pFuelGaugeDriver->GetNeedToRestoreParameters(&needToRestoreParameters);
            if (needToRestoreParameters)
            {
                NN_RESULT_DO(RestoreParameter());
                m_pFuelGaugeDriver->SetNeedToRestoreParameters(false);
            }
        }
        else
        {
            NN_RESULT_DO(InitializeParameter());
            ::nn::settings::system::SetPtmBatteryLot(factoryLot);
            // パラメータの永続化を保証するために読み込む
            ::nn::settings::system::GetPtmBatteryLot(&factoryLot);
        }
    }
    else
    {
        m_IsUpdateEnabled = false;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeParameterManager::InitializeParameter() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFuelGaugeDriver);

    if (!m_IsUpdateEnabled)
    {
        NN_RESULT_SUCCESS;
    }

    IFuelGaugeDriver::Parameter parameter;
    NN_RESULT_DO(m_pFuelGaugeDriver->SaveParameter(&parameter));

    ::nn::settings::system::PtmFuelGaugeParameter newParameter;
    ConvertParameter(&newParameter, parameter);
    int delta = ::std::max(
        0, static_cast<int>(parameter.cycles) - FuelGaugeDriver::CyclesMin);
    newParameter.cycles = static_cast<uint16_t>(delta);
    ::nn::settings::system::SetPtmFuelGaugeParameter(newParameter);

    NN_RESULT_DO(FlushParameter());

    NN_DETAIL_PSM_TRACE("Initialize fuelgauge parameter.\n");

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeParameterManager::SaveParameter() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFuelGaugeDriver);

    if (!m_IsUpdateEnabled)
    {
        NN_RESULT_SUCCESS;
    }

    IFuelGaugeDriver::Parameter parameter;
    NN_RESULT_DO(m_pFuelGaugeDriver->SaveParameter(&parameter));

    ::nn::settings::system::PtmFuelGaugeParameter oldParameter;
    ::nn::settings::system::GetPtmFuelGaugeParameter(&oldParameter);

    ::nn::settings::system::PtmFuelGaugeParameter newParameter;
    ConvertParameter(&newParameter, parameter);
    if (parameter.cycles >= FuelGaugeDriver::CyclesClearThreshold)
    {
        NN_RESULT_DO(m_pFuelGaugeDriver->ClearCycles());

        // cycles レジスタの積算値を保存
        int delta = ::std::max(
            0, static_cast<int>(newParameter.cycles) - FuelGaugeDriver::CyclesMin);
        newParameter.cycles = oldParameter.cycles + delta;
    }
    else
    {
        newParameter.cycles = oldParameter.cycles;
    }

    ::nn::settings::system::SetPtmFuelGaugeParameter(newParameter);

    NN_DETAIL_PSM_TRACE("Save fuelgauge parameter.\n");

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeParameterManager::RestoreParameter() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFuelGaugeDriver);

    if (!m_IsUpdateEnabled)
    {
        NN_RESULT_SUCCESS;
    }

    ::nn::settings::system::PtmFuelGaugeParameter parameter;
    ::nn::settings::system::GetPtmFuelGaugeParameter(&parameter);

    IFuelGaugeDriver::Parameter oldParameter;
    NN_RESULT_DO(m_pFuelGaugeDriver->SaveParameter(&oldParameter));

    IFuelGaugeDriver::Parameter newParameter;
    ConvertParameter(&newParameter, parameter);
    // cycles レジスタはそのまま
    newParameter.cycles = oldParameter.cycles;
    NN_RESULT_DO(m_pFuelGaugeDriver->RestoreParameter(newParameter));

    NN_DETAIL_PSM_TRACE("Restore fuelgauge parameter.\n");

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeParameterManager::FlushParameter() NN_NOEXCEPT
{
    if (!m_IsUpdateEnabled)
    {
        NN_RESULT_SUCCESS;
    }

    // パラメータの永続化を保証するために読み込む
    ::nn::settings::system::PtmFuelGaugeParameter parameter;
    ::nn::settings::system::GetPtmFuelGaugeParameter(&parameter);

    NN_RESULT_SUCCESS;
}

}}}} // namespace nn::psm::driver::detail
