﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/os.h>

#include <nn/os/os_SdkMutex.h>

#include "Handlers/psm_EventHandler.h"
#include "psm_EventMonitor.h"
#include "psm_ErrorReporter.h"

NN_DISABLE_WARNING_ARRAY_DEFAULT_INITIALIZATION_IN_CONSTRUCTOR

namespace nn { namespace psm { namespace driver { namespace detail {

EventMonitor::EventMonitor() NN_NOEXCEPT
    : m_Thread()
    , m_TerminationEvent()
    , m_TerminationMultiWaitHolder()
    , m_MultiWait()
    , m_EventHandlerList()
    , m_ThreadSyncEvent()
    , m_ThreadSyncMultiWaitHolder()
    , m_ThreadSyncBarrier()
    , m_LinkHandlerMutex()
{
    for ( auto& eventHandlerNode : m_EventHandlerList )
    {
        eventHandlerNode.Clear();
    }
}

EventMonitor::~EventMonitor() NN_NOEXCEPT
{
    // 何もしない
}

void EventMonitor::Initialize(int threadPriority, const char* pThreadName, void* pStack, size_t stackSize) NN_NOEXCEPT
{
    ::nn::os::CreateSystemEvent(&m_TerminationEvent, ::nn::os::EventClearMode_ManualClear, false);
    ::nn::os::InitializeMultiWaitHolder(&m_TerminationMultiWaitHolder, &m_TerminationEvent);
    ::nn::os::InitializeMultiWait(&m_MultiWait);
    ::nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_TerminationMultiWaitHolder);

    ::nn::os::InitializeEvent(&m_ThreadSyncEvent, false, ::nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeMultiWaitHolder(&m_ThreadSyncMultiWaitHolder, &m_ThreadSyncEvent);
    ::nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_ThreadSyncMultiWaitHolder);

    ::nn::os::InitializeBarrier(&m_ThreadSyncBarrier, 2);

    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::os::CreateThread(&m_Thread, EventMonitor::ThreadFunc, this, pStack, stackSize, threadPriority));

    ::nn::os::SetThreadNamePointer(&m_Thread, pThreadName);

    ::nn::os::StartThread(&m_Thread);
}

void EventMonitor::Finalize() NN_NOEXCEPT
{
    ::nn::os::SignalSystemEvent(&m_TerminationEvent);
    ::nn::os::WaitThread(&m_Thread);
    ::nn::os::DestroyThread(&m_Thread);

    ::nn::os::FinalizeBarrier(&m_ThreadSyncBarrier);

    ::nn::os::UnlinkMultiWaitHolder(&m_ThreadSyncMultiWaitHolder);
    ::nn::os::FinalizeMultiWaitHolder(&m_ThreadSyncMultiWaitHolder);
    ::nn::os::FinalizeEvent(&m_ThreadSyncEvent);

    ::nn::os::UnlinkMultiWaitHolder(&m_TerminationMultiWaitHolder);
    ::nn::os::FinalizeMultiWait(&m_MultiWait);
    ::nn::os::FinalizeMultiWaitHolder(&m_TerminationMultiWaitHolder);
    ::nn::os::DestroySystemEvent(&m_TerminationEvent);
}

// TODO: 優先度を追加できたほうが良いです。
void EventMonitor::LinkEventHandler(EventHandler* pEventHandler) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_LinkHandlerMutex)> lock(m_LinkHandlerMutex);

    auto isOwnThread = ::nn::os::GetCurrentThread() == &m_Thread;
    if ( !isOwnThread )
    {
        ::nn::os::SignalEvent(&m_ThreadSyncEvent);
        ::nn::os::AwaitBarrier(&m_ThreadSyncBarrier);
    }

    for ( auto& eventHandlerNode : m_EventHandlerList )
    {
        if ( !eventHandlerNode.valid )
        {
            pEventHandler->LinkMultiWaitHolders(&m_MultiWait);
            eventHandlerNode.pEventHandler = pEventHandler;
            eventHandlerNode.valid = true;

            if ( !isOwnThread )
            {
                ::nn::os::AwaitBarrier(&m_ThreadSyncBarrier);
            }

            return;
        }
    }

    // プログラムミスが無ければ到達しない
    NN_ABORT("psm_EventHandlerList is full!");
}

void EventMonitor::UnlinkEventHandler(EventHandler* pEventHandler) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_LinkHandlerMutex)> lock(m_LinkHandlerMutex);

    auto isOwnThread = ::nn::os::GetCurrentThread() == &m_Thread;
    if ( !isOwnThread )
    {
        ::nn::os::SignalEvent(&m_ThreadSyncEvent);
        ::nn::os::AwaitBarrier(&m_ThreadSyncBarrier);
    }

    for ( auto& eventHandlerNode : m_EventHandlerList )
    {
        if ( eventHandlerNode.valid && eventHandlerNode.pEventHandler == pEventHandler )
        {
            pEventHandler->UnlinkMultiWaitHolders();
            eventHandlerNode.pEventHandler = nullptr;
            eventHandlerNode.valid = false;

            if ( !isOwnThread )
            {
                ::nn::os::AwaitBarrier(&m_ThreadSyncBarrier);
            }

            return;
        }
    }

    // プログラムミスが無ければ到達しない
    NN_ABORT("Target EventHandler wasn't found!");
}

void EventMonitor::ThreadFunc(void* value) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(value);
    static_cast<EventMonitor*>(value)->InvokeEventLoop();
}

void EventMonitor::InvokeEventLoop() NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        auto needsExit = HandleEvents();

        GetErrorReporter().Submit();

        // 停止が要求されたならばループを脱出
        if (needsExit)
        {
            return;
        }
    }
}

bool EventMonitor::HandleEvents() NN_NOEXCEPT
{
    ::nn::os::MultiWaitHolderType* waitId = ::nn::os::WaitAny(&m_MultiWait);

    if (waitId == &m_TerminationMultiWaitHolder)
    {
        ::nn::os::ClearSystemEvent(&m_TerminationEvent);
        // 停止が要求された
        return true;
    }
    else if ( waitId == &m_ThreadSyncMultiWaitHolder )
    {
        ::nn::os::ClearEvent(&m_ThreadSyncEvent);
        ::nn::os::AwaitBarrier(&m_ThreadSyncBarrier);
        ::nn::os::AwaitBarrier(&m_ThreadSyncBarrier);

        return false;
    }
    for ( auto& eventHandlerNode : m_EventHandlerList )
    {
        if ( eventHandlerNode.valid )
        {
            if ( eventHandlerNode.pEventHandler->HandleEvent() )
            {
                break;
            }
        }
    }

    return false;
}

}}}} // namespace nn::psm::driver::detail
