﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/psm/detail/psm_Log.h>

#include "psm_ErrorReporter.h"

namespace nn { namespace psm { namespace driver { namespace detail {

namespace {

struct StaticMutex
{
    ::nn::os::MutexType mutex;

    void lock() NN_NOEXCEPT
    {
        ::nn::os::LockMutex(&mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        ::nn::os::UnlockMutex(&mutex);
    }
};

// Signal のスレッドセーフを目的とする Mutex
StaticMutex g_ErrorReporterMutex = { NN_OS_MUTEX_INITIALIZER(false) };

} // namespace

ErrorReporter::ErrorReporter() NN_NOEXCEPT
    : m_Dirty(false)
    , m_BatteryChargeInfoFields()
    , m_BatteryChargeInfoEvent()
{
    // エラーコンテキストの初期化
    m_BatteryChargeInfoFields.Clear();

    // エラーレポートおよびコンテキスト収集プロセスへの通知イベントの初期化
    ::nn::os::CreateSystemEvent(&m_BatteryChargeInfoEvent, ::nn::os::EventClearMode_ManualClear, true);
}

ErrorReporter::~ErrorReporter() NN_NOEXCEPT
{
    // エラーレポートおよびコンテキスト収集プロセスへの通知イベントの破棄
    ::nn::os::DestroySystemEvent(&m_BatteryChargeInfoEvent);
}

void ErrorReporter::SetInputCurrentLimit(int limitMilliAmpere) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    if ( m_BatteryChargeInfoFields.inputCurrentLimit != limitMilliAmpere )
    {
        m_BatteryChargeInfoFields.inputCurrentLimit = limitMilliAmpere;
        m_Dirty = true;
    }
}

void ErrorReporter::SetBoostModeCurrentLimit(int limitMilliAmpere) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    if ( m_BatteryChargeInfoFields.boostModeCurrentLimit != limitMilliAmpere )
    {
        m_BatteryChargeInfoFields.boostModeCurrentLimit = limitMilliAmpere;
        m_Dirty = true;
    }
}

void ErrorReporter::SetFastChargeCurrentLimit(int limitMilliAmpere) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    if ( m_BatteryChargeInfoFields.fastChargeCurrentLimit != limitMilliAmpere )
    {
        m_BatteryChargeInfoFields.fastChargeCurrentLimit = limitMilliAmpere;
        m_Dirty = true;
    }
}

void ErrorReporter::SetChargeVoltageLimit(int limitMilliVolt) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    if ( m_BatteryChargeInfoFields.chargeVoltageLimit != limitMilliVolt )
    {
        m_BatteryChargeInfoFields.chargeVoltageLimit = limitMilliVolt;
        m_Dirty = true;
    }
}

void ErrorReporter::SetChargeConfiguration(int chargeConfiguration) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    if ( m_BatteryChargeInfoFields.chargeConfiguration != chargeConfiguration )
    {
        m_BatteryChargeInfoFields.chargeConfiguration = chargeConfiguration;
        m_Dirty = true;
    }
}

void ErrorReporter::SetHizMode(bool hizMode) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    if ( m_BatteryChargeInfoFields.hizMode != hizMode )
    {
        m_BatteryChargeInfoFields.hizMode = hizMode;
        m_Dirty = true;
    }
}

void ErrorReporter::SetChargeEnabled(bool chargeEnabled) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    if ( m_BatteryChargeInfoFields.chargeEnabled != chargeEnabled )
    {
        m_BatteryChargeInfoFields.chargeEnabled = chargeEnabled;
        m_Dirty = true;
    }
}

void ErrorReporter::SetPowerSupplyPath(int state) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    int powerSupplyPath = state;

    if ( m_BatteryChargeInfoFields.powerSupplyPath != powerSupplyPath )
    {
        m_BatteryChargeInfoFields.powerSupplyPath = powerSupplyPath;
        m_Dirty = true;
    }
}

void ErrorReporter::SetBatteryTemperature(double temperature) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    int batteryTemperature = static_cast<int>(temperature * 1000.0);

    if ( m_BatteryChargeInfoFields.batteryTemperature != batteryTemperature )
    {
        m_BatteryChargeInfoFields.batteryTemperature = batteryTemperature;
        m_Dirty = true;
    }
}

void ErrorReporter::SetBatteryChargePercent(double percent) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    int milliPercent = static_cast<int>(percent * 1000.0);

    if ( m_BatteryChargeInfoFields.batteryChargePercent != milliPercent )
    {
        m_BatteryChargeInfoFields.batteryChargePercent = milliPercent;
        m_Dirty = true;
    }
}

void ErrorReporter::SetBatteryChargeVoltage(int milliVolt) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    if ( m_BatteryChargeInfoFields.batteryChargeVoltage != milliVolt )
    {
        m_BatteryChargeInfoFields.batteryChargeVoltage = milliVolt;
        m_Dirty = true;
    }
}

void ErrorReporter::SetBatteryAge(double percent) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    int milliPercent = static_cast<int>(percent * 1000.0);

    if ( m_BatteryChargeInfoFields.batteryAge != milliPercent )
    {
        m_BatteryChargeInfoFields.batteryAge = milliPercent;
        m_Dirty = true;
    }
}

void ErrorReporter::SetPowerRole(::nn::usb::UsbPowerRole usbPowerRole) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    int powerRole = static_cast<int>(usbPowerRole);

    if ( m_BatteryChargeInfoFields.powerRole != powerRole )
    {
        m_BatteryChargeInfoFields.powerRole = powerRole;
        m_Dirty = true;
    }
}

void ErrorReporter::SetPowerSupplyType(::nn::usb::UsbChargerType usbChargerType) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    int powerSupplyType = static_cast<int>(usbChargerType);

    if ( m_BatteryChargeInfoFields.powerSupplyType != powerSupplyType )
    {
        m_BatteryChargeInfoFields.powerSupplyType = powerSupplyType;
        m_Dirty = true;
    }
}

void ErrorReporter::SetPowerSupplyVoltage(int powerSupplyVoltage) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    if ( m_BatteryChargeInfoFields.powerSupplyVoltage != powerSupplyVoltage )
    {
        m_BatteryChargeInfoFields.powerSupplyVoltage = powerSupplyVoltage;
        m_Dirty = true;
    }
}

void ErrorReporter::SetPowerSupplyCurrent(int powerSupplyCurrent) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    if ( m_BatteryChargeInfoFields.powerSupplyCurrent != powerSupplyCurrent )
    {
        m_BatteryChargeInfoFields.powerSupplyCurrent = powerSupplyCurrent;
        m_Dirty = true;
    }
}

void ErrorReporter::SetFastBatteryChargingEnabled(bool fastBatteryChargingEnabled) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    if ( m_BatteryChargeInfoFields.fastBatteryChargingEnabled != fastBatteryChargingEnabled )
    {
        m_BatteryChargeInfoFields.fastBatteryChargingEnabled = fastBatteryChargingEnabled;
        m_Dirty = true;
    }
}

void ErrorReporter::SetControllerPowerSupplyAcquired(bool controllerPowerSupplyAcquired) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    if ( m_BatteryChargeInfoFields.controllerPowerSupplyAcquired != controllerPowerSupplyAcquired )
    {
        m_BatteryChargeInfoFields.controllerPowerSupplyAcquired = controllerPowerSupplyAcquired;
        m_Dirty = true;
    }
}

void ErrorReporter::SetOtgRequested(bool otgRequested) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    if ( m_BatteryChargeInfoFields.otgRequested != otgRequested )
    {
        m_BatteryChargeInfoFields.otgRequested = otgRequested;
        m_Dirty = true;
    }
}

void ErrorReporter::Submit() NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_ErrorReporterMutex);

    // 更新時のみシグナル
    if ( m_Dirty )
    {
        ::nn::os::SignalSystemEvent(&m_BatteryChargeInfoEvent);
        m_Dirty = false;
    }
}

void ErrorReporter::GetBatteryChargeInfoEvent(::nn::os::SystemEventType** pOutSystemEventPointer) NN_NOEXCEPT
{
    *pOutSystemEventPointer = &m_BatteryChargeInfoEvent;
}

void ErrorReporter::GetBatteryChargeInfoFields(BatteryChargeInfoFields* pOutFields) NN_NOEXCEPT
{
    *pOutFields = m_BatteryChargeInfoFields;
}

ErrorReporter& GetErrorReporter() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(ErrorReporter, s_ErrorReporter);

    return s_ErrorReporter;
}

}}}} // namespace nn::psm::driver::detail
