﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os.h>
#include <nn/gpio/gpio.h>
#include <nne/bq2419x/bq2419x.h>
#include <nne/../../Sources/Libraries/bq2419x/detail/bq2419x_Session.h>

#include "psm_ChargerDriver.h"
#include "psm_ErrorReporter.h"
#include "psm_MacroForDriver.h"

namespace nn { namespace psm { namespace driver { namespace detail {

namespace {

    // 充電 IC OTG 用 FET OFF 待ち時間（SDEV 対応）
    const ::nn::TimeSpan ChargerOtgFetOffTimeSpan = nn::TimeSpan::FromMilliSeconds( 150 );

    // 充電 IC デフォルト用 FET OFF 待ち時間（SDEV 対応）
    const ::nn::TimeSpan ChargerDefaultFetOffTimeSpan = nn::TimeSpan::FromMilliSeconds( 100 );

} // namespace

const ::nn::TimeSpan ChargerDriver::WatchdogTimerPeriodMax = ::nn::TimeSpan::FromSeconds(160);

ChargerDriver::ChargerDriver() NN_NOEXCEPT
    : m_PowerSupplySwitchingMutex(false)
{
    //何もしない
}

void ChargerDriver::Initialize() NN_NOEXCEPT
{
    ::nne::bq2419x::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nne::bq2419x::SetInputCurrentLimit(500));
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nne::bq2419x::SetBoostModeCurrentLimit(500));
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nne::bq2419x::SwitchHiZ(false));
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nne::bq2419x::ConfigureCharger(
            ::nne::bq2419x::ChargerType_ChargeBattery));

    // ErrorReporter 初期値設定
    GetErrorReporter().SetInputCurrentLimit(500);
    GetErrorReporter().SetBoostModeCurrentLimit(500);
    GetErrorReporter().SetHizMode(false);
    GetErrorReporter().SetChargeConfiguration(static_cast<int>(::nne::bq2419x::ChargerType_ChargeBattery));

    // GPIO 初期化
    ::nn::gpio::Initialize();
    ::nn::gpio::OpenSession( &m_PdVConnEnSession, gpio::GpioPadName_PdVconnEn );  // GPIO_PK5
    ::nn::gpio::SetDirection( &m_PdVConnEnSession, gpio::Direction_Output );
    ::nn::gpio::OpenSession( &m_OtgFet1Session, gpio::GpioPadName_OtgFet1ForSdev );  // GPIO_DAP4_DIN
    ::nn::gpio::SetDirection( &m_OtgFet1Session, gpio::Direction_Output );
    ::nn::gpio::OpenSession( &m_OtgFet2Session, gpio::GpioPadName_OtgFet2ForSdev );  // GPIO_PL0
    ::nn::gpio::SetDirection( &m_OtgFet2Session, gpio::Direction_Output );

    // 割込み用 GPIO 初期化
    ::nn::gpio::OpenSession(&m_ChargerInterruptSession, ::nn::gpio::GpioPadName_Bq24190Irq);
    ::nn::gpio::SetDirection(&m_ChargerInterruptSession, ::nn::gpio::Direction_Input);
    ::nn::gpio::SetInterruptMode(&m_ChargerInterruptSession, ::nn::gpio::InterruptMode_FallingEdge);
    ::nn::gpio::SetInterruptEnable(&m_ChargerInterruptSession, true);

    // SoC の電源が切れた場合に充電を有効に戻すために GPIO 経由で充電許可を操作する
    ::nn::gpio::OpenSession(&m_ChargeEnableSession, ::nn::gpio::GpioPadName_BattChgEnableN);
    ::nn::gpio::SetDirection(&m_ChargeEnableSession, ::nn::gpio::Direction_Output);

    m_WatchdogTimerPeriod = WatchdogTimerPeriodMax;
    m_IsWatchdogTimerEnabled = false;
    m_GpioSettings = GpioSettings::Uninitialized;

    // SIGLO-82567: VCONN_IN を常時遮断する
    ::nn::gpio::SetValue(&m_PdVConnEnSession, gpio::GpioValue_Low);

    // 給電元の初期値の設定 (EnoughPowerChargeEmulation 無効, OnTheGo 要求なし)
    SetupPowerSupply(false, false);
}

void ChargerDriver::Finalize() NN_NOEXCEPT
{
    ::nn::gpio::CloseSession(&m_ChargeEnableSession);

    // 割込み用 GPIO 終了処理
    ::nn::gpio::UnbindInterrupt(&m_ChargerInterruptSession);
    ::nn::gpio::CloseSession(&m_ChargerInterruptSession);

    // GPIO 終了処理
    ::nn::gpio::CloseSession( &m_OtgFet2Session );
    ::nn::gpio::CloseSession( &m_OtgFet1Session );
    ::nn::gpio::CloseSession( &m_PdVConnEnSession );
    ::nn::gpio::Finalize();

    ::nne::bq2419x::Finalize();
}

void ChargerDriver::SetUpUsbVbusChargingGpio() NN_NOEXCEPT
{
    // WiiU アダプタからの Charger への電圧印可防止（SDEV 対応）
    ::nn::gpio::SetValue( &m_OtgFet2Session, gpio::GpioValue_High );
    ::nn::os::SleepThread( ChargerOtgFetOffTimeSpan );

    // Charger IC の VBUS 端子と TypeC コネクタを接続（SDEV 対応）
    ::nn::gpio::SetValue( &m_OtgFet1Session, gpio::GpioValue_High );
}

::nn::Result ChargerDriver::EnableOnTheGo() NN_NOEXCEPT
{
    NN_RESULT_DO(SetupPowerSupply(true, m_IsEnoughPowerChargeEmulationEnabled));
    NN_RESULT_SUCCESS;
}

void ChargerDriver::SetUpVdd15HbChargingGpio() NN_NOEXCEPT
{
    // Charger IC の VBUS 端子と TypeC コネクタを遮断（SDEV 対応）
    ::nn::gpio::SetValue( &m_OtgFet1Session, gpio::GpioValue_Low );

    // Charger の VBUS 入力切り換え待ち
    ::nn::os::SleepThread( ChargerDefaultFetOffTimeSpan );

    // WiiU アダプタからの Charger への電圧印可（SDEV 対応）
    ::nn::gpio::SetValue( &m_OtgFet2Session, gpio::GpioValue_Low );
}

::nn::Result ChargerDriver::EnableBatteryCharging() NN_NOEXCEPT
{
    NN_RESULT_DO(SetupPowerSupply(false, m_IsEnoughPowerChargeEmulationEnabled));
    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::EnterHiZMode() NN_NOEXCEPT
{
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::bq2419x::SwitchHiZ(true));

    GetErrorReporter().SetHizMode(true);

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::ExitHiZMode() NN_NOEXCEPT
{
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::bq2419x::SwitchHiZ(false));

    GetErrorReporter().SetHizMode(false);

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::SetInputVoltageLimitMilliVolt(int inputVoltageLimitMilliVolt) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(inputVoltageLimitMilliVolt, 0);

    // int -> uint32_t への変換
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::bq2419x::SetInputVoltageLimit(inputVoltageLimitMilliVolt));

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::SetInputCurrentLimit(
    int limitMilliAmpere) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(limitMilliAmpere, 0);

    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(
        ::nne::bq2419x::SetInputCurrentLimit(
            static_cast<uint32_t>(limitMilliAmpere)));

    GetErrorReporter().SetInputCurrentLimit(limitMilliAmpere);

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::SetFastChargeCurrentLimit(
    int limitMilliAmpere) NN_NOEXCEPT
{
    return SetFastChargeCurrentLimit(limitMilliAmpere, FastChargeCurrentLimitOption::None);
}

::nn::Result ChargerDriver::SetFastChargeCurrentLimit(
    int limitMilliAmpere, FastChargeCurrentLimitOption fastChargeCurrentLimitOption) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(limitMilliAmpere, 0);

    const int InternalMosFet12MohmThresholdMilliAmpere = 1024;

    int fastChargeCurrentLimit = limitMilliAmpere;
    bool current20Percent = (limitMilliAmpere < InternalMosFet12MohmThresholdMilliAmpere)
        && (fastChargeCurrentLimitOption == FastChargeCurrentLimitOption::Current20Percent);

    // 1024 mA よりも小さい上限電流の場合 MOS-FET 抵抗値を 12 mOhm に維持するため 20% 設定を使用する。（bq24193 依存実装）
    if ( current20Percent )
    {
        fastChargeCurrentLimit = limitMilliAmpere * 5;
        NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::bq2419x::Force20PercentChargeCurrent(true));
    }

    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(
        ::nne::bq2419x::SetFastChargeCurrentLimit(
            static_cast<uint32_t>(fastChargeCurrentLimit)));

    if ( !current20Percent )
    {
        NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::bq2419x::Force20PercentChargeCurrent(false));
    }

    GetErrorReporter().SetFastChargeCurrentLimit(limitMilliAmpere);

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::SetBoostModeCurrentLimit(
    int limitMilliAmpere) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(limitMilliAmpere, 0);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nne::bq2419x::SetBoostModeCurrentLimit(
            static_cast<uint32_t>(limitMilliAmpere)));

    GetErrorReporter().SetBoostModeCurrentLimit(limitMilliAmpere);

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::GetStatus(Status* pStatus) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pStatus);

    ::nne::bq2419x::ChargeStatusType status;
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::bq2419x::GetChargeStatus(&status));

    switch (status)
    {
    case ::nne::bq2419x::ChargeStatusType_NotCharging:
        *pStatus = Status::NotCharging;
        break;
    case ::nne::bq2419x::ChargeStatusType_PreCharge:
        *pStatus = Status::PreCharge;
        break;
    case ::nne::bq2419x::ChargeStatusType_FastCharging:
        *pStatus = Status::FastCharge;
        break;
    case ::nne::bq2419x::ChargeStatusType_ChargeTerminationDone:
        *pStatus = Status::ChargeTermination;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::SetChargeVoltageLimit(int limitMilliVolt) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(limitMilliVolt, 0);

    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(
        ::nne::bq2419x::SetChargeVoltageLimit(
            static_cast<uint32_t>(limitMilliVolt)));

    GetErrorReporter().SetChargeVoltageLimit(limitMilliVolt);

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::GetFaultStatus(FaultStatus* pFaultStatus) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pFaultStatus);

    uint8_t fault;
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::bq2419x::GetFaults(&fault));

    pFaultStatus->Clear();
    pFaultStatus->isBoostFaultGenerated = (fault & ::nne::bq2419x::FaultType_BoostFault) == ::nne::bq2419x::FaultType_BoostFault;

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::SetWatchdogTimerPeriod(::nn::TimeSpan period) NN_NOEXCEPT
{
    NN_SDK_ASSERT_MINMAX(
        period.GetSeconds(),
        static_cast<int64_t>(1),
        WatchdogTimerPeriodMax.GetSeconds());

    m_WatchdogTimerPeriod = period;

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::EnableWatchdogTimer() NN_NOEXCEPT
{
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(
        ::nne::bq2419x::detail::Session::GetInstance().ResetI2CWatchdogTimer());

    uint32_t seconds = static_cast<uint32_t>(m_WatchdogTimerPeriod.GetSeconds());
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(
        ::nne::bq2419x::detail::Session::GetInstance().SetI2CWatchdogTimerSetting(seconds));

    m_IsWatchdogTimerEnabled = true;

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::DisableWatchdogTimer() NN_NOEXCEPT
{
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(
        ::nne::bq2419x::detail::Session::GetInstance().SetI2CWatchdogTimerSetting(0));

    m_IsWatchdogTimerEnabled = false;

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::ResetWatchdogTimer() NN_NOEXCEPT
{
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(
        ::nne::bq2419x::detail::Session::GetInstance().ResetI2CWatchdogTimer());

    NN_RESULT_SUCCESS;
}

bool ChargerDriver::IsWatchdogTimerEnabled() NN_NOEXCEPT
{
    return m_IsWatchdogTimerEnabled;
}

::nn::Result ChargerDriver::BindInterrupt(::nn::os::SystemEventType* pOutInterruptEvent) NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::gpio::BindInterrupt(pOutInterruptEvent, &m_ChargerInterruptSession));
    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::ClearInterrupt(::nn::os::SystemEventType* pInterruptEvent) NN_NOEXCEPT
{
    // 連続で割り込みを受け付けないように最後に有効化します。
    ::nn::gpio::ClearInterruptStatus(&m_ChargerInterruptSession);
    ::nn::os::ClearSystemEvent(pInterruptEvent);
    ::nn::gpio::SetInterruptEnable(&m_ChargerInterruptSession, true);
    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::SetChargeEnable(bool enabled) NN_NOEXCEPT
{
    ::nn::gpio::SetValue(&m_ChargeEnableSession, enabled ? (::nn::gpio::GpioValue_Low) : (::nn::gpio::GpioValue_High));
    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::SetEnoughPowerChargeEmulation(bool isEnabled) NN_NOEXCEPT
{
    NN_RESULT_DO(SetupPowerSupply(m_IsOnTheGoEnabled, isEnabled));

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::SetupPowerSupply(bool isOnTheGoRequested, bool isEnoughPowerChargeEmulationRequested) NN_NOEXCEPT
{
    // GPIOが操作中に他のスレッドから触られるのを防ぐためにロック
    // シングルスレッドなので必要ないが, 将来の変更を考えて念のため入れている
    std::lock_guard<decltype(m_PowerSupplySwitchingMutex)> lock(m_PowerSupplySwitchingMutex);

    // TODO: Firmware Debug Settings を使って給電元を HostBridge/USB から選択できるようにする.
    // USB 給電のときは製品機相当の挙動になるようにする.

    // OTG要求が来たときのみChargerとVbus を繋ぐようにする.
    // ただし, EnoughPowerChargeEmulation有効時はICが壊れる可能性があるので繋がない
    const GpioSettings gpioSettingsRequest
        = isEnoughPowerChargeEmulationRequested
              ? GpioSettings::Vdd15Hb                       // EnoughPowerChargeEmulation 有効時
              : isOnTheGoRequested ? GpioSettings::UsbVbus  // OTG要求時
                                   : GpioSettings::Vdd15Hb; // その他

    // 状態変化がない場合はリセット
    if ( gpioSettingsRequest == m_GpioSettings && isOnTheGoRequested == m_IsOnTheGoEnabled)
    {
        NN_RESULT_SUCCESS;
    }

    // USB を OnTheGo 設定以外に切り替える場合は充電ICを充電許可設定にする
    if (!(gpioSettingsRequest == GpioSettings::UsbVbus && isOnTheGoRequested))
    {
        // Charger IC 設定の切り替え
        const auto chargerType = ::nne::bq2419x::ChargerType_ChargeBattery;

        NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(
            ::nne::bq2419x::ConfigureCharger(chargerType));
        GetErrorReporter()
            .SetChargeConfiguration(static_cast<int>(chargerType));
    }

    // 給電元の選択
    switch (gpioSettingsRequest)
    {
    case GpioSettings::Vdd15Hb:
        SetUpVdd15HbChargingGpio();
        break;
    case GpioSettings::UsbVbus:
        SetUpUsbVbusChargingGpio();
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // USB 接続 かつ OnTheGo 有効が要求された時のみ充電ICを OnTheGo 設定にする
    if ( gpioSettingsRequest == GpioSettings::UsbVbus && isOnTheGoRequested )
    {
        // Charger IC 設定の切り替え
        const auto chargerType = ::nne::bq2419x::ChargerType_OTG;
        NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(
            ::nne::bq2419x::ConfigureCharger(chargerType));
        GetErrorReporter()
            .SetChargeConfiguration(static_cast<int>(chargerType));
    }

    m_IsOnTheGoEnabled = isOnTheGoRequested;
    m_IsEnoughPowerChargeEmulationEnabled =  isEnoughPowerChargeEmulationRequested;
    m_GpioSettings = gpioSettingsRequest;

    NN_RESULT_SUCCESS;
}
}}}} // namespace nn::psm::driver::detail
