﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/psm/psm_System.h>
#include <nn/psm/psm_SystemProcess.h>

#include "psm_BatteryVoltageLevelForCharge.h"
#include "psm_IBatteryState.h"

namespace nn { namespace psm { namespace driver { namespace detail {

class BatteryState final : public IBatteryState
{
    NN_DISALLOW_COPY(BatteryState);
    NN_DISALLOW_MOVE(BatteryState);

public:
    BatteryState() NN_NOEXCEPT;

    virtual ~BatteryState() NN_NOEXCEPT;

    //! 状態を初期化します。
    ::nn::Result Initialize() NN_NOEXCEPT;

    //! 終了処理をします。
    void Finalize() NN_NOEXCEPT;

    //! 電池残量を取得します。
    virtual int GetBatteryChargePercentage() NN_NOEXCEPT NN_OVERRIDE;

    //! 未加工の電池残量を取得します。
    virtual double GetRawBatteryChargePercentage() NN_NOEXCEPT NN_OVERRIDE;

    //! 電池劣化量を取得します。
    virtual double GetBatteryAgePercentage() NN_NOEXCEPT NN_OVERRIDE;

    //! 本体に接続された充電器の種別を取得します。
    virtual ChargerType GetChargerType() NN_NOEXCEPT NN_OVERRIDE;

    //! 未加工の電池電圧を取得します。
    virtual int GetRawBatteryChargeMilliVoltage() NN_NOEXCEPT NN_OVERRIDE;

    //! 電池電圧状態を取得します。
    virtual BatteryVoltageState GetBatteryVoltageState() NN_NOEXCEPT NN_OVERRIDE;

    //! 十分な給電がされているかどうかを取得します。
    virtual bool IsEnoughPowerSupplied() NN_NOEXCEPT NN_OVERRIDE;

    //! 未加工の電池残量を設定します。
    void SetRawBatteryChargePercentage(double percentage) NN_NOEXCEPT;

    //! 電池劣化量を設定します。
    void SetBatteryAgePercentage(double percentage) NN_NOEXCEPT;

    //! 本体に接続された充電器の種別を設定します。
    void SetChargerType(ChargerType chargerType) NN_NOEXCEPT;

    //! 未加工の電池電圧を設定します。
    void SetRawBatteryChargeMilliVoltage(int milliVoltage) NN_NOEXCEPT;

    //! 十分な給電がされているかどうかを設定します。
    void SetEnoughPowerSupplied(bool isEnoughPowerSupplied) NN_NOEXCEPT;

    //! 充電設定判断用の電池電圧レベルを取得します。
    BatteryVoltageLevelForCharge GetBatteryVoltageLevelForCharge() NN_NOEXCEPT;

    //! 平均充電電流を設定します。
    void SetAverageCurrentMilliAmpere(int averageCurrentMilliAmpere) NN_NOEXCEPT;

    //! 電圧残量ゲージによる残量を設定します。
    void SetVoltageFuelGaugePercentage(double voltageFuelGaugePercentage) NN_NOEXCEPT;

    //! 満充電か否かを取得します。
    bool IsFullCharge() NN_NOEXCEPT;

    //! 電池残量値の生値の補正用のオフセットを更新します。
    void UpdateRawBatteryChargeOffsetPercentage() NN_NOEXCEPT;

    //! 電池残量値の生値の補正用のオフセットを可能ならば減少させます。
    void TryDecreaseRawBatteryChargeOffsetPercentage() NN_NOEXCEPT;

    //! 電池残量値の生値の補正が行われたことを伝えるシステムイベントを取得します。
    nn::os::SystemEventType* GetBatteryChargeCalibratedEvent() NN_NOEXCEPT;

public:
    //! 強制シャットダウンする電圧閾値
    static const int ShutdownVoltageThresholdMilliVolt;

private:
    //! 未加工の電池残量
    ::std::atomic<double> m_RawBatteryChargePercentage;

    //! 本体に接続されている充電器の種別
    ::std::atomic<ChargerType> m_ChargerType;

    //! 未加工の電池電圧
    ::std::atomic<int> m_RawBatteryChargeMilliVoltage;

    //! 十分な給電がされているかどうか
    ::std::atomic<bool> m_IsEnoughPowerSupplied;

    //! 電池劣化量
    ::std::atomic<double> m_BatteryAgePercentage;

    //! 平均充電電流
    ::std::atomic<int> m_AverageCurrentMilliAmpere;

    //! 電圧残量ゲージによる残量
    ::std::atomic<double> m_VoltageFuelGaugePercentage;

    //! 未加工の電池残量の補正用オフセット
    ::std::atomic<double> m_RawBatteryChargeOffsetPercentage;

    //! 電池残量値の生値の補正が行われたことを伝えるシステムイベント
    nn::os::SystemEvent m_BatteryChargeCalibratedEvent;
};

}}}} // namespace nn::psm::driver::detail
