﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/os.h>

#include <nn/psc/psc_PmModule.h>
#include <nn/result/result_HandlingUtility.h>

#include "../psm_PowerStateManager.h"
#include "psm_SystemPowerStateEventHandler.h"

namespace nn { namespace psm { namespace driver { namespace detail {

SystemPowerStateEventHandler::SystemPowerStateEventHandler() NN_NOEXCEPT
    : m_PmModule()
    , m_PmModuleMultiWaitHolder()
    , m_pPowerStateManager(nullptr)
    , m_PscEvent()
{
    // 何もしません。
}

SystemPowerStateEventHandler::~SystemPowerStateEventHandler() NN_NOEXCEPT
{
    // 何もしません。
}

::nn::Result SystemPowerStateEventHandler::Initialize(PowerStateManager* pPowerStateManager) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pPowerStateManager);

    m_pPowerStateManager = pPowerStateManager;

    ::nn::os::InitializeEvent(&m_PscEvent, false, ::nn::os::EventClearMode_ManualClear);

    // TODO: USB, GPIO, I2C, Time への依存は SettingsHolder の値によっては切るべきです。
    ::nn::psc::PmModuleId dependencies[] =
    {
        ::nn::psc::PmModuleId_Fan,
        ::nn::psc::PmModuleId_Fs,
        ::nn::psc::PmModuleId_Gpio,
        ::nn::psc::PmModuleId_I2c,
        ::nn::psc::PmModuleId_Time,
        ::nn::psc::PmModuleId_Usb,
    };

    NN_RESULT_DO(
        m_PmModule.Initialize(
            ::nn::psc::PmModuleId_Psm,
            dependencies,
            sizeof(dependencies) / sizeof(dependencies[0]),
            ::nn::os::EventClearMode_ManualClear));

    ::nn::os::InitializeMultiWaitHolder(&m_PmModuleMultiWaitHolder, m_PmModule.GetEventPointer()->GetBase());

    NN_RESULT_SUCCESS;
}

void SystemPowerStateEventHandler::Finalize() NN_NOEXCEPT
{
    ::nn::os::FinalizeMultiWaitHolder(&m_PmModuleMultiWaitHolder);

    m_PmModule.Finalize();

    ::nn::os::FinalizeEvent(&m_PscEvent);
}

void SystemPowerStateEventHandler::LinkMultiWaitHolders(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pMultiWait);

    ::nn::os::LinkMultiWaitHolder(pMultiWait, &m_PmModuleMultiWaitHolder);
}

void SystemPowerStateEventHandler::UnlinkMultiWaitHolders() NN_NOEXCEPT
{
    ::nn::os::UnlinkMultiWaitHolder(&m_PmModuleMultiWaitHolder);
}

bool SystemPowerStateEventHandler::HandleEvent() NN_NOEXCEPT
{
    if ( m_PmModule.GetEventPointer()->TryWait() )
    {
        m_PmModule.GetEventPointer()->Clear();

        ::nn::psc::PmState state;
        ::nn::psc::PmFlagSet flags;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_PmModule.GetRequest(&state, &flags));

        auto result = m_pPowerStateManager->Update(state, flags);

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_PmModule.Acknowledge(state, result));

        ::nn::os::SignalEvent(&m_PscEvent);

        return true;
    }

    return false;
}

::nn::Result SystemPowerStateEventHandler::GetPmModuleForTest(::nn::psc::PmModule** pOutPmModulePointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutPmModulePointer);

    *pOutPmModulePointer = &m_PmModule;

    NN_RESULT_SUCCESS;
}

::nn::Result SystemPowerStateEventHandler::GetPscEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    *pOutEventPointer = &m_PscEvent;

    NN_RESULT_SUCCESS;
}

}}}} // namespace nn::psm::driver::detail
