﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/os.h>

#include "../psm_EventMonitor.h"
#include "../psm_IChargerDriver.h"
#include "../psm_IUsbPowerManager.h"
#include "psm_ChargerEventHandler.h"

namespace nn { namespace psm { namespace driver { namespace detail {

ChargerEventHandler::ChargerEventHandler() NN_NOEXCEPT
    : m_pUsbPowerManager(nullptr)
    , m_pChargerDriver(nullptr)
    , m_ChargerInterruptEvent()
    , m_ChargerInterruptMultiWaitHolder()
{
    // 何もしない。
}

ChargerEventHandler::~ChargerEventHandler() NN_NOEXCEPT
{
    // 何もしない。
}

void ChargerEventHandler::Initialize(IUsbPowerManager* pUsbPowerManager, IChargerDriver* pChargerDriver) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pUsbPowerManager);
    NN_SDK_ASSERT_NOT_NULL(pChargerDriver);

    m_pUsbPowerManager = pUsbPowerManager;
    m_pChargerDriver = pChargerDriver;

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_pChargerDriver->BindInterrupt(&m_ChargerInterruptEvent));
    ::nn::os::InitializeMultiWaitHolder(&m_ChargerInterruptMultiWaitHolder, &m_ChargerInterruptEvent);

    // 割り込みを発生させるためリードによりフォルトをクリアします。
    IChargerDriver::FaultStatus faultStatus;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_pChargerDriver->GetFaultStatus(&faultStatus));
    NN_UNUSED(faultStatus);
}

void ChargerEventHandler::Finalize() NN_NOEXCEPT
{
    ::nn::os::FinalizeMultiWaitHolder(&m_ChargerInterruptMultiWaitHolder);
}

void ChargerEventHandler::LinkMultiWaitHolders(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    ::nn::os::LinkMultiWaitHolder(pMultiWait, &m_ChargerInterruptMultiWaitHolder);
}

void ChargerEventHandler::UnlinkMultiWaitHolders() NN_NOEXCEPT
{
    ::nn::os::UnlinkMultiWaitHolder(&m_ChargerInterruptMultiWaitHolder);
}

bool ChargerEventHandler::HandleEvent() NN_NOEXCEPT
{
    if ( ::nn::os::TryWaitSystemEvent(&m_ChargerInterruptEvent) )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(ProcessChargerInterrupt());
        return true;
    }

    return false;
}

::nn::Result ChargerEventHandler::ProcessChargerInterrupt() NN_NOEXCEPT
{
    IChargerDriver::FaultStatus status;

    // 2 回目のリード結果が現在発生中のフォルトです。
    NN_RESULT_DO(m_pChargerDriver->GetFaultStatus(&status));
    NN_RESULT_DO(m_pChargerDriver->GetFaultStatus(&status));

    if ( status.isBoostFaultGenerated )
    {
        // OTG モードからチャージモードにします。
        NN_RESULT_DO(m_pUsbPowerManager->DisallowOtg());

        // フォルトをクリアします。
        auto result = m_pChargerDriver->GetFaultStatus(&status);

        // フォルトのクリアでエラーが発生した場合も AllowOtg までは試行します。
        NN_RESULT_DO(m_pUsbPowerManager->AllowOtg());

        NN_RESULT_DO(result);
    }

    NN_RESULT_DO(m_pChargerDriver->ClearInterrupt(&m_ChargerInterruptEvent));

    NN_RESULT_SUCCESS;
}

}}}} // namespace nn::psm::driver::detail
