﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/psc/psc_Types.h>
#include <nn/psc/psc_PmControlTypes.h>
#include <nn/psc/psc_PmModuleId.h>

#include "psc_Allocator.h"
#include "psc_RefCount.h"

namespace nn     {
namespace psc    {
namespace server {

class PmDependent;
class PmModule;
class PmControl;
class PmModuleSfObject;

class PmModule :
    public nn::util::IntrusiveListBaseNode<PmModule>,
    public Allocator,
    public RefCount
{
    friend PmDependent;
    friend PmControl;
    friend PmModuleSfObject;

private:
    static nn::util::IntrusiveList<PmModule, nn::util::IntrusiveListBaseNodeTraits<PmModule>> m_ModuleList;
    static nn::os::Mutex m_ModuleListLock;

    const PmModuleId            m_Id;
    PmState                     m_CurrentState;
    PmState                     m_RequestedState;
    PmState                     m_NextState;
    PmFlagSet                   m_Flags;
    bool                        m_Initialized;
    bool                        m_LinkedToMultiWait;
    nn::os::Mutex               m_ModuleLock;
    nn::os::MultiWaitHolderType m_Holder;
    nn::os::SystemEvent         m_StartEvent;
    nn::os::SystemEvent         m_DoneEvent;

    #if defined(NN_DETAIL_PSC_ENABLE_PERFORMANCE_CHECK)
    nn::os::Tick                m_StartTick;
    nn::os::Tick                m_EndTick;
    int32_t                     m_Level;

    void CalculateLevel(int baseLevel)
    NN_NOEXCEPT;
    #endif

    nn::util::IntrusiveList<PmDependent, nn::util::IntrusiveListBaseNodeTraits<PmDependent>>
                                m_ParentList;

    nn::util::IntrusiveList<PmDependent, nn::util::IntrusiveListBaseNodeTraits<PmDependent>>
                                m_ChildList;

    static PmModule* GetModuleById(PmModuleId id)
    NN_NOEXCEPT;

    nn::os::MultiWaitHolderType* GetMultiWaitHolderPointer()
    NN_NOEXCEPT;

    nn::Result AddParent(PmModule* pParent)
    NN_NOEXCEPT;

    nn::Result AddChild(PmModule* pChild)
    NN_NOEXCEPT;

    void RemoveParent(PmModule* pParent)
    NN_NOEXCEPT;

    void RemoveChild(PmModule* pChild)
    NN_NOEXCEPT;

    void ReleaseChildren()
    NN_NOEXCEPT;

    PmModuleId Id()
    NN_NOEXCEPT
    {
        return m_Id;
    }

    bool IsUnimplementedModule()
    NN_NOEXCEPT;

    nn::Result ValidateDependency(PmModule* pModule, uint32_t iterationCounter)
    NN_NOEXCEPT;

    nn::Result SendRequestToModule(PmState state, PmFlagSet flags)
    NN_NOEXCEPT;

    nn::Result DispatchDescendingOrder(nn::os::SystemEvent** pEventOut, PmState state, PmFlagSet flags, uint32_t iterationCounter)
    NN_NOEXCEPT;

    nn::Result DispatchAscendingOrder(nn::os::SystemEvent** pEventOut, PmState state, PmFlagSet flags, uint32_t iterationCounter)
    NN_NOEXCEPT;

    nn::Result DispatchRequest(nn::os::SystemEvent**  pEventOut, PmState state, PmFlagSet flags, PmTransitionOrder order)
    NN_NOEXCEPT;

    uint32_t ChildDependencyLevels(uint32_t* pLevelsOut, uint32_t iterationCounter)
    NN_NOEXCEPT;

    uint32_t ParentDependencyLevels(uint32_t* pLevelsOut, uint32_t iterationCounter)
    NN_NOEXCEPT;

public:
    nn::Result GetRequest(PmState* pStateOut, PmFlagSet* pFlagsOut)
    NN_NOEXCEPT;

    nn::Result Acknowledge(PmState state)
    NN_NOEXCEPT;

    nn::Result Initialize(nn::os::SystemEvent** pEventPointerOut, const PmModuleId* pClients, const uint32_t clientCount)
    NN_NOEXCEPT;

    nn::Result Finalize()
    NN_NOEXCEPT;

    explicit PmModule(PmModuleId id)
    NN_NOEXCEPT;

    ~PmModule()
    NN_NOEXCEPT;
};

}}}
