﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/psc/detail/psc_Log.h>
#include <nn/psc/psc_Types.h>
#include <nn/psc/psc_Result.h>
#include <nn/psc/psc_PmModuleId.h>
#include <nn/psc/util/psc_Util.h>

#include "psc_PmControl.h"
#include "psc_PmDependent.h"
#include "psc_PmModule.h"

namespace nn     {
namespace psc    {
namespace server {

void PmControl::LinkEvent(PmModule* pModule, nn::os::SystemEvent* pEvent)
NN_NOEXCEPT
{
    nn::os::MultiWaitHolderType* pHolder = pModule->GetMultiWaitHolderPointer();
    nn::os::InitializeMultiWaitHolder(pHolder, pEvent->GetBase());
    nn::os::SetMultiWaitHolderUserData(pHolder, reinterpret_cast<uintptr_t>(pModule));
    nn::os::LinkMultiWaitHolder(&m_MultiWait, pHolder);
    pModule->m_LinkedToMultiWait = true;
    m_PendingCount++;
}

void PmControl::UnlinkEvent(PmModule* pModule)
NN_NOEXCEPT
{
    nn::os::MultiWaitHolderType* pHolder = pModule->GetMultiWaitHolderPointer();
    nn::os::UnlinkMultiWaitHolder(pHolder);
    nn::os::FinalizeMultiWaitHolder(pHolder);
    pModule->m_LinkedToMultiWait = false;
    m_PendingCount--;
}

void PmControl::UnlinkAll()
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(PmModule::m_ModuleListLock);

    for (auto module  = PmModule::m_ModuleList.begin();
              module != PmModule::m_ModuleList.end();
              )
    {
        auto tmp = module++;

        if (tmp->m_LinkedToMultiWait)
        {
            UnlinkEvent(&(*tmp));
            if (tmp->RemoveReference())
            {
                delete (&(*tmp));
            }
        }
    }
}

nn::Result PmControl::SendRequestToOne(PmModule* pModule, PmState state, PmFlagSet flags, PmTransitionOrder order)
NN_NOEXCEPT
{
    nn::Result result;
    nn::os::SystemEvent* pEvent;

    result = pModule->DispatchRequest(&pEvent, state, flags, order);
    if (result <= nn::psc::ResultWaitTransition())
    {
        pModule->AddReference();
        LinkEvent(pModule, pEvent);
        result = ResultSuccess();
    }

    return result;
}

nn::Result PmControl::SendRequestToAll(PmState state, PmFlagSet flags, PmTransitionOrder order)
NN_NOEXCEPT
{
    nn::Result result;

    m_PendingCount  = 0;

    std::lock_guard<nn::os::Mutex> lock(PmModule::m_ModuleListLock);

    for (auto module  = PmModule::m_ModuleList.begin();
              module != PmModule::m_ModuleList.end();
              module++)
    {
        if ((result = SendRequestToOne(&(*module), state, flags, order)).IsFailure())
        {
            return result;
        }
    }

    return ResultSuccess();
}

nn::Result PmControl::WaitCompletionAll(PmState state, PmFlagSet flags, PmTransitionOrder order)
NN_NOEXCEPT
{
    nn::Result result = ResultSuccess();

    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_CancelEventHolder);

    while (m_PendingCount)
    {
        nn::os::MultiWaitHolderType* pHolder = nn::os::WaitAny(&m_MultiWait);

        if (pHolder == &m_CancelEventHolder)
        {
            result = nn::psc::ResultCancel();
            break;
        }

        {
            std::lock_guard<nn::os::Mutex> lock(PmModule::m_ModuleListLock);

            PmModule* pModule = reinterpret_cast<PmModule*>(nn::os::GetMultiWaitHolderUserData(pHolder));

            UnlinkEvent(pModule);

            if (pModule->RemoveReference())
            {
                // unlikely to happen, pm module has been finalized/removed
                delete pModule;
                continue;
            }

            if ((result = SendRequestToOne(pModule, state, flags, order)).IsFailure())
            {
                break;
            }
        }
    }

    UnlinkAll();

    nn::os::UnlinkMultiWaitHolder(&m_CancelEventHolder);

    return result;
}

void PmControl::Thread()
NN_NOEXCEPT
{
    nn::Result result;

    while (NN_STATIC_CONDITION(true))
    {
        m_StartEvent.Wait();

        if (!m_Running)
        {
            break;
        }

        {
            PmState           state;
            PmFlagSet         flags;
            PmTransitionOrder order;

            m_Result       = ResultSuccess();
            m_ControlState = PmControlState_Running;
            state          = m_State;
            flags          = m_Flags;
            order          = m_Order;

            if ((result = SendRequestToAll(state, flags, order)).IsFailure() ||
                (result = WaitCompletionAll(state, flags, order)).IsFailure())
            {
                m_Result       = result;
                m_ControlState = PmControlState_Error;
            }
            else
            {
                m_ControlState = PmControlState_Idle;
            }
            m_DoneEvent.Signal();

#if defined(NN_DETAIL_PSC_ENABLE_PERFORMANCE_CHECK)
            m_EndTick = nn::os::GetSystemTick();
#endif
        }
    }
}

nn::Result PmControl::DispatchRequest(PmState state, PmFlagSet flags, PmTransitionOrder order)
NN_NOEXCEPT
{
    m_PreviousState = m_State;
    m_State = state;
    m_Flags = flags;
    m_Order = order;

#if defined(NN_DETAIL_PSC_ENABLE_PERFORMANCE_CHECK)
    m_StartTick = nn::os::GetSystemTick();
#endif

    m_CancelEvent.Clear();
    m_DoneEvent.Clear();
    m_StartEvent.Signal();

    return ResultSuccess();
}

void PmControl::GetTransitionInformation(nn::psc::util::TransitionInformation* pOutInfo)
NN_NOEXCEPT
{
    *pOutInfo = {
#if defined(NN_DETAIL_PSC_ENABLE_PERFORMANCE_CHECK)
        m_StartTick.GetInt64Value(),
        m_EndTick.GetInt64Value(),
#else
        0L,
        0L,
#endif
        m_Result.GetInnerValueForDebug(),
        m_PreviousState,
        m_State
    };
}

int PmControl::GetModuleCount()
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(PmModule::m_ModuleListLock.IsLockedByCurrentThread());

    return PmModule::m_ModuleList.size();
}

nn::Result PmControl::GetModuleTransitionInformation(int* pOutCount, nn::psc::util::ModuleTransitionInformation* pOutInfo, int maxCount)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(PmModule::m_ModuleListLock.IsLockedByCurrentThread());

    int index = 0;
    for (auto pModule  = PmModule::m_ModuleList.begin();
              pModule != PmModule::m_ModuleList.end();
              pModule++)
    {
        if (index >= maxCount)
        {
            return ResultBufferOverflow();
        }

        pOutInfo[index++] = {
#if defined(NN_DETAIL_PSC_ENABLE_PERFORMANCE_CHECK)
            pModule->m_StartTick.GetInt64Value(),
            pModule->m_EndTick.GetInt64Value(),
            pModule->m_Level,
#else
            0,
            0L,
            0L,
#endif
            pModule->Id(),
            pModule->m_CurrentState,
            pModule->m_NextState,
            pModule->m_Initialized
        };
    }

    *pOutCount = index;
    return ResultSuccess();
}

int PmControl::GetDependencyCount()
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(PmModule::m_ModuleListLock.IsLockedByCurrentThread());

    int count = 0;
    for (auto pModule  = PmModule::m_ModuleList.begin();
              pModule != PmModule::m_ModuleList.end();
              pModule++)
    {
        count += pModule->m_ChildList.size();
    }

    return count;
}

nn::Result PmControl::GetDependencyInformation(int *pOutCount, nn::psc::util::DependencyInformation* pOutInfo, int maxCount)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(PmModule::m_ModuleListLock.IsLockedByCurrentThread());

    int index = 0;
    for (auto pModule = PmModule::m_ModuleList.begin();
              pModule != PmModule::m_ModuleList.end();
              pModule++)
    {
        for (auto pChild  = pModule->m_ChildList.begin();
                  pChild != pModule->m_ChildList.end();
                  pChild++)
        {
            if (index >= maxCount)
            {
                return ResultBufferOverflow();
            }
            pOutInfo[index++] = { pModule->Id(), pChild->Id() };
        }
    }

    *pOutCount = index;
    return ResultSuccess();
}

nn::Result PmControl::GetModuleInformation(
    nn::psc::util::TransitionInformation* pOutTransitionInfo,
    int *pOutModuleCount,
    nn::psc::util::ModuleTransitionInformation* pOutModuleInfo,
    int maxModuleCount,
    int *pOutDependencyCount,
    nn::psc::util::DependencyInformation* pOutDependencyInfo,
    int maxDependencyCount
) NN_NOEXCEPT
{
    nn::Result result;

    std::lock_guard<nn::os::Mutex> lock(PmModule::m_ModuleListLock);

    GetTransitionInformation(pOutTransitionInfo);

    result = GetModuleTransitionInformation(pOutModuleCount, pOutModuleInfo, maxModuleCount);
    if (result.IsFailure())
    {
        return result;
    }

    result = GetDependencyInformation(pOutDependencyCount, pOutDependencyInfo, maxDependencyCount);
    if (result.IsFailure())
    {
        return result;
    }

    return ResultSuccess();
}

nn::Result PmControl::PrintModuleInformation()
NN_NOEXCEPT
{
#if defined(NN_DETAIL_PSC_ENABLE_PERFORMANCE_CHECK)
    nn::Result result;
    int count;

    nn::psc::util::TransitionInformation transitionInfo;
    GetTransitionInformation(&transitionInfo);

    {
        std::lock_guard<nn::os::Mutex> lock(PmModule::m_ModuleListLock);

        // Calculate level
        for (auto pModule  = PmModule::m_ModuleList.begin();
                  pModule != PmModule::m_ModuleList.end();
                  pModule++)
        {
            pModule->CalculateLevel(0);
        }

        // Module transition info
        const int moduleCount = GetModuleCount();

        nn::psc::util::ModuleTransitionInformation moduleInfoList[moduleCount];

        result = GetModuleTransitionInformation(&count, moduleInfoList, moduleCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        NN_SDK_ASSERT(count == moduleCount);

        // Dependency info
        const int dependencyCount = GetDependencyCount();

        nn::psc::util::DependencyInformation dependencyInfoList[dependencyCount];

        result = GetDependencyInformation(&count, dependencyInfoList, dependencyCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        NN_SDK_ASSERT(count == dependencyCount);

        nn::psc::util::PrintModuleInformation(transitionInfo, moduleCount, moduleInfoList, dependencyCount, dependencyInfoList);
    }
#endif

    return ResultSuccess();
}

nn::Result PmControl::GetState(PmControlState* pStateOut)
NN_NOEXCEPT
{
    *pStateOut = m_ControlState;
    return ResultSuccess();
}

nn::Result PmControl::GetResult()
NN_NOEXCEPT
{
    return m_Result;
}

nn::Result PmControl::Cancel()
NN_NOEXCEPT
{
    m_CancelEvent.Signal();
    return ResultSuccess();
}

nn::Result PmControl::Initialize(nn::os::SystemEvent** pEventOut)
NN_NOEXCEPT
{
    if (m_Initialized)
    {
        return nn::psc::ResultAlreadyInitialized();
    }

    m_Initialized  = true;
    m_Running      = true;
    m_PreviousState = PmState_Unknown;
    m_State        = PmState_Unknown;
    m_ControlState = PmControlState_Idle;

    m_StartEvent.Clear();
    m_DoneEvent.Clear();

    nn::os::InitializeMultiWaitHolder(&m_CancelEventHolder, m_CancelEvent.GetBase());
    nn::os::InitializeMultiWait(&m_MultiWait);

    nn::os::CreateThread(&m_Thread, ThreadEntry, this, m_Stack, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(psc, PmControl));
    nn::os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(psc, PmControl));
    nn::os::StartThread(&m_Thread);

    *pEventOut = &m_DoneEvent;

    return ResultSuccess();
}

void PmControl::Finalize()
NN_NOEXCEPT
{
    if (m_Initialized)
    {
        m_Running = false;
        m_StartEvent.Signal();
        nn::os::WaitThread(&m_Thread);
        nn::os::DestroyThread(&m_Thread);
        nn::os::UnlinkAllMultiWaitHolder(&m_MultiWait);
        nn::os::FinalizeMultiWaitHolder(&m_CancelEventHolder);
        nn::os::FinalizeMultiWait(&m_MultiWait);
        m_Initialized = false;
    }
}

PmControl::PmControl()
NN_NOEXCEPT :
    m_StartEvent(nn::os::EventClearMode_AutoClear, false),
    m_DoneEvent(nn::os::EventClearMode_AutoClear, true),
    m_CancelEvent(nn::os::EventClearMode_AutoClear, false),
    m_Initialized(false),
    m_Running(false),
    m_PreviousState(PmState_Unknown),
    m_State(PmState_Unknown),
    m_ControlState(PmControlState_Idle),
    m_Result(ResultSuccess())
{

}

PmControl::~PmControl()
NN_NOEXCEPT
{
    Finalize();
}

}}}
