﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ExpHeapAllocator.h>

#include <nn/psc/server/psc_Server.h>
#include <nn/psc/psc_PmControl.h>
#include <nn/psc/sfdl/psc.sfdl.h>
#include <nn/psc/psc_Result.h>

namespace nn      {
namespace psc     {
namespace         {

const int SessionHeapSize = 8 * 1024;

class InitExpHeap
{
public:
    InitExpHeap()
    NN_NOEXCEPT
    {
        nn::sf::ExpHeapStaticAllocator
                <SessionHeapSize, PmControl>::
                Initialize(nn::lmem::CreationOption_ThreadSafe);
    }
} g_InitExpHeap;

} // namespace

nn::Result PmControl::Initialize(nn::os::EventClearMode eventClearMode)
NN_NOEXCEPT
{
    nn::sf::NativeHandle sfHandle;
    nn::Result result;

    if (m_Initialized)
    {
        return nn::psc::ResultAlreadyInitialized();
    }

    result = nn::sf::CreateHipcProxyByName
                    <nn::psc::sf::IPmControl, nn::sf::ExpHeapStaticAllocator
                    <SessionHeapSize, PmControl>::Policy>
                    (&m_InterfaceImpl, PmControlPortName);

    if (result.IsSuccess())
    {
        result = m_InterfaceImpl->Initialize(&sfHandle);

        if (result.IsSuccess())
        {
            m_SystemEvent.AttachReadableHandle(
                    sfHandle.GetOsHandle(),
                    sfHandle.IsManaged(),
                    eventClearMode);

            sfHandle.Detach();

            m_Initialized = true;
        }
        else
        {
            m_InterfaceImpl = nullptr;
        }
    }

    return result;
}

nn::Result PmControl::Finalize()
NN_NOEXCEPT
{
    if (m_Initialized)
    {
        m_InterfaceImpl = nullptr;
        m_Initialized   = false;
        nn::os::DestroySystemEvent(m_SystemEvent.GetBase());
    }

    return ResultSuccess();
}

nn::Result PmControl::DispatchRequest(PmState state, PmFlagSet flags, PmTransitionOrder order)
NN_NOEXCEPT
{
    return m_Initialized ?
           m_InterfaceImpl->DispatchRequest(state, flags, order) :
           nn::psc::ResultNotInitialized();
}

nn::Result PmControl::GetState(PmControlState* pStateOut)
NN_NOEXCEPT
{
    return m_Initialized ?
           m_InterfaceImpl->GetState(pStateOut) :
           nn::psc::ResultNotInitialized();
}

nn::Result PmControl::GetResult()
NN_NOEXCEPT
{
    return m_Initialized ?
           m_InterfaceImpl->GetResult() :
           nn::psc::ResultNotInitialized();
}

nn::Result PmControl::Cancel()
NN_NOEXCEPT
{
    return m_Initialized ?
           m_InterfaceImpl->Cancel() :
           nn::psc::ResultNotInitialized();
}

nn::os::SystemEvent* PmControl::GetEventPointer()
NN_NOEXCEPT
{
    return &m_SystemEvent;
}

nn::Result PmControl::PrintModuleInformation()
NN_NOEXCEPT
{
    return m_Initialized ?
           m_InterfaceImpl->PrintModuleInformation() :
           nn::psc::ResultNotInitialized();
}

nn::Result PmControl::GetModuleInformation(
    nn::psc::util::TransitionInformation* pOutTransitionInfo,
    int *pOutModuleCount,
    nn::psc::util::ModuleTransitionInformation* pOutModuleInfo,
    int maxModuleCount,
    int *pOutDependencyCount,
    nn::psc::util::DependencyInformation* pOutDependencyInfo,
    int maxDependencyCount
)
NN_NOEXCEPT
{
    nn::sf::OutBuffer moduleInfoBuffer( (char*)pOutModuleInfo, sizeof(nn::psc::util::ModuleTransitionInformation) * maxModuleCount );
    nn::sf::OutBuffer dependencyInfoBuffer( (char*)pOutDependencyInfo, sizeof(nn::psc::util::DependencyInformation) * maxDependencyCount );

    return m_Initialized ?
           m_InterfaceImpl->GetModuleInformation(
               pOutTransitionInfo,
               pOutModuleCount,
               moduleInfoBuffer,
               pOutDependencyCount,
               dependencyInfoBuffer
           ) : nn::psc::ResultNotInitialized();
}

PmControl::PmControl()
NN_NOEXCEPT
    : m_InterfaceImpl(nullptr),
      m_Initialized(false)
{

}

PmControl::~PmControl()
NN_NOEXCEPT
{
    Finalize();
}

}}
