﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring> // memset
#include <new>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_TransferMemory.h>

#include "profiler_DataStream.h"
#include "profiler_Defines.h"
#include "profiler_Hipc.h"
#include "profiler_Logging.h"
#include "profiler_Memory.h"
#include "profiler_SamplingThread.h"
#include "profiler_SettingsFromThePcGui.h"
#include "profiler_Workarea.h"
#include "profiler_WriteToBuffer.h"



namespace nn { namespace profiler {

namespace /*anonymous*/
{
    const size_t WorkAreaCount = SampleBufferIndex_MAX;

    struct Globals
    {
        WorkArea workAreas[WorkAreaCount];
        nn::os::MutexType mutexes[WorkAreaCount];
    };
    Globals* globals;
} // anonymous



void InitializeWorkAreas()
{
    globals = Memory::GetInstance()->Allocate<Globals>();
    new (globals) Globals;
    for (size_t i = 0; i < WorkAreaCount; ++i)
    {
        InitializeCoreWorkArea(SampleBufferIndex(i));
        nn::os::InitializeMutex(&globals->mutexes[i], false, 0);
    }
}



void FinalizeWorkAreas()
{
    for (size_t i = 0; i < WorkAreaCount; ++i)
    {
        nn::os::FinalizeMutex(&globals->mutexes[i]);
    }
    Memory::GetInstance()->Free(globals);
    globals = nullptr;
}



WorkArea* GetWorkAreaForCore(SampleBufferIndex core)
{
    NN_SDK_ASSERT(core < WorkAreaCount);
    return &globals->workAreas[core];
}



void InitializeCoreWorkArea(SampleBufferIndex core)
{
    NN_SDK_ASSERT(core < WorkAreaCount);
    WorkArea* ws = &globals->workAreas[core];
    memset(ws, 0, sizeof(WorkArea));
    ws->core_number = static_cast<uint8_t>(core);
    ws->recordCount = 0;
}



void FinalizeCoreWorkArea(SampleBufferIndex core)
{
    NN_SDK_ASSERT(core < WorkAreaCount);
    WorkArea* ws = &globals->workAreas[core];
    if (ws->settings->IsInProcess())
    {
        DUMP_CURRENT_LINE();
        SampleBuffers::GetInstance()->CloseCurrent(ws->curPtr, core);
        SampleBuffers::GetInstance()->FinalizeBuffer(core);
    }
    else
    {
        NN_SDK_ASSERT(core == SampleBufferIndex_Instrumentation, "Invalid core based on sampling mode\n");

        if (ws->startPtr == nullptr)
        {
            // We never actually gave this a buffer, don't attempt to send anything.
            DUMP_CURRENT_LINE();
            return;
        }

        // Fill in an epilogue as transfers must be 4K aligned
        // This means that we need to ensure that the out-of-process version will stop reading
        // To make things simpler on the PC side, fill in an empty "packet" into the Instrumentation Buffer
        uint8_t* writePtr = ws->curPtr.fetch_add(20);
        writePtr = WriteToBuffer(writePtr, PayloadEvents_Base);
        writePtr = WriteToBuffer(writePtr, static_cast<uint64_t>(0)); // time
        writePtr = WriteToBuffer(writePtr, static_cast<uint32_t>(0)); // core
        writePtr = WriteToBuffer(writePtr, PayloadGenericEvents_End);

        DUMP_CURRENT_LINE();
    }
}



bool GetMoreSampleBuffer(SampleBufferIndex core)
{
    NN_SDK_ASSERT(core < WorkAreaCount);
    WorkArea* ws = &globals->workAreas[core];
    bool result = true;

    nn::os::LockMutex(&globals->mutexes[core]);
    if (ws->curPtr >= ws->sentinel)
    {
        INFO_LOG("Buffer full, attempting to start a new buffer\n");
        if (ws->settings->IsInProcess())
        {
            SampleBuffers::GetInstance()->CloseCurrent(ws->curPtr, core);
            result = ObtainAndSetupSampleBuffer(core);
        }
        else
        {
            NN_SDK_ASSERT(core == SampleBufferIndex_Instrumentation, "Invalid core based on sampling mode\n");

            if (ws->startPtr == nullptr)
            {
                // Give the Instrumentation Buffer all available memory when
                // using the out-of-process profiling mode.
                void* addr = SampleBuffers::GetInstance()->GetStartAddress();
                size_t size = SampleBuffers::GetInstance()->GetSize();
                ws->startPtr = reinterpret_cast<uint8_t*>(addr);
                ws->curPtr = ws->startPtr;
                ws->sentinel = ws->startPtr + size - SentinelSize;
            }
            else
            {
                // There is no more available memory
                result = false;
            }
        }
    }
    nn::os::UnlockMutex(&globals->mutexes[core]);

    return result;
}



bool ObtainAndSetupSampleBuffer(SampleBufferIndex core)
{
    NN_SDK_ASSERT(core < WorkAreaCount);
    WorkArea *ws = &globals->workAreas[core];

    void* memory = SampleBuffers::GetInstance()->Allocate(core);
    if (memory != nullptr)
    {
        ws->startPtr = reinterpret_cast<uint8_t*>(memory);
        ws->curPtr = ws->startPtr;
        ws->sentinel = ws->startPtr + SampleMemoryBlockSize - SentinelSize;
        return true;
    }
    else
    {
        return false;
    }
}



bool CheckAndExpandBuffersIfNeeded(SampleBufferIndex core)
{
    NN_SDK_ASSERT(core < WorkAreaCount);
    WorkArea* ws = &globals->workAreas[core];

    //if (!(ws->record_cores & (1 << core)))
    //{
    //    DEBUG_LOG("Core %d no longer active, stopping profile\n", core);
    //    SampleBuffers::GetInstance()->CloseCurrent(ws->curPtr, core);
    //    //StopProfilingSamplingThreads();
    //    SendBasicIpcMessage(ProfilerIpcMessage_StopProfiling, 0);
    //    return false;
    //}

    if (ws->curPtr >= ws->sentinel)
    {
        bool moreMemory = GetMoreSampleBuffer(core);
        if (!moreMemory)
        {
            INFO_LOG("Core %d is full, stop recording\n", core);
            //StopProfilingSamplingThreads();
            SendBasicIpcMessage(ProfilerIpcMessage_StopProfiling, 0);
            return false;
        }
        DEBUG_LOG("Got new buffer for core %d - %p\n", core, ws->curPtr.load());
    }

    return true;
}


} // profiler
} // nn
