﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>

#include "profiler_LibPrivate.h"
#include "profiler_Logging.h"
#include "profiler_Memory.h"
#include "profiler_TargetApplication.h"
#include "profiler_Vsync.h"

namespace nn { namespace profiler {

namespace /*anonymous*/
{
    bool shouldExit;
    nn::os::ThreadType thread;
    int initializeCount = 0;

    const size_t VsyncThreadStackSize = 16 * 1024;
    uint8_t* stackMemory;

    void VsyncThread_ThreadFunc(void *arg)
    {
        NN_UNUSED(arg);
        nn::os::SystemEventType vsyncEvent;
        nn::Result result;

        result = nn::vi::profiler::GetVsyncEvent(&vsyncEvent);

        if (result.IsFailure())
        {
            ERROR_LOG("Error registering vsync event!\n");
            DumpResultInformation(LOG_AS_ERROR, result);
            return;
        }

        while (!shouldExit)
        {
            nn::os::WaitSystemEvent(&vsyncEvent);
            nn::os::ClearSystemEvent(&vsyncEvent);
            RecordVsync();
        }

        nn::os::DestroySystemEvent(&vsyncEvent);
    }
}

void InitializeVsyncSupport()
{
    if (nn::vi::profiler::Initialize)
    {
        NN_SDK_ASSERT(nn::vi::profiler::GetVsyncEvent);
        NN_SDK_ASSERT(nn::vi::profiler::Finalize);

        nn::Result result;
        shouldExit = false;

        result = nn::vi::profiler::Initialize();
        if (result.IsFailure())
        {
            ERROR_LOG("Could not initialize profiler vsync hook!\n");
            DumpResultInformation(LOG_AS_ERROR, result);
            shouldExit = true;
            return;
        }

        stackMemory = reinterpret_cast<uint8_t*>(
            Memory::GetInstance()->Allocate(VsyncThreadStackSize, nn::os::GuardedStackAlignment));

        result = nn::os::CreateThread(
            &thread,
            &VsyncThread_ThreadFunc,
            nullptr,
            stackMemory,
            VsyncThreadStackSize,
            TargetApplication::GetMaximumThreadPriority());

        if (result.IsFailure())
        {
            ERROR_LOG("Error starting vsync threads!\n");
            DumpResultInformation(LOG_AS_ERROR, result);
            NN_ABORT();
        }

        nn::os::SetThreadName(&thread, "NX CPU Profiler: Vsync");
        nn::os::StartThread(&thread);

        ++initializeCount;
    }
}

void FinalizeVsyncSupport()
{
    if (nn::vi::profiler::Finalize && initializeCount > 0)
    {
        --initializeCount;
        shouldExit = true;
        nn::os::WaitThread(&thread);
        nn::os::DestroyThread(&thread);
        Memory::GetInstance()->Free(stackMemory);
        nn::vi::profiler::Finalize();
    }
}

} // profiler
} // nn
