#/*--------------------------------------------------------------------------------*
#  Copyright (C)Nintendo All rights reserved.
#
#  These coded instructions, statements, and computer programs contain proprietary
#  information of Nintendo and/or its licensed developers and are protected by
#  national and international copyright laws. They may not be disclosed to third
#  parties or copied or duplicated in any form, in whole or in part, without the
#  prior written consent of Nintendo.
#
#  The content herein is highly confidential and should be handled accordingly.
# *--------------------------------------------------------------------------------*/

param (
    [Parameter(Mandatory=$true)][string]$OutputPath,
    [string]$SdkRoot = $ENV:NINTENDO_SDK_ROOT
)

$svcSrcFolder = "$SdkRoot/Programs/Chris/Sources/Libraries/svc"
if ($false -eq (test-path $svcSrcFolder))
{
    write-error "Could not find SVC src folder: $svcSrcFolder"
    exit 1
}

$outputLocation = $OutputPath | resolve-path

if ([String]::IsNullOrEmpty($env:SIGLO_TOOLCHAINS_ROOT) -or ($false -eq (Test-Path $env:SIGLO_TOOLCHAINS_ROOT)))
{
    Write-Error "Environment variable SIGLO_TOOLCHAINS_ROOT is either not set, or set to a non-existant path."
    exit 1
}

$toolchain = (Get-ChildItem $env:SIGLO_TOOLCHAINS_ROOT -Filter "clang-for-nx*" | where {$_.PSIsContainer} | sort Name -Descending | select -f 1).FullName

if ($false -eq (Test-Path $toolchain))
{
    Write-Error "Could not find: $toolchain"
    exit 1
}

function CppFilter
{
    param([string]$func)
    $cppFilter = "$toolchain/nx/aarch64/bin/aarch64-nintendo-nx-elf-c++filt.exe" | Resolve-Path
    & $cppFilter $func
}


function Append-File
{
    param(
        [string]$Path,
        $InputObject
    )
    $InputObject | out-file $Path -Append -Encoding "UTF8"
}


if ($false -eq (test-path $outputLocation))
{
    mkdir $outputLocation
}

$svc64 = "$svcSrcFolder/svc_Base.aarch64.lp64.autogen.S"
$svc32 = "$svcSrcFolder/svc_Base.aarch32.autogen.S"
$profilerSvc = "$outputLocation/profiler_Svc.autogen.S"
$profilerSvcH = "$outputLocation/profiler_Svc.autogen.h"

[string[]]$functionList = "QueryMemory", "GetInfo", "GetProcessId"
foreach ($func in $functionList)
{
    write-output "  -> Looking for function [$func]"
}

[string[]]$global:overridenFunctions = @()

# Rename the function to be in our namespace, but keep the same arguments
function ChangeFunctionName
{
    param([string]$func, [string]$replaceIn)

    $nameLength = $func.Length.ToString().Length
    $replaceRegex = '2nn3svc.*?(\d{' + $nameLength + '}' + $func + ')'
    [string]$temp = $replaceIn -replace $replaceRegex,'2nn3svc8profiler$1'
    return $temp
}

function CopyLines
{
    param([string]$_path, [ref]$_overrides)

    [string[]]$copiedLines = @()
    [string[]]$overrides = @()
    $fileContents = [string[]](get-content $_path)
    [regex]$searchFuncStart = [regex]'^\s*\.section\s*\.text\.([A-Za-z0-9_]+).*$'
    [regex]$searchFuncEnd = [regex]'^\s*\.size\s*([A-Za-z0-9_]+).*$'
    $shouldCopy = $false
    [string]$matchingFunc = ""
    foreach ($line in $fileContents)
    {
        if ($line -like '#*')
        {
            $copiedLines += ,[Environment]::NewLine,$line,[Environment]::NewLine
            continue
        }

        $match = $searchFuncStart.Match($line)
        if ($match.Success)
        {
            foreach ($func in $functionList)
            {
                [string]$group = $match.Groups[1]
                if ($group -like "*$func*")
                {
                    $matchingFunc = $func
                    $shouldCopy = $true
                }
            }
        }

        if ($shouldCopy)
        {
            [string]$toCopy = ChangeFunctionName $matchingFunc $line
            $copiedLines += ,$toCopy

            $match = $searchFuncEnd.Match($toCopy)
            if ($match.Success)
            {
                [string]$group = $match.Groups[1]
                $overrides += ,$group
                $matchingFunc = ""
                $shouldCopy = $false
                $copiedLines += ,[Environment]::NewLine
            }
        }
    }

    $_overrides.Value = [string[]]$overrides
    return $copiedLines
}

[string[]]$overrides64 = @()
$lines64 = CopyLines $svc64 ([ref]$overrides64)
[string[]]$overrides32 = @()
$lines32 = CopyLines $svc32 ([ref]$overrides32)

[string]$copyrightNotice = 
"/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
// This file was automatically generated. Do not modify it directly."

$copyrightNotice | out-file $profilerSvc -Encoding "UTF8"
append-file $profilerSvc ""
append-file $profilerSvc "#include <nn/TargetConfigs/build_Base.h>"
append-file $profilerSvc ""
append-file $profilerSvc "#if defined(NN_BUILD_CONFIG_CPU_CORTEX_A57_AARCH64)"
append-file $profilerSvc ""
foreach ($line in $lines64)
{
    append-file $profilerSvc $line
}
append-file $profilerSvc "#elif defined(NN_BUILD_CONFIG_CPU_CORTEX_A57_AARCH32)"
append-file $profilerSvc ""
foreach ($line in $lines32)
{
    append-file $profilerSvc $line
}
append-file $profilerSvc "#else"
append-file $profilerSvc "#error profiler_svc Unknown assembler type"
append-file $profilerSvc "#endif"


function WriteFunctionToHeader
{
    param([string[]]$functionList)
    foreach ($func in $functionList)
    {
        $f = ChangeFunctionName $func $func
        $f = CppFilter $f
        $f = $f -replace "nn::svc::profiler::","nn::Result "
        $f += ';'
        Append-File $profilerSvcH $f
    }
}


$copyrightNotice | Out-File $profilerSvcH -Encoding "UTF8"
append-file $profilerSvcH ""
append-file $profilerSvcH "#pragma once"
append-file $profilerSvcH ""
append-file $profilerSvcH "#include <nn/TargetConfigs/build_Base.h>"
Append-File $profilerSvcH "#include <nn/svc/svc_Base.h>"
Append-File $profilerSvcH "namespace nn { namespace svc { namespace profiler {"
append-file $profilerSvcH "#if defined(NN_BUILD_CONFIG_CPU_CORTEX_A57_AARCH64)"
WriteFunctionToHeader $overrides64
append-file $profilerSvcH "#elif defined(NN_BUILD_CONFIG_CPU_CORTEX_A57_AARCH32)"
WriteFunctionToHeader $overrides32
append-file $profilerSvcH "#else"
append-file $profilerSvcH "#error profiler_svc Unknown assembler type"
append-file $profilerSvcH "#endif"
Append-File $profilerSvcH "}}} // nn::svc::profiler"
