﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/*
* Copyright (c) 2016, NVIDIA CORPORATION.  All rights reserved.
*
* NVIDIA CORPORATION and its licensors retain all intellectual property
* and proprietary rights in and to this software, related documentation
* and any modifications thereto.  Any use, reproduction, disclosure or
* distribution of this software and related documentation without an express
* license agreement from NVIDIA CORPORATION is strictly prohibited
*/

#include <atomic>
#include <cstdio>
#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/dd.h>
#include <nn/htcs.h>
#include <nn/diag/diag_Backtrace.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/profiler/profiler_Api.h>

#include "profiler_Logging.h"
#include "profiler_Memory.h"
#include "profiler_MemoryApi.h"
#include "profiler_MemoryServer.h"
#include "profiler_RecordMethods.h"
#include "profiler_ResultPrivate.h"
#include "profiler_TargetApplication.h"
#include "profiler_Svc.autogen.h"

#include "profiler_Defines.h"


namespace nn { namespace profiler {


namespace /*anonymous*/
{
    enum op : uint16_t
    {
        // uint32_t version;  // should be 3
        // uint32_t len_name;
        // char[]   name;     // plugin name
        INIT = 0x00,

        // uint16_t heap;
        // size_t   address;
        // size_t   size_requested;
        // size_t   size_received;
        // uint16_t num_callstacks;
        // size_t   callstacks[];
        ALLOC = 0x01,

        // uint16_t heap;
        // size_t   address;
        FREE = 0x02,

        // uint16_t heap;
        // uint32_t name_len;
        // char     name[];
        HEAP = 0x03,
    };

    const int MaximumCallstackDepth = 32;
    const int MaximumHeapNameLength = 64;

    static nn::os::MutexType s_lock = NN_OS_MUTEX_INITIALIZER(false);
    static bool g_Connected = false;
    static bool g_Enabled = true;
    static SharedArea* g_SharedArea = nullptr;

    static struct
    {
        char name[MaximumHeapNameLength];
    } s_registeredHeaps[NN_PROFILER_MEMORY_MAX_HEAPS];
    static int s_numHeaps = 0;

    static void Send(const void *packet, size_t packetsize);
    static void SendHeapName(MemoryHeapHandle heap);
    static bool InitializeLocked();


    static uint16_t getpid()
    {
        static int storedValue = 0;
        if (storedValue >= 0)
        {
            nn::Bit64 processId;
            nn::svc::profiler::GetProcessId(&processId, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
            storedValue = (1 << 31 | static_cast<uint16_t>(processId & 0xFFFF));
        }
        return static_cast<uint16_t>(storedValue & 0xFFFF);
    }


    static nn::Result ConnectLocked()
    {
        INFO_LOG("Trying to connect to NvMemoryProfiler service...\n");
        InitializeMemoryProfilerServer(g_SharedArea);

        getpid();

        g_Connected = true;

        INFO_LOG("Connected memory client library.\n");

        return nn::ResultSuccess();
    }


    static void InitSharedMemoryLocked()
    {
        nn::Result result;
        auto temp = reinterpret_cast<SharedArea*>(Memory::GetInstance()->Allocate(SharedMemorySizeMax));
        NN_SDK_ASSERT(temp != nullptr);
        memset(temp, 0, SharedMemorySizeMax);
        std::atomic_thread_fence(std::memory_order_seq_cst);
        g_SharedArea = temp;
    }


    static void FreeSharedMemoryLocked()
    {
        auto temp = g_SharedArea;
        g_SharedArea = nullptr;
        Memory::GetInstance()->Free(temp);
    }


    bool InitializeLocked()
    {
        if (!g_Enabled)
        {
            return false;
        }

        //if (g_SharedArea == nullptr && GetProfilerStatus() != ProfilerStatus_Offline)
        //{
        //    InitSharedMemoryLocked();
        //}

        return (g_SharedArea != nullptr) && IsConnectedToMemoryProfilerServer();
    }


    static void MemoryAllocationLocked(MemoryHeapHandle heap, const void *ptr, size_t size)
    {
        if (!InitializeLocked())
        {
            return;
        }

        const int skipFrames = 2;

        int numCallstacks = 0;
        uint64_t callstack64[MaximumCallstackDepth];
        uintptr_t callstack[MaximumCallstackDepth];
        memset(callstack, 0, sizeof(callstack));
        int numUnwind = MaximumCallstackDepth;

        // If we are connected to the server, obey the requested callstack depth as
        // requested by the server.
        if (g_SharedArea)
        {
            if (g_SharedArea->numCallstacks == 0)
            {
                // User has disabled callstacks.
                numUnwind = 0;
            }
            else if ((int)g_SharedArea->numCallstacks > 0)
            {
                // User is setting a specific maximum depth for callstacks.
                numUnwind = skipFrames + static_cast<int>(g_SharedArea->numCallstacks);
                if (numUnwind > MaximumCallstackDepth)
                {
                    numUnwind = MaximumCallstackDepth;
                }
            }
        }
        if (numUnwind)
        {
            int unwound = static_cast<int>(nn::diag::GetBacktrace(callstack, numUnwind));

            for (int i = skipFrames; i < unwound; i++)
            {
                callstack64[i] = callstack[i];
                NN_SDK_ASSERT((callstack64[i] & (0xFFULL << 56)) == 0);
                // off-by-one error for some reason
                if (callstack64[i])
                {
                    callstack64[i] -= sizeof(size_t);
                }
                callstack64[i] |= (getpid() & 0xFFULL) << 56;
                numCallstacks++;
            }
        }

        Buffer<400> buf;
        buf.Write<uint16_t>(ALLOC);
        buf.Write<uint16_t>(2 + 8 + 8 + 8 + 2 + 8 * static_cast<uint16_t>(numCallstacks));
        buf.Write<uint16_t>(static_cast<uint16_t>(heap | getpid() << 8));
        buf.Write<uint64_t>((uintptr_t)ptr);
        buf.Write<uint64_t>(size);
        buf.Write<uint64_t>(size);

        buf.Write<uint16_t>(static_cast<uint16_t>(numCallstacks));
        if (numCallstacks > 0)
        {
            buf.Write(callstack64 + skipFrames, 8 * static_cast<size_t>(numCallstacks));
        }

        Send(buf.buf, buf.size);
    }


    static void MemoryFreeLocked(MemoryHeapHandle heap, const void *ptr)
    {
        if (!InitializeLocked())
        {
            return;
        }

        Buffer<14> buf;
        buf.Write<uint16_t>(FREE);
        buf.Write<uint16_t>(2 + 8);
        buf.Write<uint16_t>(static_cast<uint16_t>(heap | getpid() << 8));
        buf.Write<uint64_t>((uintptr_t)ptr);

        Send(buf.buf, buf.size);
    }


    static void SendHeapName(MemoryHeapHandle heap)
    {
        // 256 - headersize - rest of packet size
        const int MaxStrLen = 256 - 4 - (2 + 4);
        size_t len = strnlen(s_registeredHeaps[heap].name, MaxStrLen);

        Buffer<256> buf;
        buf.Write<uint16_t>(HEAP);
        buf.Write<uint16_t>(2 + 4 + static_cast<uint16_t>(len));
        buf.Write<uint16_t>(static_cast<uint16_t>(heap | getpid() << 8));
        buf.Write<uint32_t>(static_cast<uint32_t>(len));
        buf.Write(s_registeredHeaps[heap].name, len);

        Send(buf.buf, buf.size);
    }


    int PrintAppName(char *buf, size_t len)
    {
        const char *path = nullptr;
        size_t length;

        auto app = TargetApplication::GetCurrent();
        if (app != nullptr)
        {
            app->GetApplicationName(path, length);
        }
        if (path == nullptr)
        {
            return 0;
        }

        const char *name = path;
        const char *dot = NULL;

        size_t index = 0;
        while (*path && index <= length)
        {
            if (*path == '/' || *path == '\\')
            {
                name = path + 1;
            }
            else if (*path == '.')
            {
                dot = path;
            }
            path++;
            index++;
        }

        // Strip extension.
        if (dot > name)
        {
            size_t baselen = static_cast<size_t>(dot - name);
            if (len > baselen + 1)
            {
                len = baselen + 1;
            }
        }
        return snprintf(buf, len, "%s", name);
    }


    static void Send(const void *packet, size_t packetsize)
    {
        uint32_t get;
        uint32_t put;
        uint32_t newput;
        size_t freespace;
        int sleeps = 0;

        do
        {
            put = g_SharedArea->put;
            get = g_SharedArea->get;

            if (put >= get)
            {
                freespace = RingSize - (put - get);
            }
            else
            {
                freespace = get - put;
            }

            if (freespace > packetsize)
            {
                break;
            }

            if (!g_Connected)
            {
                // We have filled up the buffer. If user did not call
                // NvMemoryProfilerConnect before this happens, we assume
                // memory profiling is disabled.
                g_Enabled = false;
                return;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(10));

            if (sleeps++ == 10000)
            {
                INFO_LOG("Please connect memory profiling tool to proceed.\n");
            }
        } while (true);

        NN_SDK_ASSERT(IsMutexLockedByCurrentThread(&s_lock));

        // Make sure that another core is not still reading the bytes that
        // we're about to overwrite.
        std::atomic_thread_fence(std::memory_order_acquire);

        if (put + packetsize > RingSize)
        {
            uint32_t half = RingSize - put;
            memcpy(g_SharedArea->ring + put, packet, half);
            memcpy(g_SharedArea->ring, (char *)packet + half, packetsize - half);
            newput = static_cast<uint32_t>(packetsize - half);
        }
        else
        {
            memcpy(g_SharedArea->ring + put, packet, packetsize);
            newput = static_cast<uint32_t>(put + packetsize);
        }

        // Make sure all writes are seen by the other cores before updating
        // the put pointer.
        std::atomic_thread_fence(std::memory_order_release);

        g_SharedArea->put = newput;
        NN_SDK_ASSERT(newput != get);

        SendMemoryEventToPc();
    }
} // anonymous


nn::Result ConnectToMemoryProfiler()
{
    nn::Result result = nn::ResultSuccess();

    nn::os::LockMutex(&s_lock);

    if (g_Enabled)
    {
        if (g_SharedArea == nullptr)
        {
            InitSharedMemoryLocked();

            // Since we initialized shared memory late, we may have some heaps whose names
            // need to be sent over now.
            for (int heap = 0; heap < s_numHeaps; heap++)
            {
                SendHeapName(static_cast<MemoryHeapHandle>(heap));
            }
        }

        result = ConnectLocked();
    }
    else
    {
        // If g_Enabled == false at this point, then we have
        // exhausted the profiling buffer before arriving at
        // NvMemoryProfilerConnect.  That means we can't enable
        // reliable profiling any more since the data would be
        // corrupted.
    }

    nn::os::UnlockMutex(&s_lock);
    return result;
}


void DisconnectFromMemoryProfiler()
{
    nn::os::LockMutex(&s_lock);
    FreeSharedMemoryLocked();
    nn::os::UnlockMutex(&s_lock);
}


MemoryHeapHandle RegisterMemoryHeap(const char *name)
{
    nn::os::LockMutex(&s_lock);
    char appname[32];
    MemoryHeapHandle heap = MemoryHeapHandle(s_numHeaps++);

    NN_SDK_ASSERT(heap < NN_PROFILER_MEMORY_MAX_HEAPS);

    // Always store the heap name...
    if (PrintAppName(appname, sizeof(appname)))
    {
        snprintf(
            s_registeredHeaps[heap].name,
            sizeof(s_registeredHeaps[heap].name),
            "%02d - %s - %s",
            (int)getpid(),
            name,
            appname);
    }
    else
    {
        snprintf(
            s_registeredHeaps[heap].name,
            sizeof(s_registeredHeaps[heap].name),
            "%02d - %s",
            (int)getpid(),
            name);
    }

    // ...and if we are connected, send the heap name to the profiling client.
    // If we are not yet connected, the heap name will be sent when connecting.
    if (InitializeLocked())
    {
        SendHeapName(heap);
    }

    nn::os::UnlockMutex(&s_lock);
    return heap;
}


void SetMemoryApiLock()
{
    nn::os::LockMutex(&s_lock);
}


void BeginMemoryAllocation(MemoryHeapHandle heap)
{
    NN_UNUSED(heap);
    nn::os::LockMutex(&s_lock);
}


void BeginMemoryFree(MemoryHeapHandle heap)
{
    NN_UNUSED(heap);
    nn::os::LockMutex(&s_lock);
}


void BeginMemoryReallocation(MemoryHeapHandle heap)
{
    NN_UNUSED(heap);
    nn::os::LockMutex(&s_lock);
}


void EndMemoryAllocation(MemoryHeapHandle heap, const void *ptr, size_t size)
{
    MemoryAllocationLocked(heap, ptr, size);
    nn::os::UnlockMutex(&s_lock);
}


void EndMemoryFree(MemoryHeapHandle heap, const void *ptr)
{
    MemoryFreeLocked(heap, ptr);
    nn::os::UnlockMutex(&s_lock);
}


void EndMemoryReallocation(MemoryHeapHandle heap, const void *oldptr, const void *newptr, size_t newsize)
{
    MemoryFreeLocked(heap, oldptr);
    MemoryAllocationLocked(heap, newptr, newsize);
    nn::os::UnlockMutex(&s_lock);
}

}}
