﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/prepo/detail/service/core/prepo_UserAgreementChecker.h>
#include <nn/account.h>
#include <nn/account/account_ApiForSystemServices.h>

#define SKIP_CHECK_IF_DISABLED() \
    do                                          \
    {                                           \
        bool _isEnabled;                        \
        NN_RESULT_DO(IsEnabled(&_isEnabled));   \
        if (!_isEnabled)                        \
        {                                       \
            NN_RESULT_SUCCESS;                  \
        }                                       \
    }                                           \
    while (NN_STATIC_CONDITION(false))

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

namespace
{
    const char* FilePath = "prepo-sys:/is_user_agreement_check_enabled.bin";
    const size_t FileSize = sizeof(bool);
    const bool IsEnabledDefault = true;
}

nn::Result UserAgreementChecker::IsEnabled(bool* outIsEnabled) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outIsEnabled);

    NN_UTIL_LOCK_GUARD(m_EnableFlagMutex);

    NN_FUNCTION_LOCAL_STATIC(bool, s_IsInitialized, = false);

    if (!s_IsInitialized)
    {
        NN_DETAIL_PREPO_STORAGE_SCOPED_LOCK("prepo-sys");

        NN_RESULT_DO(IsEnabledImpl());

        s_IsInitialized = true;
    }

    *outIsEnabled = m_IsEnabled;

    NN_RESULT_SUCCESS;
}

nn::Result UserAgreementChecker::IsEnabledImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_EnableFlagMutex.IsLockedByCurrentThread());

    nn::fs::FileHandle handle = {};

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, FilePath, nn::fs::OpenMode_Read))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            m_IsEnabled = IsEnabledDefault;

            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    bool data;

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, &data, FileSize));

    m_IsEnabled = data;

    NN_RESULT_SUCCESS;
}

nn::Result UserAgreementChecker::SetEnabled(bool isEnabled) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_EnableFlagMutex);

    NN_DETAIL_PREPO_STORAGE_SCOPED_LOCK("prepo-sys");

    NN_RESULT_DO(SetEnabledImpl(isEnabled));

    NN_RESULT_DO(FileSystem::Commit("prepo-sys"));

    NN_RESULT_SUCCESS;
}

nn::Result UserAgreementChecker::SetEnabledImpl(bool isEnabled) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_EnableFlagMutex.IsLockedByCurrentThread());

    nn::fs::FileHandle handle = {};

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, FilePath, nn::fs::OpenMode_Write))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateFile(FilePath, FileSize));
            NN_RESULT_DO(nn::fs::OpenFile(&handle, FilePath, nn::fs::OpenMode_Write));
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::WriteFile(handle, 0, &isEnabled, FileSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

    m_IsEnabled = isEnabled;

    NN_RESULT_SUCCESS;
}

nn::Result UserAgreementChecker::IsUserAgreedWithAnalystics(const nn::account::NetworkServiceAccountId& nsaId) NN_NOEXCEPT
{
    SKIP_CHECK_IF_DISABLED();

    NN_UTIL_LOCK_GUARD(m_AgreementInfoMutex);

    auto end = m_Accounts + m_AccountCount;
    auto it = std::find_if(m_Accounts, end, [&nsaId](auto&& user){
        return user.nsaId == nsaId; });

    NN_RESULT_THROW_UNLESS(it != end && it->isAgreed, ResultTransmissionNotAgreed());

    NN_RESULT_SUCCESS;
}

nn::Result UserAgreementChecker::IsAnyoneAgreedWithAnalystics() NN_NOEXCEPT
{
    SKIP_CHECK_IF_DISABLED();

    NN_UTIL_LOCK_GUARD(m_AgreementInfoMutex);

    auto end = m_Accounts + m_AccountCount;
    auto it = std::find_if(m_Accounts, end, [](auto&& user){
        return user.isAgreed; });

    NN_RESULT_THROW_UNLESS(it != end, ResultTransmissionNotAgreed());

    NN_RESULT_SUCCESS;
}

nn::Result UserAgreementChecker::IsUserAgreedWithNaEula(const nn::account::NetworkServiceAccountId& nsaId) NN_NOEXCEPT
{
    SKIP_CHECK_IF_DISABLED();

    NN_UTIL_LOCK_GUARD(m_AgreementInfoMutex);

    auto end = m_Accounts + m_AccountCount;
    auto it = std::find_if(m_Accounts, end, [&nsaId](auto&& user){
        return user.nsaId == nsaId; });

    NN_RESULT_THROW_UNLESS(it != end, ResultTransmissionNotAgreed());

    NN_RESULT_SUCCESS;
}

nn::Result UserAgreementChecker::IsAnyoneAgreedWithNaEula() NN_NOEXCEPT
{
    SKIP_CHECK_IF_DISABLED();

    NN_UTIL_LOCK_GUARD(m_AgreementInfoMutex);

    NN_RESULT_THROW_UNLESS(m_AccountCount > 0, ResultTransmissionNotAgreed());

    NN_RESULT_SUCCESS;
}

nn::Result UserAgreementChecker::Update() NN_NOEXCEPT
{
    SKIP_CHECK_IF_DISABLED();

    NN_UTIL_LOCK_GUARD(m_AgreementInfoMutex);

    nn::account::Uid users[nn::account::UserCountMax] = {};
    int userCount;
    NN_RESULT_DO(nn::account::ListAllUsers(&userCount, users, NN_ARRAY_SIZE(users)));

    m_AccountCount = 0;

    for (int i = 0; i < userCount; i++)
    {
        if (m_NintendoAccountCacheExpiredUsers.Contains(users[i]))
        {
            continue;
        }

        nn::account::NetworkServiceAccountManager manager;
        if (nn::account::GetNetworkServiceAccountManager(&manager, users[i]).IsFailure())
        {
            continue;
        }

        if (manager.GetNetworkServiceAccountId(&(m_Accounts[m_AccountCount].nsaId)).IsFailure())
        {
            continue;
        }

        // 2 KiB あるのでスタックサイズに注意。
        Bit8 buffer[nn::account::RequiredBufferSizeForCachedNintendoAccountInfo];
        nn::account::CachedNintendoAccountInfoForSystemService accountInfo;

        m_Accounts[m_AccountCount].isAgreed = true
            && manager.LoadCachedNintendoAccountInfo(&accountInfo, buffer, sizeof(buffer)).IsSuccess()
            && accountInfo.GetAnalyticsOptedInFlag();

        m_AccountCount++;
    }

    NN_RESULT_SUCCESS;
}

void UserAgreementChecker::NotifyUpdateNintendoAccountCacheResult(const nn::account::Uid& uid, nn::Result result) NN_NOEXCEPT
{
    if (result.IsSuccess())
    {
        m_NintendoAccountCacheExpiredUsers.RemoveIfContains(uid);
    }
    else
    {
        m_NintendoAccountCacheExpiredUsers.Add(uid);
    }
}

}}}}}
