﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/prepo/detail/prepo_PlayReportGenerator.h>
#include <nn/prepo/detail/service/prepo_Common.h>
#include <nn/prepo/detail/service/core/prepo_ReportBuffer.h>
#include <nn/prepo/detail/service/core/prepo_SystemInfo.h>
#include <nn/prepo/detail/service/core/prepo_StatisticsManager.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

const char* StatisticsManager::FilePath = "prepo-sys:/statistics.bin";

StatisticsManager::StatisticsManager() NN_NOEXCEPT
    : m_Event(nullptr)
{
    std::memset(&m_Statistics, 0, sizeof (m_Statistics));
}

void StatisticsManager::AddUploaded(ReportCategory category, size_t size, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(size, 0u);
    NN_SDK_REQUIRES_GREATER(count, 0);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_Statistics.groups[category].uploaded.size += static_cast<int64_t>(size);
    m_Statistics.groups[category].uploaded.count += count;

    SignalUpdateEvent();
}

void StatisticsManager::AddLostByBufferShortage(ReportCategory category, size_t size, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(size, 0u);
    NN_SDK_REQUIRES_GREATER(count, 0);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_Statistics.groups[category].lostByBufferShortage.size += static_cast<int64_t>(size);
    m_Statistics.groups[category].lostByBufferShortage.count += count;

    SignalUpdateEvent();
}

void StatisticsManager::AddLostByStorageShortage(ReportCategory category, size_t size, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(size, 0u);
    NN_SDK_REQUIRES_GREATER(count, 0);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_Statistics.groups[category].lostByStorageShortage.size += static_cast<int64_t>(size);
    m_Statistics.groups[category].lostByStorageShortage.count += count;

    SignalUpdateEvent();
}

void StatisticsManager::AddLostByDisagree(ReportCategory category, size_t size, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(size, 0u);
    NN_SDK_REQUIRES_GREATER(count, 0);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_Statistics.groups[category].lostByDisagree.size += static_cast<int64_t>(size);
    m_Statistics.groups[category].lostByDisagree.count += count;

    SignalUpdateEvent();
}

void StatisticsManager::GetStatistics(Statistics* out) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(out);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    *out = m_Statistics;
}

nn::Result StatisticsManager::Save() NN_NOEXCEPT
{
    Statistics statistics;

    {
        NN_UTIL_LOCK_GUARD(m_Mutex);
        statistics = m_Statistics;
    }

    NN_DETAIL_PREPO_STORAGE_SCOPED_LOCK("prepo-sys");

    NN_RESULT_DO(SaveImpl(statistics));

    NN_RESULT_DO(FileSystem::Commit("prepo-sys"));

    NN_RESULT_SUCCESS;
}

nn::Result StatisticsManager::SaveImpl(const Statistics& statistics) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, FilePath, nn::fs::OpenMode_Write))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateFile(FilePath, sizeof (statistics)));
            NN_RESULT_DO(nn::fs::OpenFile(&handle, FilePath, nn::fs::OpenMode_Write));
        }
    NN_RESULT_END_TRY;

    bool isCloseRequired = true;

    NN_UTIL_SCOPE_EXIT
    {
        if (isCloseRequired)
        {
            nn::fs::CloseFile(handle);
        }
    };

    int64_t fileSize;

    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

    if (!(fileSize == sizeof (statistics)))
    {
        nn::fs::CloseFile(handle);

        isCloseRequired = false;

        NN_RESULT_DO(nn::fs::DeleteFile(FilePath));
        NN_RESULT_DO(nn::fs::CreateFile(FilePath, sizeof (statistics)));
        NN_RESULT_DO(nn::fs::OpenFile(&handle, FilePath, nn::fs::OpenMode_Write));

        isCloseRequired = true;
    }

    NN_RESULT_DO(nn::fs::WriteFile(handle, 0, &statistics, sizeof (statistics), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

    NN_RESULT_SUCCESS;
}

void StatisticsManager::SignalUpdateEvent() NN_NOEXCEPT
{
    if (m_Event)
    {
        nn::os::SignalEvent(m_Event);
    }
}

nn::Result StatisticsManager::Load() NN_NOEXCEPT
{
    Statistics statistics;

    NN_DETAIL_PREPO_STORAGE_SCOPED_LOCK("prepo-sys");

    NN_RESULT_TRY(LoadImpl(&statistics))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            std::memset(&statistics, 0, sizeof (statistics));
        }
    NN_RESULT_END_TRY;

    {
        NN_UTIL_LOCK_GUARD(m_Mutex);
        m_Statistics = statistics;
    }

    NN_RESULT_SUCCESS;
}

nn::Result StatisticsManager::LoadImpl(Statistics* pStatistics) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;

    NN_RESULT_DO(nn::fs::OpenFile(&handle, FilePath, nn::fs::OpenMode_Read));

    bool isCloseRequired = true;

    NN_UTIL_SCOPE_EXIT
    {
        if (isCloseRequired)
        {
            nn::fs::CloseFile(handle);
        }
    };

    int64_t fileSize;

    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

    if (!(fileSize == sizeof (*pStatistics)))
    {
        nn::fs::CloseFile(handle);

        isCloseRequired = false;

        NN_RESULT_DO(nn::fs::DeleteFile(FilePath));
        NN_RESULT_DO(FileSystem::Commit("prepo-sys"));
        NN_RESULT_THROW(nn::fs::ResultPathNotFound());
    }

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, pStatistics, sizeof (*pStatistics)));

    NN_RESULT_SUCCESS;
}

nn::Result StatisticsManager::Clear() NN_NOEXCEPT
{
    NN_DETAIL_PREPO_STORAGE_SCOPED_LOCK("prepo-sys");

    NN_RESULT_TRY(nn::fs::DeleteFile(FilePath))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            // ファイルが存在しないときは何もしない。
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO(FileSystem::Commit("prepo-sys"));

    {
        NN_UTIL_LOCK_GUARD(m_Mutex);
        std::memset(&m_Statistics, 0, sizeof (m_Statistics));
    }

    NN_RESULT_SUCCESS;
}

void StatisticsManager::RegisterUpdateEvent(nn::os::EventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(event);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_Event = event;
}

nn::os::EventType* StatisticsManager::UnregisterUpdateEvent() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    auto event = m_Event;

    m_Event = nullptr;

    return event;
}

nn::Result StatisticsManager::Post() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(nn::os::SdkMutex, s_PostMutex);
    NN_UTIL_LOCK_GUARD(s_PostMutex);

    Statistics statistics;

    {
        NN_UTIL_LOCK_GUARD(m_Mutex);
        statistics = m_Statistics;
    }

    const int KeyValueCount = 8 * ReportCategory_Count;

    NN_FUNCTION_LOCAL_STATIC(nn::Bit8, s_Data[nn::prepo::ReportBufferSizeMin + (nn::prepo::KeyValueSizeMax * KeyValueCount)]);

    size_t position;

    detail::PlayReportGenerator::Initialize(&position, s_Data, sizeof (s_Data));

    const char* KeySuffixes[] =
    {
        "",
        "_of_anti_piracy"
    };
    NN_STATIC_ASSERT(NN_ARRAY_SIZE(KeySuffixes) == ReportCategory_Count);

    auto Add = [&](ReportCategory category, const char* key, auto value)
    {
        char keyBuffer[KeyLengthMax + 1];
        const int count = nn::util::SNPrintf(keyBuffer, sizeof (keyBuffer), "%s%s", key, KeySuffixes[category]);
        NN_UNUSED(count);
        NN_SDK_ASSERT_LESS(static_cast<size_t>(count), sizeof (keyBuffer));

        return detail::PlayReportGenerator::AddKeyValue(
            &position, keyBuffer, value, s_Data, sizeof (s_Data), position);
    };

    for (int i = 0; i < ReportCategory_Count; i++)
    {
        const auto category = static_cast<ReportCategory>(i);

        NN_RESULT_DO(Add(category, "uploaded_size", statistics.groups[category].uploaded.size));
        NN_RESULT_DO(Add(category, "uploaded_count", statistics.groups[category].uploaded.count));
        NN_RESULT_DO(Add(category, "lost_by_buffer_shortage_size", statistics.groups[category].lostByBufferShortage.size));
        NN_RESULT_DO(Add(category, "lost_by_buffer_shortage_count", statistics.groups[category].lostByBufferShortage.count));
        NN_RESULT_DO(Add(category, "lost_by_storage_shortage_size", statistics.groups[category].lostByStorageShortage.size));
        NN_RESULT_DO(Add(category, "lost_by_storage_shortage_count", statistics.groups[category].lostByStorageShortage.count));
        NN_RESULT_DO(Add(category, "lost_by_disagree_size", statistics.groups[category].lostByDisagree.size));
        NN_RESULT_DO(Add(category, "lost_by_disagree_count", statistics.groups[category].lostByDisagree.count));
    }

    NN_SDK_ASSERT_EQUAL(detail::PlayReportGenerator::GetKeyValueCount(s_Data, sizeof (s_Data)), static_cast<int>(KeyValueCount));

    const char eventId[] = "statistics";
    const nn::ApplicationId appId = { 0x0100000000001023 };

    NN_FUNCTION_LOCAL_STATIC(nn::Bit8, s_SysInfo[SystemInfo::RequiredBufferSize]);
    size_t sysInfoSize = 0;
    NN_RESULT_DO(SystemInfo::CollectSync(&sysInfoSize, s_SysInfo, sizeof (s_SysInfo), appId, eventId));

    NN_RESULT_DO(ReportBuffer::GetInstance().Add(ReportCategory_Normal, s_Data, position, s_SysInfo, sysInfoSize));

    NN_RESULT_SUCCESS;
}

}}}}}
