﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  SDEV を利用したデバッグ機能に関する API
*/

#include <mutex>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/pl/pl_Result.h>
#include <nn/pl/pl_BridgeApi.h>
#include <nn/os.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/htc/htc_IHtcManager.h>
#else
#include "pl_HtcManagerByHipc.h"
#endif

//--------------------------------------------------------------------------
//  C++ 向けの宣言
//--------------------------------------------------------------------------
namespace nn { namespace pl {

namespace
{
    struct StaticMutex
    {
        os::MutexType    m_Mutex;
        void lock() NN_NOEXCEPT
        {
            nn::os::LockMutex( &m_Mutex );
        }
        void unlock() NN_NOEXCEPT
        {
            nn::os::UnlockMutex( &m_Mutex );
        }
    };
    StaticMutex g_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

    Result ConvertToPlResult(nn::Result result)
    {
        if( nn::os::ResultNotSupported::Includes(result) )
        {
            return nn::pl::ResultNotSupportedDevice();
        }
        else
        {
            return result;
        }
    }
}

    Result SetUsbDetachedForDebug(bool bDetached) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        NN_UNUSED(bDetached);

        return nn::pl::ResultNotSupportedDevice();
#else
        nn::sf::SharedPointer<tma::IHtcManager> manager;
        std::lock_guard<StaticMutex> lock(g_Mutex);

        manager = htc::CreateHtcManagerByHipc();

        Result result = manager->SetCradleAttached(!bDetached);
        manager = nullptr;

        return ConvertToPlResult(result);
#endif
    }
}}
