﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pdm/srv/pdm_QueryServiceImpl.h>
#include <nn/pdm/pdm_PrivateTypes.h>
#include <nn/pdm/pdm_SystemTypes.h>
#include <nn/pdm/pdm_QueryLastPlayTimeApi.h>
#include <nn/pdm/detail/pdm_PlayEventBuffer.h>
#include <nn/pdm/detail/pdm_AccountPlayEventBuffer.h>
#include <nn/pdm/detail/pdm_Fs.h>
#include <nn/pdm/detail/pdm_Log.h>
#include <nn/pdm/detail/pdm_Time.h>
#include <nn/pdm/detail/pdm_Util.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/time.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace pdm { namespace srv {

namespace
{
    void SetEventTimeData(EventTimeData* outValue, const PlayEvent& src, uint32_t index) NN_NOEXCEPT
    {
        outValue->eventIndex = index;
        outValue->userClockTime = detail::GetElapsedMinutesSinceInputPosixTimeMin(src.userTime);
        outValue->networkClockTime = detail::GetElapsedMinutesSinceInputPosixTimeMin(src.networkTime);
    }

    void InitializePlayStatistics(PlayStatistics* outValue, const ncm::ApplicationId& applicationId, const PlayEvent& firstEvent, uint32_t firstEventIndex) NN_NOEXCEPT
    {
        detail::QueryServiceHelper::Initialize(outValue, applicationId);
        SetEventTimeData(&(outValue->firstEventTime), firstEvent, firstEventIndex);
    }

    void InitializeApplicationPlayStatistics(ApplicationPlayStatistics* outValue, const ncm::ApplicationId& applicationId) NN_NOEXCEPT
    {
        outValue->applicationId.value = applicationId.value;
        outValue->totalPlayCount = 0;
        outValue->totalPlayTime = nn::TimeSpan(0);
    }

    util::optional<int> Find(const ApplicationPlayStatistics stats[], int count, Bit64 applicationIdValue) NN_NOEXCEPT
    {
        for( int i = 0; i < count; i++ )
        {
            if( stats[i].applicationId.value == applicationIdValue )
            {
                return i;
            }
        }
        return util::optional<int>();
    }

    // 秒単位で加算しているので注意。最終的にクライアントに返す値は分単位です。
    void UpdateTotalTime(PlayStatistics* outValue, const PlayEvent& start, const PlayEvent& end) NN_NOEXCEPT
    {
        // アプレットのイベントの end から start までに単調増加時計のリセットを含むことはないのでその値を使用する。
        if( end.steadyTime >= start.steadyTime )
        {
            auto time = end.steadyTime - start.steadyTime;
            outValue->totalPlayTime += static_cast<uint32_t>(time);
        }
        else
        {
            NN_DETAIL_PDM_WARN("Unexpected PlayEvent pair in UpdateTotalTime : end.steadyTime(%u) < start.steadyTime(%u).\n", end.steadyTime, start.steadyTime);
        }
    }

    void UpdateTotalTime(ApplicationPlayStatistics* outValue, const PlayEvent& start, const PlayEvent& end) NN_NOEXCEPT
    {
        if( end.steadyTime >= start.steadyTime )
        {
            auto time = end.steadyTime - start.steadyTime;
            outValue->totalPlayTime += nn::TimeSpan::FromSeconds(time);
        }
        else
        {
            NN_DETAIL_PDM_WARN("Unexpected PlayEvent pair in UpdateTotalTime : end.steadyTime(%u) < start.steadyTime(%u).\n", end.steadyTime, start.steadyTime);
        }
    }

    NN_FORCEINLINE void ConvertTotalTimeFromSecondsToMinutes(PlayStatistics outList[], int count) NN_NOEXCEPT
    {
        detail::QueryServiceHelper::ConvertTotalTimeFromSecondsToMinutes(outList, count);
    }

    nn::util::optional<int> GetPlayStatisticsIndex(const nn::pdm::PlayStatistics list[], int applicationCount, const nn::ncm::ApplicationId& applicationId) NN_NOEXCEPT
    {
        for( int i = 0; i < applicationCount; i++ )
        {
            if( list[i].applicationId.value == applicationId.value )
            {
                return i;
            }
        }
        return nn::util::optional<int>();
    }

    NN_FORCEINLINE int SelectPlayStatisticsIndexToReplace(const pdm::PlayStatistics list[], int count) NN_NOEXCEPT
    {
        return detail::QueryServiceHelper::SelectPlayStatisticsIndexToReplace(list, count);
    }

    void InitializeLastPlayTime(pdm::LastPlayTime* pOutValue) NN_NOEXCEPT
    {
        pOutValue->applicationId = nn::ApplicationId::GetInvalidId();
        pOutValue->userClockTime = 0;
        pOutValue->networkClockTime = 0;
        pOutValue->isElapsedMinutesAvailable = false;
        pOutValue->elapsedMinutesSinceLastPlay = 0;
    }

    util::optional<int> Find(const nn::ApplicationId list[], int count, nn::Bit64 applicationIdValue) NN_NOEXCEPT
    {
        for( int i = 0; i < count; i++ )
        {
            if( list[i].value == applicationIdValue )
            {
                return i;
            }
        }
        return util::optional<int>();
    }

    void SetLastPlayTime(LastPlayTime* pOutValue, const PlayEvent& playEvent, uint32_t eventIndex, int64_t currentSteadyClockTime) NN_NOEXCEPT
    {
        pOutValue->applicationId = nn::ApplicationId{ detail::GetApplicationId(playEvent).value };
        pOutValue->userClockTime = detail::GetElapsedMinutesSinceInputPosixTimeMin(playEvent.userTime);
        pOutValue->networkClockTime = detail::GetElapsedMinutesSinceInputPosixTimeMin(playEvent.networkTime);

        if( eventIndex > detail::GetLastSteadyClockResetEventIndex() )
        {
            if( currentSteadyClockTime >= playEvent.steadyTime )
            {
                pOutValue->isElapsedMinutesAvailable = true;
                pOutValue->elapsedMinutesSinceLastPlay = static_cast<uint32_t>( (currentSteadyClockTime - playEvent.steadyTime) / 60 );
            }
            else
            {
                NN_DETAIL_PDM_WARN("elapsedMinutesSinceLastPlay should be available, but currentSteadyClockTime(%u) is smaller than playEvent.steadyTime(%u).\n", currentSteadyClockTime, playEvent.steadyTime);
                pOutValue->isElapsedMinutesAvailable = false;
                pOutValue->elapsedMinutesSinceLastPlay = 0;
            }
        }
        else
        {
            pOutValue->isElapsedMinutesAvailable = false;
            pOutValue->elapsedMinutesSinceLastPlay = 0;
        }
    }
}

QueryServiceImpl::QueryServiceImpl(pdm::detail::LockableMemoryBuffer* pLockableBuffer) NN_NOEXCEPT
    : m_pLockableBuffer(pLockableBuffer)
    , m_PlayEventReadBuffer(reinterpret_cast<PlayEvent*>(pLockableBuffer->GetPointer()))
    , m_PlayEventReadBufferCount(static_cast<int>(pLockableBuffer->GetCapacity() / sizeof(PlayEvent)))
{
}

nn::Result QueryServiceImpl::QueryApplicationEvent(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::pdm::ApplicationEvent>& outList, std::int32_t eventIndexOffset) NN_NOEXCEPT
{
    auto& buffer = *m_pLockableBuffer;
    NN_UTIL_LOCK_GUARD(buffer);

    detail::PlayEventBuffer::GetInstance().Flush();
    detail::CommitSaveData();
    detail::PlayEventBuffer::GetInstance().BeginRead();
    NN_UTIL_SCOPE_EXIT{ detail::PlayEventBuffer::GetInstance().EndRead(); };

    int32_t outCountMax = static_cast<int32_t>(outList.GetLength());
    *outCount = 0;
    uint32_t startIndex = std::max(static_cast<uint32_t>(eventIndexOffset), detail::PlayEventBuffer::GetInstance().GetStartIndex());
    uint32_t readCountSum = 0;
    while( NN_STATIC_CONDITION(true) )
    {
        uint32_t readCount = detail::PlayEventBuffer::GetInstance().Read(m_PlayEventReadBuffer, startIndex + readCountSum, m_PlayEventReadBufferCount);
        if( readCount == 0 )
        {
            // 最後まで読み込み、これ以上読み込むものがなくなった。
            NN_RESULT_SUCCESS;
        }
        for( uint32_t i = 0; i < readCount; i++ )
        {
            // 読み込んだイベントの中からアプリケーションのイベントを出力に詰める。
            if( detail::IsApplicationEvent(m_PlayEventReadBuffer[i]) && m_PlayEventReadBuffer[i].appletEventData.playLogPolicy == ns::PlayLogPolicy::All )
            {
                detail::ConvertAppletEventToApplicationEvent(&outList[*outCount], m_PlayEventReadBuffer[i]);
                outList[*outCount].eventTime.eventIndex = startIndex + readCountSum + i;
                *outCount = *outCount + 1;
                if( *outCount == outCountMax )
                {
                    NN_RESULT_SUCCESS;
                }
            }
        }
        readCountSum += readCount;
    }
}

nn::Result QueryServiceImpl::QueryAccountEvent(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::pdm::AccountEvent>& outList, std::int32_t eventIndexOffset) NN_NOEXCEPT
{
    auto& buffer = *m_pLockableBuffer;
    NN_UTIL_LOCK_GUARD(buffer);

    detail::PlayEventBuffer::GetInstance().Flush();
    detail::CommitSaveData();
    detail::PlayEventBuffer::GetInstance().BeginRead();
    NN_UTIL_SCOPE_EXIT{ detail::PlayEventBuffer::GetInstance().EndRead(); };

    int32_t outCountMax = static_cast<int32_t>(outList.GetLength());
    *outCount = 0;
    uint32_t startIndex = std::max(static_cast<uint32_t>(eventIndexOffset), detail::PlayEventBuffer::GetInstance().GetStartIndex());
    uint32_t readCountSum = 0;
    while( NN_STATIC_CONDITION(true) )
    {
        uint32_t readCount = detail::PlayEventBuffer::GetInstance().Read(m_PlayEventReadBuffer, startIndex + readCountSum, m_PlayEventReadBufferCount);
        if( readCount == 0 )
        {
            // 最後まで読み込み、これ以上読み込むものがなくなった。
            NN_RESULT_SUCCESS;
        }
        for( uint32_t i = 0; i < readCount; i++ )
        {
            // 読み込んだイベントの中からアカウントのイベントを出力に詰める。
            if( m_PlayEventReadBuffer[i].eventCategory == PlayEventCategory::UserAccount
             && (m_PlayEventReadBuffer[i].userAccountEventData.eventType == UserAccountEventType::Open || m_PlayEventReadBuffer[i].userAccountEventData.eventType == UserAccountEventType::Close))
            {
                outList[*outCount].uid = detail::GetUid(m_PlayEventReadBuffer[i]);
                outList[*outCount].eventTime.eventIndex = startIndex + readCountSum + i;
                outList[*outCount].eventTime.userClockTime = m_PlayEventReadBuffer[i].userTime;
                outList[*outCount].eventTime.networkClockTime = m_PlayEventReadBuffer[i].networkTime;
                outList[*outCount].eventTime.steadyClockTimeValueSinceBase = m_PlayEventReadBuffer[i].steadyTime;;
                outList[*outCount].eventType = m_PlayEventReadBuffer[i].userAccountEventData.eventType;
                *outCount = *outCount + 1;
                if( *outCount == outCountMax )
                {
                    NN_RESULT_SUCCESS;
                }
            }
        }
        readCountSum += readCount;
    }
}

nn::Result QueryServiceImpl::QueryPlayStatistics(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::pdm::PlayStatistics>& outList) NN_NOEXCEPT
{
    auto& buffer = *m_pLockableBuffer;
    NN_UTIL_LOCK_GUARD(buffer);

    detail::PlayEventBuffer::GetInstance().Flush();
    detail::CommitSaveData();
    detail::PlayEventBuffer::GetInstance().BeginRead();
    NN_UTIL_SCOPE_EXIT{ detail::PlayEventBuffer::GetInstance().EndRead(); };

    int32_t applicationCount = 0;
    nn::ncm::ApplicationId currentApplication = ncm::ApplicationId::GetInvalidId();

    uint32_t startIndex = detail::PlayEventBuffer::GetInstance().GetStartIndex();
    uint32_t readCountSum = 0;
    nn::util::optional<PlayEvent> startEvent;

    while( NN_STATIC_CONDITION(true) )
    {
        uint32_t readCount = detail::PlayEventBuffer::GetInstance().Read(m_PlayEventReadBuffer, startIndex + readCountSum, m_PlayEventReadBufferCount);
        if( readCount == 0 )
        {
            // 最後まで読み込み、これ以上読み込むものがなくなった。
            *outCount = applicationCount;
            ConvertTotalTimeFromSecondsToMinutes(outList.GetData(), *outCount);
            NN_RESULT_SUCCESS;
        }
        for( uint32_t i = 0; i < readCount; i++ )
        {
            const auto& currentEvent = m_PlayEventReadBuffer[i];

            if( detail::IsApplicationEvent(currentEvent) && currentEvent.appletEventData.playLogPolicy == ns::PlayLogPolicy::All )
            {
                auto currentEventIndex = startIndex + readCountSum + i;
                auto applicationId = nn::pdm::detail::GetApplicationId(currentEvent);
                auto statisticsIndex = GetPlayStatisticsIndex(outList.GetData(), applicationCount, applicationId);
                if( !statisticsIndex )
                {
                    if( applicationCount < static_cast<int32_t>(outList.GetLength()) )
                    {
                        statisticsIndex = applicationCount;
                        applicationCount++;
                    }
                    else
                    {
                        statisticsIndex = SelectPlayStatisticsIndexToReplace(outList.GetData(), applicationCount);
                    }
                    InitializePlayStatistics(&outList[*statisticsIndex], applicationId, currentEvent, currentEventIndex);
                }
                // 最終イベントの更新（イベントの種類に関わらず更新）。
                SetEventTimeData(&outList[*statisticsIndex].latestEventTime, currentEvent, currentEventIndex);

                switch( currentEvent.appletEventData.eventType )
                {
                // Launch と InFocus の場合に開始イベントを設定。
                case AppletEventType::Launch:
                    outList[*statisticsIndex].totalPlayCount++;
                    NN_FALL_THROUGH;
                case AppletEventType::InFocus:
                    currentApplication = applicationId;
                    startEvent = currentEvent;
                    break;
                // 終了系イベントの場合に開始イベントまでの時間をプレイ時間に追加。
                case AppletEventType::OutOfFocus:
                case AppletEventType::Background:
                case AppletEventType::Exit:
                case AppletEventType::AbnormalExit:
                case AppletEventType::Terminate:
                    if( startEvent )
                    {
                        if( currentApplication == applicationId )
                        {
                            UpdateTotalTime(&outList[*statisticsIndex], *startEvent, currentEvent);
                        }
                        else
                        {
                            NN_DETAIL_PDM_WARN("Unexpected event sequence. ApplicationId mismatch : 0x%16x -> 0x%16x\n",
                                detail::GetApplicationId(*startEvent).value, detail::GetApplicationId(currentEvent).value);
                        }
                        startEvent = nullptr;
                    }
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
            else if( detail::IsPowerOnEvent(currentEvent) )
            {
                startEvent = nullptr;
            }
            else if( detail::IsPowerOffEvent(currentEvent) && startEvent )
            {
                // アプリの起点が設定されている状態で電源オフされた場合、電源オフをアプリ終了のイベントとみなす。
                auto applicationId = nn::pdm::detail::GetApplicationId(*startEvent);
                auto statisticsIndex = GetPlayStatisticsIndex(outList.GetData(), applicationCount, applicationId);
                if( !statisticsIndex )
                {
                    continue;
                }
                SetEventTimeData(&outList[*statisticsIndex].latestEventTime, currentEvent, startIndex + readCountSum + i);
                UpdateTotalTime(&outList[*statisticsIndex], *startEvent, currentEvent);
                startEvent = nullptr;
            }
        }
        readCountSum += readCount;
    }

    NN_RESULT_SUCCESS;
}

nn::Result QueryServiceImpl::QueryPlayStatisticsByUserAccountId(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::pdm::PlayStatistics>& outList, const nn::account::Uid& user) NN_NOEXCEPT
{
    auto& buffer = *m_pLockableBuffer;
    NN_UTIL_LOCK_GUARD(buffer);

    uint32_t count = 0;
    uint32_t capacity = static_cast<uint32_t>(outList.GetLength());
    const auto result = detail::AccountPlayEventProvider::GetInstance().Query(&count, outList.GetData(), capacity, buffer.GetPointer(), buffer.GetCapacity(), user);
    if (result.IsFailure())
    {
        NN_DETAIL_PDM_WARN("[pdm] QueryPlayStatistics(0x%016llx-0x%016llx): Unexpected result received. -> (0x%lx)\n",
            user._data[0], user._data[1], result.GetInnerValueForDebug());
        count = 0;
    }
    *outCount = count;
    NN_RESULT_SUCCESS;
}

nn::Result QueryServiceImpl::QueryPlayStatisticsByNetworkServiceAccountId(nn::sf::Out<std::int32_t>, const nn::sf::OutArray<nn::pdm::PlayStatistics>&, nn::account::NetworkServiceAccountId) NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

nn::Result QueryServiceImpl::QueryPlayStatisticsByApplicationId(nn::sf::Out<nn::pdm::PlayStatistics> outValue, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    auto& buffer = *m_pLockableBuffer;
    NN_UTIL_LOCK_GUARD(buffer);

    detail::PlayEventBuffer::GetInstance().Flush();
    detail::CommitSaveData();
    detail::PlayEventBuffer::GetInstance().BeginRead();
    NN_UTIL_SCOPE_EXIT{ detail::PlayEventBuffer::GetInstance().EndRead(); };

    uint32_t startIndex = detail::PlayEventBuffer::GetInstance().GetStartIndex();
    uint32_t readCountSum = 0;
    nn::util::optional<PlayEvent> startEvent;
    bool isFirstEvent = true;

    while( NN_STATIC_CONDITION(true) )
    {
        uint32_t readCount = detail::PlayEventBuffer::GetInstance().Read(m_PlayEventReadBuffer, startIndex + readCountSum, m_PlayEventReadBufferCount);
        if( readCount == 0 )
        {
            // 最後まで読み込み、これ以上読み込むものがなくなった。
            ConvertTotalTimeFromSecondsToMinutes(outValue.GetPointer(), 1);
            NN_RESULT_SUCCESS;
        }
        for( uint32_t i = 0; i < readCount; i++ )
        {
            const auto& currentEvent = m_PlayEventReadBuffer[i];

            if( detail::IsApplicationEvent(currentEvent) && detail::GetApplicationId(currentEvent) == applicationId )
            {
                auto currentEventIndex = startIndex + readCountSum + i;
                if( isFirstEvent )
                {
                    InitializePlayStatistics(outValue.GetPointer(), applicationId, currentEvent, currentEventIndex);
                    isFirstEvent = false;
                }
                // 最終イベントの更新（イベントの種類に関わらず更新）。
                SetEventTimeData(&(*outValue).latestEventTime, currentEvent, currentEventIndex);

                switch( currentEvent.appletEventData.eventType )
                {
                    // Launch と InFocus の場合に開始イベントを設定。
                case AppletEventType::Launch:
                    (*outValue).totalPlayCount++;
                    NN_FALL_THROUGH;
                case AppletEventType::InFocus:
                    startEvent = currentEvent;
                    break;
                    // 終了系イベントの場合に開始イベントまでの時間をプレイ時間に追加。
                case AppletEventType::OutOfFocus:
                case AppletEventType::Background:
                case AppletEventType::Exit:
                case AppletEventType::AbnormalExit:
                case AppletEventType::Terminate:
                    if( startEvent )
                    {
                        UpdateTotalTime(outValue.GetPointer(), *startEvent, currentEvent);
                        startEvent = nullptr;
                    }
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
            else if( detail::IsPowerOnEvent(currentEvent) )
            {
                startEvent = nullptr;
            }
            else if( detail::IsPowerOffEvent(currentEvent) && startEvent )
            {
                // アプリの起点が設定されている状態で電源オフされた場合、電源オフをアプリ終了のイベントとみなす。
                SetEventTimeData(&(*outValue).latestEventTime, currentEvent, startIndex + readCountSum + i);
                UpdateTotalTime(outValue.GetPointer(), *startEvent, currentEvent);
                startEvent = nullptr;
            }
        }
        readCountSum += readCount;
    }

    NN_RESULT_SUCCESS;
}

nn::Result QueryServiceImpl::QueryPlayStatisticsByApplicationIdAndUserAccountId(nn::sf::Out<nn::pdm::PlayStatistics> outValue, nn::ncm::ApplicationId applicationId, const nn::account::Uid& user) NN_NOEXCEPT
{
    auto& buffer = *m_pLockableBuffer;
    NN_UTIL_LOCK_GUARD(buffer);

    const auto result = detail::AccountPlayEventProvider::GetInstance().Query(outValue.GetPointer(), applicationId, buffer.GetPointer(), buffer.GetCapacity(), user);
    if (result.IsFailure())
    {
        detail::QueryServiceHelper::Initialize(outValue.GetPointer(), applicationId);
        NN_DETAIL_PDM_WARN("[pdm] QueryPlayStatistics(0x%016llx, 0x%016llx-0x%016llx): Unexpected result received. -> (0x%lx)\n",
            applicationId.value, user._data[0], user._data[1], result.GetInnerValueForDebug());
    }
    NN_RESULT_SUCCESS;
}

nn::Result QueryServiceImpl::QueryPlayStatisticsByApplicationIdAndNetworkServiceAccountId(nn::sf::Out<nn::pdm::PlayStatistics>, nn::ncm::ApplicationId, nn::account::NetworkServiceAccountId) NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

nn::Result QueryServiceImpl::QueryLastPlayTime(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::pdm::LastPlayTime>& outList, const nn::sf::InArray<nn::ApplicationId>& applicationIdList) NN_NOEXCEPT
{
    auto& buffer = *m_pLockableBuffer;
    NN_UTIL_LOCK_GUARD(buffer);

    detail::PlayEventBuffer::GetInstance().Flush();
    detail::CommitSaveData();
    detail::PlayEventBuffer::GetInstance().BeginRead();
    NN_UTIL_SCOPE_EXIT{ detail::PlayEventBuffer::GetInstance().EndRead(); };

    *outCount = 0;
    auto currentSteadyClockTime = detail::GetSteadyClockTimeForPlayEvent();
    for( size_t i = 0; i < outList.GetLength(); i++ )
    {
        InitializeLastPlayTime(&outList[i]);
    }

    if( detail::PlayEventBuffer::GetInstance().GetCount() == 0 )
    {
        NN_DETAIL_PDM_INFO("QueryLastPlayTime : No PlayEvent is stored yet.\n");
        NN_RESULT_SUCCESS;
    }

    uint32_t bufferStartIndex = detail::PlayEventBuffer::GetInstance().GetStartIndex();
    uint32_t readEndIndex = detail::PlayEventBuffer::GetInstance().GetLastIndex();
    uint32_t readStartIndex = (readEndIndex > static_cast<uint32_t>(m_PlayEventReadBufferCount - 1)) ? (readEndIndex - m_PlayEventReadBufferCount + 1) : bufferStartIndex;

    while( NN_STATIC_CONDITION(true) )
    {
        uint32_t readCount = detail::PlayEventBuffer::GetInstance().Read(m_PlayEventReadBuffer, readStartIndex, (readEndIndex + 1u - readStartIndex));
        NN_SDK_ASSERT(readCount > 0);
        for( uint32_t i = readCount - 1; /* i >= 0 */ ; i-- )
        {
            const auto& currentEvent = m_PlayEventReadBuffer[i];

            if( detail::IsApplicationEvent(currentEvent) )
            {
                if( currentEvent.appletEventData.playLogPolicy != ns::PlayLogPolicy::All )
                {
                    NN_DETAIL_PDM_WARN("QueryLastPlayTime : Application 0x%016x is not queriable.\n", detail::GetApplicationId(currentEvent).value);
                    continue;
                }

                auto applicationId = detail::GetApplicationId(currentEvent);
                auto targetIndex = Find(applicationIdList.GetData(), static_cast<int>(applicationIdList.GetLength()), applicationId.value);

                if( targetIndex )
                {
                    if( outList[*targetIndex].applicationId == nn::ApplicationId::GetInvalidId() )
                    {
                        SetLastPlayTime(&outList[*targetIndex], currentEvent, readStartIndex + i, currentSteadyClockTime);
                        (*outCount)++;
                        if( static_cast<size_t>(*outCount) == outList.GetLength() )
                        {
                            NN_RESULT_SUCCESS;
                        }
                    }
                    else
                    {
                        // 既にイベント登録済みなので何もしない。
                    }
                }
            }
            if( i == 0u )
            {
                break;
            }
        }

        if( readStartIndex == bufferStartIndex )
        {
            // 始点まで読み込みが終わったので終了。
            NN_RESULT_SUCCESS;
        }

        readEndIndex = readStartIndex - 1;
        if( readStartIndex - bufferStartIndex > static_cast<uint32_t>(m_PlayEventReadBufferCount) )
        {
            readStartIndex -= m_PlayEventReadBufferCount;
        }
        else
        {
            readStartIndex = bufferStartIndex;
        }
    }
    NN_RESULT_SUCCESS;
}

nn::Result QueryServiceImpl::QueryPlayEvent(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::pdm::PlayEvent>& outList, std::int32_t offset) NN_NOEXCEPT
{
    detail::PlayEventBuffer::GetInstance().Flush();
    detail::CommitSaveData();
    detail::PlayEventBuffer::GetInstance().BeginRead();
    NN_UTIL_SCOPE_EXIT{ detail::PlayEventBuffer::GetInstance().EndRead(); };

    *outCount = detail::PlayEventBuffer::GetInstance().Read(outList.GetData(),
        detail::PlayEventBuffer::GetInstance().GetStartIndex() + static_cast<uint32_t>(offset), static_cast<uint32_t>(outList.GetLength()));

    NN_RESULT_SUCCESS;
}

nn::Result QueryServiceImpl::GetAvailablePlayEventRange(nn::sf::Out<std::int32_t> outCount, nn::sf::Out<std::int32_t> startIndex, nn::sf::Out<std::int32_t> lastIndex) NN_NOEXCEPT
{
    detail::PlayEventBuffer::GetInstance().Flush();
    detail::CommitSaveData();
    *outCount = detail::PlayEventBuffer::GetInstance().GetCount();
    *startIndex = detail::PlayEventBuffer::GetInstance().GetStartIndex();
    *lastIndex = detail::PlayEventBuffer::GetInstance().GetLastIndex();
    NN_RESULT_SUCCESS;
}

nn::Result QueryServiceImpl::QueryAccountPlayEvent(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::pdm::AccountPlayEvent>& outList, std::int32_t offset, const nn::account::Uid& user) NN_NOEXCEPT
{
    uint32_t count = 0;
    uint32_t capacity = static_cast<uint32_t>(outList.GetLength());
    const auto result = detail::AccountPlayEventProvider::GetInstance().QueryRaw(&count, outList.GetData(), capacity, offset, user);
    if (result.IsFailure())
    {
        NN_DETAIL_PDM_WARN("[pdm] QueryAccountPlayEvent[0x%016llx-0x%016llx](%ld -> %lu): Unexpected result received. -> (0x%lx)\n",
            user._data[0], user._data[1], offset, capacity, result.GetInnerValueForDebug());
        count = 0;
    }
    *outCount = count;
    NN_RESULT_SUCCESS;
}

nn::Result QueryServiceImpl::GetAvailableAccountPlayEventRange(nn::sf::Out<std::int32_t> outCount, nn::sf::Out<std::int32_t> startIndex, nn::sf::Out<std::int32_t> lastIndex, const nn::account::Uid& user) NN_NOEXCEPT
{
    uint32_t count = 0;
    uint32_t start = 0;
    uint32_t last = 0;
    const auto result = detail::AccountPlayEventProvider::GetInstance().QueryAvailableRange(&count, &start, &last, user);
    if (result.IsFailure())
    {
        NN_DETAIL_PDM_WARN("[pdm] GetAvailableAccountPlayEventRange[0x%016llx-0x%016llx](%lu, %lu, %lu): Unexpected result received. -> (0x%lx)\n",
            user._data[0], user._data[1], count, start, last, result.GetInnerValueForDebug());
        count = 0;
        start = 0;
        last = 0;
    }
    *outCount = count;
    *lastIndex = last;
    *startIndex = start;
    NN_RESULT_SUCCESS;
}

nn::Result QueryServiceImpl::QueryApplicationPlayStatisticsForSystem(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::pdm::ApplicationPlayStatistics>& outList, const nn::sf::InArray<nn::ncm::ApplicationId>& applicationIdList) NN_NOEXCEPT
{
    auto& buffer = *m_pLockableBuffer;
    NN_UTIL_LOCK_GUARD(buffer);

    detail::PlayEventBuffer::GetInstance().Flush();
    detail::CommitSaveData();
    detail::PlayEventBuffer::GetInstance().BeginRead();
    NN_UTIL_SCOPE_EXIT{ detail::PlayEventBuffer::GetInstance().EndRead(); };

    nn::ncm::ApplicationId currentApplication = ncm::ApplicationId::GetInvalidId();

    uint32_t startIndex = detail::PlayEventBuffer::GetInstance().GetStartIndex();
    uint32_t readCountSum = 0;
    nn::util::optional<PlayEvent> startEvent;

    for( size_t i = 0; i < outList.GetLength(); i++ )
    {
        InitializeApplicationPlayStatistics(&outList[i], applicationIdList[i]);
    }
    *outCount = static_cast<int>(outList.GetLength());

    while( NN_STATIC_CONDITION(true) )
    {
        uint32_t readCount = detail::PlayEventBuffer::GetInstance().Read(m_PlayEventReadBuffer, startIndex + readCountSum, m_PlayEventReadBufferCount);
        if( readCount == 0 )
        {
            // 最後まで読み込み、これ以上読み込むものがなくなった。
            NN_RESULT_SUCCESS;
        }
        for( uint32_t i = 0; i < readCount; i++ )
        {
            const auto& currentEvent = m_PlayEventReadBuffer[i];

            if( detail::IsApplicationEvent(currentEvent) && currentEvent.appletEventData.playLogPolicy == ns::PlayLogPolicy::All )
            {
                auto applicationId = nn::pdm::detail::GetApplicationId(currentEvent);
                auto statisticsIndex = Find(outList.GetData(), static_cast<int>(outList.GetLength()), applicationId.value);
                if( !statisticsIndex )
                {
                    continue;
                }

                switch( currentEvent.appletEventData.eventType )
                {
                // Launch と InFocus の場合に開始イベントを設定。
                case AppletEventType::Launch:
                    outList[*statisticsIndex].totalPlayCount++;
                    NN_FALL_THROUGH;
                case AppletEventType::InFocus:
                    currentApplication = applicationId;
                    startEvent = currentEvent;
                    break;
                // 終了系イベントの場合に開始イベントまでの時間をプレイ時間に追加。
                case AppletEventType::OutOfFocus:
                case AppletEventType::Background:
                case AppletEventType::Exit:
                case AppletEventType::AbnormalExit:
                case AppletEventType::Terminate:
                    if( startEvent )
                    {
                        if( currentApplication == applicationId )
                        {
                            UpdateTotalTime(&outList[*statisticsIndex], *startEvent, currentEvent);
                        }
                        else
                        {
                            NN_DETAIL_PDM_WARN("Unexpected event sequence. ApplicationId mismatch : 0x%16x -> 0x%16x\n",
                                detail::GetApplicationId(*startEvent).value, detail::GetApplicationId(currentEvent).value);
                        }
                        startEvent = nullptr;
                    }
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
            else if( detail::IsPowerOnEvent(currentEvent) )
            {
                startEvent = nullptr;
            }
            else if( detail::IsPowerOffEvent(currentEvent) && startEvent )
            {
                // アプリの起点が設定されている状態で電源オフされた場合、電源オフをアプリ終了のイベントとみなす。
                auto applicationId = nn::pdm::detail::GetApplicationId(*startEvent);
                auto statisticsIndex = Find(outList.GetData(), static_cast<int>(outList.GetLength()), applicationId.value);
                if( !statisticsIndex )
                {
                    continue;
                }
                UpdateTotalTime(&outList[*statisticsIndex], *startEvent, currentEvent);
                startEvent = nullptr;
            }
        }
        readCountSum += readCount;
    }
    NN_RESULT_SUCCESS;
}

nn::Result QueryServiceImpl::QueryRecentlyPlayedApplication(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::ncm::ApplicationId>& outList, const nn::account::Uid& user) NN_NOEXCEPT
{
    auto& buffer = *m_pLockableBuffer;
    NN_UTIL_LOCK_GUARD(buffer);

    uint32_t count = 0;
    NN_RESULT_DO(detail::AccountPlayEventProvider::GetInstance().QueryRecentlyPlayedApplication(
        &count, outList.GetData(), static_cast<uint32_t>(outList.GetLength()), buffer.GetPointer(), buffer.GetCapacity(), user));
    *outCount = count;
    NN_RESULT_SUCCESS;
}

nn::Result QueryServiceImpl::GetRecentlyPlayedApplicationUpdateEvent(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT
{
    auto pEvent = detail::AccountPlayEventProvider::GetInstance().GetRecentlyPlayedApplicationUpdateEvent();
    *pOutHandle = sf::NativeHandle(pEvent->GetReadableHandle(), false);
    NN_RESULT_SUCCESS;
}

}}}
