﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pdm/pdm_PrivateTypes.h>
#include <nn/pdm/pdm_Result.h>
#include <nn/pdm/detail/pdm_Log.h>
#include <nn/pdm/detail/pdm_PlayEventFactory.h>
#include <nn/pdm/detail/pdm_PlayLogGenerator.h>
#include <nn/pdm/detail/pdm_AccountPlayEventBuffer.h>
#include <nn/pdm/detail/pdm_Config.h>
#include <nn/pdm/srv/pdm_NotifyServiceImpl.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/time.h>
#include <mutex>

namespace nn { namespace pdm { namespace srv {

nn::Result NotifyServiceImpl::NotifyAppletEvent(nn::pdm::AppletEventType eventType, nn::ncm::ProgramId programId, std::uint32_t version, std::uint8_t appletId, nn::ncm::StorageId storageId, nn::ns::PlayLogPolicy logPolicy) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!detail::IsLibraryApplet(static_cast<applet::AppletId>(appletId)));
    if( logPolicy != nn::ns::PlayLogPolicy::None )
    {
        detail::PlayEventBuffer::GetInstance().Add(detail::MakeAppletEvent(eventType, programId, version, static_cast<nn::applet::AppletId>(appletId), storageId, logPolicy));
        detail::PlayLogGenerator::GetInstance().ProcessAppletEvent(eventType, ncm::ApplicationId{ programId.value }, static_cast<applet::AppletId>(appletId), version, storageId, detail::PlayLogEntryPointTime::MakeWithCurrentTime());
    }
    NN_RESULT_SUCCESS;
}

nn::Result NotifyServiceImpl::NotifyLibraryAppletEvent(nn::pdm::AppletEventType eventType, nn::ncm::ProgramId mainProgramId, std::uint8_t libraryAppletMode, std::uint8_t appletId, nn::ncm::StorageId storageId, nn::ns::PlayLogPolicy logPolicy) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(detail::IsLibraryApplet(static_cast<applet::AppletId>(appletId)));
    if( logPolicy != nn::ns::PlayLogPolicy::None )
    {
        detail::PlayEventBuffer::GetInstance().Add(detail::MakeLibraryAppletEvent(eventType, mainProgramId, static_cast<nn::applet::LibraryAppletMode>(libraryAppletMode), static_cast<nn::applet::AppletId>(appletId), storageId, logPolicy));
        detail::PlayLogGenerator::GetInstance().ProcessAppletEvent(eventType, ncm::ApplicationId{ mainProgramId.value }, static_cast<applet::AppletId>(appletId), 0u /* プレイレポート用情報は固定 */, storageId, detail::PlayLogEntryPointTime::MakeWithCurrentTime());
    }
    NN_RESULT_SUCCESS;
}

nn::Result NotifyServiceImpl::NotifyPowerStateChangeEvent(nn::pdm::PowerStateChangeEventType eventType) NN_NOEXCEPT
{
    detail::PlayEventBuffer::GetInstance().Add(detail::MakePowerStateChangeEvent(eventType));
    detail::PlayLogGenerator::GetInstance().ProcessPowerStateChangeEvent(eventType, detail::PlayLogEntryPointTime::MakeWithCurrentTime());
    NN_RESULT_SUCCESS;
}

nn::Result NotifyServiceImpl::NotifyOperationModeChangeEvent(nn::pdm::OperationMode eventType) NN_NOEXCEPT
{
    detail::PlayEventBuffer::GetInstance().Add(detail::MakeOperationModeChangeEvent(eventType));
    NN_RESULT_SUCCESS;
}

nn::Result NotifyServiceImpl::NotifyClearAllEvent() NN_NOEXCEPT
{
    detail::PlayEventBuffer::GetInstance().Clear();
    detail::AccountPlayEventProvider::GetInstance().Discard();
    NN_RESULT_SUCCESS;
}

nn::Result NotifyServiceImpl::NotifyEventForDebug(const nn::sf::InArray<nn::pdm::PlayEvent>& playEventArray) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    bool isProd = true;
    if( nn::settings::fwdbg::GetSettingsItemValue(&isProd, sizeof(isProd), "pdm", "is_production") != sizeof(isProd) )
    {
        NN_DETAIL_PDM_ERROR("[pdm] NotifyEventForDebug is disabled as the system failed to get pdm.is_production setting.\n");
        NN_RESULT_THROW(pdm::ResultDebugFunction());
    }
    if( isProd )
    {
        NN_DETAIL_PDM_TRACE("[pdm] NotifyEventForDebug is disabled as pdm.is_production setting is true.\n");
        NN_RESULT_THROW(pdm::ResultDebugFunction());
    }
#elif defined(NN_BUILD_CONFIG_OS_WIN)
    // 常に有効
#else
    #error "Unsupported os"
#endif

#define NN_PDM_RESULT_THROW_UNLESS_MINMAX(v, min, max, result) NN_RESULT_THROW_UNLESS((v >= (min)) && (v <= (max)), result)

    size_t count = playEventArray.GetLength();
    int64_t preSteadyTime = 0;
    for( size_t i = 0; i < count; i++ )
    {
        auto playevent = playEventArray[i];
        NN_PDM_RESULT_THROW_UNLESS_MINMAX(playevent.userTime, time::InputPosixTimeMin, time::InputPosixTimeMax, ResultInvalidTime());
        // ネットワーク時計無効時は 0 なので 0 は許容する
        if( playevent.networkTime.value != 0 )
        {
            NN_PDM_RESULT_THROW_UNLESS_MINMAX(playevent.networkTime, time::InputPosixTimeMin, time::InputPosixTimeMax, ResultInvalidTime());
        }
        // 単調増加時計時刻の単調増加を保証する
        NN_RESULT_THROW_UNLESS(playevent.steadyTime >= preSteadyTime, ResultInvalidTime());
        // 安全のため reserved 領域を 0 クリアする
        switch( playevent.eventCategory )
        {
        case PlayEventCategory::Applet:
            {
                NN_PDM_RESULT_THROW_UNLESS_MINMAX(
                    playevent.appletEventData.eventType, AppletEventType::Launch, AppletEventType::AbnormalExit,
                    ResultInvalidAppletEventType());
                std::memset(playevent.appletEventData.reserved, 0, sizeof(playevent.appletEventData.reserved));
            }
            break;
        case PlayEventCategory::UserAccount:
            {
                NN_PDM_RESULT_THROW_UNLESS_MINMAX(
                    playevent.userAccountEventData.eventType, UserAccountEventType::Open, UserAccountEventType::NetworkServiceAccountUnavailable,
                    ResultInvalidUserAccountEventType());
                if( playevent.userAccountEventData.eventType != UserAccountEventType::NetworkServiceAccountAvailable )
                {
                    std::memset(&playevent.userAccountEventData.networkServiceAccountAvailableData, 0, sizeof(playevent.userAccountEventData.networkServiceAccountAvailableData));
                }
            }
            break;
        case PlayEventCategory::PowerStateChange:
            {
                NN_PDM_RESULT_THROW_UNLESS_MINMAX(
                    playevent.powerStateChangeEventData.eventType, PowerStateChangeEventType::On, PowerStateChangeEventType::Terminate,
                    ResultInvalidPowerState());
                std::memset(playevent.powerStateChangeEventData.reserved, 0, sizeof(playevent.powerStateChangeEventData.reserved));
            }
            break;
        case PlayEventCategory::OperationModeChange:
            {
                NN_PDM_RESULT_THROW_UNLESS_MINMAX(
                    playevent.operationModeEventData.operationMode, OperationMode::Handheld, OperationMode::Console,
                    ResultInvalidOperationMode());
                std::memset(playevent.operationModeEventData.reserved, 0, sizeof(playevent.operationModeEventData.reserved));
            }
            break;
        case PlayEventCategory::SteadyClockReset:
            {
                // 固有データなし
                std::memset(playevent.eventData, 0, sizeof(playevent.eventData));
            }
            break;
        default:
            NN_RESULT_THROW(pdm::ResultInvalidEventCategory());
        }
        detail::PlayEventBuffer::GetInstance().Add(playevent);
        preSteadyTime = playevent.steadyTime;
    }
    NN_RESULT_SUCCESS;

#undef NN_PDM_RESULT_THROW_UNLESS_MINMAX
}

nn::Result NotifyServiceImpl::SuspendUserAccountEventService(const account::Uid& user) NN_NOEXCEPT
{
    return detail::AccountPlayEventProvider::GetInstance().Suspend(user);
}

nn::Result NotifyServiceImpl::ResumeUserAccountEventService(const account::Uid& user) NN_NOEXCEPT
{
    return detail::AccountPlayEventProvider::GetInstance().Resume(user);
}

}}}
