﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pdm/detail/pdm_AccountEventWatcherThread.h>
#include <nn/pdm/detail/pdm_Config.h>
#include <nn/pdm/detail/pdm_INotifyService.sfdl.h>
#include <nn/pdm/detail/pdm_IQueryService.sfdl.h>
#include <nn/pdm/detail/pdm_Fs.h>
#include <nn/pdm/detail/pdm_Log.h>
#include <nn/pdm/detail/pdm_PlayLogGenerator.h>
#include <nn/pdm/detail/pdm_SaveDataCommitThread.h>
#include <nn/pdm/detail/pdm_ServiceNames.h>
#include <nn/pdm/detail/pdm_Time.h>
#include <nn/pdm/detail/pdm_Util.h>
#include <nn/pdm/srv/pdm_HipcServer.h>
#include <nn/pdm/srv/pdm_NotifyServiceImpl.h>
#include <nn/pdm/srv/pdm_QueryServiceImpl.h>
#include <nn/nn_Common.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>

namespace nn { namespace pdm { namespace detail {

    void InitializeAccountPlayEventDatabase(LockableMemoryBuffer* const pLockableBuffer) NN_NOEXCEPT;

}}}

namespace nn { namespace pdm { namespace srv {

namespace
{
    nn::sf::UnmanagedServiceObject<nn::pdm::detail::INotifyService, nn::pdm::srv::NotifyServiceImpl> g_NotifyServiceObject;

    const uint32_t ReadTemporaryByteCapacity = 64 * 1024;
    pdm::detail::LockableMemoryBuiltIn<ReadTemporaryByteCapacity> g_ReadTemporaryBuffer;
    nn::sf::UnmanagedServiceObject<nn::pdm::detail::IQueryService, nn::pdm::srv::QueryServiceImpl> g_QueryServiceObject(&g_ReadTemporaryBuffer);

    class ServerManager : public nn::sf::HipcSimpleAllInOneServerManager<30, 2>
    {
    };

    std::aligned_storage<sizeof(ServerManager), NN_ALIGNOF(ServerManager)>::type g_ServerManagerStorage;
    ServerManager* g_pServerManager;
}

void CreateServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pServerManager);

    g_pServerManager = new (&g_ServerManagerStorage) ServerManager;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_pServerManager->RegisterObjectForPort(g_NotifyServiceObject.GetShared(), 15, nn::pdm::detail::ServiceNameForNotify));
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_pServerManager->RegisterObjectForPort(g_QueryServiceObject.GetShared(), 15, nn::pdm::detail::ServiceNameForQuery));
}

void StartServer() NN_NOEXCEPT
{
    g_pServerManager->Start();
}

void LoopAuto() NN_NOEXCEPT
{
    detail::InitializeFs();
    NN_UTIL_SCOPE_EXIT{ detail::FinalizeFs(); };
    detail::InitializeTime();

    detail::PlayLogGenerator::GetInstance().SetEnabledFromFwdbgSettings();
    if( detail::PlayLogGenerator::GetInstance().IsEnabled() )
    {
        // 最後に記録されたイベントを読み込み、電源OFFイベントだった場合には PlayLogGenerator に渡す。
        // PlayLogGenerator は渡された電源OFFイベントと次に受け取る電源ONイベントから電源OFF期間に関するプレイログを作成する。
        PlayEvent playEvent;
        auto& playEventBuffer = detail::PlayEventBuffer::GetInstance();
        playEventBuffer.BeginRead();
        NN_UTIL_SCOPE_EXIT { playEventBuffer.EndRead(); };
        auto readCount = playEventBuffer.Read(&playEvent, playEventBuffer.GetLastIndex(), 1);
        if( (readCount == 1) && detail::IsPowerOffEvent(playEvent) )
        {
            detail::PlayLogGenerator::GetInstance().SetLastPowerOffTime(playEvent);
        }
        else
        {
            NN_DETAIL_PDM_WARN("[PlayEventBuffer] The last event is not power off. PlayLog for power off state won't be generated.\n" );
        }
    }

    // アカウントDB初期化。
    // 必要に応じて デバイスDB → アカウントDB のマイグレーションも行います。
    detail::InitializeAccountPlayEventDatabase(&g_ReadTemporaryBuffer);

    detail::AccountEventWatcherThread::Start();
    NN_UTIL_SCOPE_EXIT{ detail::AccountEventWatcherThread::Stop(); };
    detail::SaveDataCommitThread::Start();
    NN_UTIL_SCOPE_EXIT{ detail::SaveDataCommitThread::Stop(); };

    g_pServerManager->LoopAuto();
}

void RequestStopServer() NN_NOEXCEPT
{
    g_pServerManager->RequestStop();
}

void DestroyServer() NN_NOEXCEPT
{
    g_pServerManager->~ServerManager();
    g_pServerManager = nullptr;
}

}}}
