﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pdm/detail/pdm_Config.h>
#include <nn/pdm/detail/pdm_PlayEventBuffer.h>
#include <nn/pdm/detail/pdm_PlayEventFactory.h>
#include <nn/pdm/detail/pdm_Util.h>
#include <nn/pdm/detail/pdm_Log.h>
#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs.h>
#include <nn/time.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace pdm { namespace detail {

namespace
{
    struct SteadyClockBaseTimePoint
    {
        uint32_t index;
        time::SteadyClockTimePoint steadyClockTimePoint;
    } s_SteadyClockReset;

    detail::InitializationManager g_Initialization = NN_PDM_INITIALIZATION_INITIALIZER;
}

void InitializeTime() NN_NOEXCEPT
{
    g_Initialization.Initialize([]()
    {
        const char* BaseSteadyClockTimePointFileName = "BaseTimePoint.bin";
        char filePath[64];
        auto l = nn::util::SNPrintf(filePath, 64, "%s:/%s", detail::MountName, BaseSteadyClockTimePointFileName);
        NN_SDK_ASSERT_LESS(l, 64);
        NN_UNUSED(l);

        fs::FileHandle fileHandle;
        auto openResult = fs::OpenFile(&fileHandle, filePath, fs::OpenMode_Read);
        if( openResult.IsSuccess() )
        {
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(fileHandle); };
            auto readResult = fs::ReadFile(fileHandle, 0, static_cast<void*>(&s_SteadyClockReset), sizeof(s_SteadyClockReset));
            if( readResult.IsSuccess() )
            {
                // 単調増加クロックのリセットが起きていないかのチェック。
                time::SteadyClockTimePoint timePoint;
                NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardSteadyClock::GetCurrentTimePoint(&timePoint));
                int64_t elapsedSeconds;
                auto getSpanResult = nn::time::GetSpanBetween(&elapsedSeconds, s_SteadyClockReset.steadyClockTimePoint, timePoint);
                if( getSpanResult.IsSuccess() )
                {
                    char source[time::SourceId::StringSize];
                    s_SteadyClockReset.steadyClockTimePoint.sourceId.ToString(source, sizeof(source));
                    NN_DETAIL_PDM_INFO("Succeeded to load base SteadClockTimePoint. Index = %u, SourceId = %s, Value = %llu. Epalsed %llu minutes since the last reset.\n", s_SteadyClockReset.index, source, s_SteadyClockReset.steadyClockTimePoint.value, elapsedSeconds / 60);
                    return;
                }
                else
                {
                    NN_DETAIL_PDM_WARN("Failed to get span between the current time-point and the saved time-point. (%08x).\n", getSpanResult.GetInnerValueForDebug());
                }
            }
            else
            {
                NN_DETAIL_PDM_WARN("Failed to read base SteadClockTimePoint (%08x).\n", readResult.GetInnerValueForDebug());
            }
        }
        else
        {
            NN_DETAIL_PDM_WARN("Failed to open base SteadClockTimePoint (%08x).\n", openResult.GetInnerValueForDebug());
        }
        // (ファイルが存在しない || Read に失敗した || GetSpanBetween に失敗した) 場合は、新しくベースとなる時刻を保存し直す。
        auto deleteResult = fs::DeleteFile(filePath);
        if( deleteResult.IsSuccess() || fs::ResultPathNotFound::Includes(deleteResult) )
        {
            s_SteadyClockReset.index = detail::PlayEventBuffer::GetInstance().GetLastIndex() + 1;
            NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardSteadyClock::GetCurrentTimePoint(&s_SteadyClockReset.steadyClockTimePoint));
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CreateFile(filePath, static_cast<int64_t>(sizeof(s_SteadyClockReset))));
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenFile(&fileHandle, filePath, fs::OpenMode_Write));
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::WriteFile(fileHandle, 0, &s_SteadyClockReset, sizeof(s_SteadyClockReset), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
            fs::CloseFile(fileHandle);
            detail::PlayEventBuffer::GetInstance().Add(detail::MakeSteadyClockResetEvent());
            detail::PlayEventBuffer::GetInstance().Flush();
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(detail::MountName));
            NN_DETAIL_PDM_INFO("Succeeded to create new base SteadClockTimePoint.\n");
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(deleteResult);
        }
    });
}

time::PosixTime GetUserClockTimeForPlayEvent() NN_NOEXCEPT
{
    nn::time::PosixTime posixTime;
    NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardUserSystemClock::GetCurrentTime(&posixTime));
    return posixTime;
}

time::PosixTime GetNetworkClockTimeForPlayEvent() NN_NOEXCEPT
{
    time::PosixTime posixTime;
    NN_RESULT_ABORTING_BLOCK
    {
        NN_RESULT_TRY(time::StandardNetworkSystemClock::GetCurrentTime(&posixTime))
            NN_RESULT_CATCH(time::ResultClockInvalid)
            {
                posixTime.value = 0;
            }
        NN_RESULT_END_TRY
        NN_RESULT_SUCCESS;
    };
    return posixTime;
}

int64_t GetSteadyClockTimeForPlayEvent() NN_NOEXCEPT
{
    time::SteadyClockTimePoint timePoint;
    NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardSteadyClock::GetCurrentTimePoint(&timePoint));

    int64_t elapsedSeconds;
    // SteadyClock が起動中に巻き戻ることはなく、PDM 初期化時に s_BaseSteadyClockTimePoint を設定するのでここでは失敗しない想定。
    NN_ABORT_UNLESS_RESULT_SUCCESS(time::GetSpanBetween(&elapsedSeconds, s_SteadyClockReset.steadyClockTimePoint, timePoint));

    return elapsedSeconds;
}

uint32_t GetElapsedMinutesSinceInputPosixTimeMin(const time::PosixTime& time) NN_NOEXCEPT
{
    if( time <= time::InputPosixTimeMin )
    {
        return 0;
    }
    else
    {
        return static_cast<uint32_t>((time - time::InputPosixTimeMin).GetMinutes());
    }
}

uint32_t GetLastSteadyClockResetEventIndex() NN_NOEXCEPT
{
    return s_SteadyClockReset.index;
}

void SetLastSteadyClockResetPointForDebug(uint32_t index, nn::TimeSpan epalsedTimeSinceReset) NN_NOEXCEPT
{
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    s_SteadyClockReset.index = index;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardSteadyClock::GetCurrentTimePoint(&s_SteadyClockReset.steadyClockTimePoint));
    s_SteadyClockReset.steadyClockTimePoint -= epalsedTimeSinceReset;
#else
    NN_UNUSED(index);
    NN_UNUSED(epalsedTimeSinceReset);
#endif
}

}}}
