﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <mutex>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/pdm/detail/pdm_Fs.h>
#include <nn/pdm/detail/pdm_Log.h>
#include <nn/pdm/detail/pdm_Util.h>
#include <nn/pdm/detail/pdm_PlayEventBuffer.h>
#include <nn/pdm/detail/pdm_AccountPlayEventBuffer.h>
#include <nn/pdm/detail/pdm_SaveDataCommitThread.h>
#include <nn/prepo.h>
#include <nn/prepo/prepo_SystemPlayReport.h>
#include <nn/result/result_HandlingUtility.h>


namespace nn { namespace pdm { namespace detail {

namespace {

    detail::InitializationManager g_Initialization = NN_PDM_INITIALIZATION_INITIALIZER;

    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK Bit8 g_ThreadStack[8 * 1024];
    os::Event g_PlayEventAddedEvent(os::EventClearMode_ManualClear);
    os::Event g_StopThreadEvent(os::EventClearMode_ManualClear);

    Result ReportUsage() NN_NOEXCEPT
    {
        PlayEventBuffer::ReportData reportData;
        PlayEventBuffer::GetInstance().GetReportData(&reportData);

        Bit8 buffer[128];
        prepo::SystemPlayReport deviceReport("device_event");
        deviceReport.SetBuffer(buffer, sizeof(buffer));
        NN_RESULT_DO(deviceReport.SetApplicationId({ 0x0100000000001014llu }));
        NN_RESULT_DO(deviceReport.Add("count", static_cast<int64_t>(reportData.count)));
        NN_RESULT_DO(deviceReport.Add("start_index", static_cast<int64_t>(reportData.startIndex)));
        NN_RESULT_DO(deviceReport.Add("last_index", static_cast<int64_t>(reportData.lastIndex)));
        NN_RESULT_DO(deviceReport.Save());

        for( AccountPlayEventProvider::BufferIndex i = 0; i < AccountPlayEventProvider::BufferIndexCountMax; i++ )
        {
            AccountPlayEventBuffer::ReportData accountReportData;
            if( AccountPlayEventProvider::GetInstance().GetBuffer(i).GetReportData(&accountReportData).IsFailure() )
            {
                continue;
            }

            prepo::SystemPlayReport accountReport("account_event");
            accountReport.SetBuffer(buffer, sizeof(buffer));
            NN_RESULT_DO(accountReport.SetApplicationId({ 0x0100000000001014llu }));
            NN_RESULT_DO(accountReport.Add("count", static_cast<int64_t>(accountReportData.count)));
            NN_RESULT_DO(accountReport.Add("start_index", static_cast<int64_t>(accountReportData.startIndex)));
            NN_RESULT_DO(accountReport.Add("last_index", static_cast<int64_t>(accountReportData.lastIndex)));
            NN_RESULT_DO(accountReport.Save(accountReportData.uid));
        }
        NN_RESULT_SUCCESS;
    }

    void CommitDevicePlayEvent() NN_NOEXCEPT
    {
        PlayEventBuffer::GetInstance().Flush();
#if !defined(PDM_USE_HOST_FS)
        detail::CommitSaveData();
#endif
        NN_DETAIL_PDM_TRACE("[SaveDataCommitThread] Committed SaveData.\n");
    }

    void ThreadFunc(void* arg) NN_NOEXCEPT
    {
        NN_UNUSED(arg);

        fs::SetPriorityRawOnCurrentThread(fs::PriorityRaw_Background);

        PlayEventBuffer::GetInstance().SetPlayEventAddedEvent(&g_PlayEventAddedEvent);

        os::TimerEvent reportUsageTimer(os::EventClearMode_ManualClear);
        reportUsageTimer.StartOneShot(nn::TimeSpan::FromMinutes(1));

        while( !g_StopThreadEvent.TryWait() )
        {
            auto eventId = os::WaitAny(
                g_PlayEventAddedEvent.GetBase(),
                reportUsageTimer.GetBase(),
                g_StopThreadEvent.GetBase());

            switch( eventId )
            {
            case 0: // PlayEventAdded
                {
                    g_PlayEventAddedEvent.Clear();
                    // イベント追加時はバッファに半分以上溜まっている場合に書き込み・コミット
                    AccountPlayEventProvider::GetInstance().Flush(50u);
                    if( PlayEventBuffer::GetInstance().GetFilledBufferSpaceCount() <= PlayEventBuffer::GetBufferSpaceCountMax() / 2 )
                    {
                        continue;
                    }
                    CommitDevicePlayEvent();
                }
                break;
            case 1: // reportUsage
                {
                    reportUsageTimer.Clear();
                    auto result = ReportUsage();
                    if( result.IsFailure() )
                    {
                        NN_DETAIL_PDM_WARN("[pdm] Failed to report usage : %03d-%04d (0x%08x)\n",
                            result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
                    }
                    reportUsageTimer.StartOneShot(nn::TimeSpan::FromHours(24));
                    continue;
                }
                break;
            case 2: // Stop
                {
                    // ストップ時は無条件に書き込み・コミット
                    AccountPlayEventProvider::GetInstance().Flush();
                    CommitDevicePlayEvent();
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    }
}

void SaveDataCommitThread::Start() NN_NOEXCEPT
{
    g_Initialization.Initialize([]
    {
        NN_DETAIL_PDM_TRACE("[SaveDataCommitThread] Start()\n");
        nn::os::CreateThread(&g_Thread, ThreadFunc, nullptr,
            g_ThreadStack, sizeof(g_ThreadStack), NN_SYSTEM_THREAD_PRIORITY(pdmSrv, CommitSaveData));
        nn::os::SetThreadName(&g_Thread, NN_SYSTEM_THREAD_NAME(pdmSrv, CommitSaveData));
        nn::os::StartThread(&g_Thread);
    });
}

void SaveDataCommitThread::Stop() NN_NOEXCEPT
{
    g_Initialization.Finalize([]
    {
        NN_DETAIL_PDM_TRACE("[SaveDataCommitThread] Stop()\n");
        g_StopThreadEvent.Signal();
        nn::os::WaitThread(&g_Thread);
        nn::os::DestroyThread(&g_Thread);
    });
}

}}}
