﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pdm/detail/pdm_Log.h>
#include <nn/pdm/detail/pdm_PlayLogGenerator.h>
#include <nn/pdm/detail/pdm_Time.h>
#include <nn/pdm/detail/pdm_Util.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/prepo.h>
#include <nn/prepo/prepo_SystemPlayReport.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/util/util_LockGuard.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace pdm { namespace detail {

namespace {

    nn::ApplicationId PdmPlayReportApplicationId = { 0x0100000000001014llu };

    // 作成するプレイレポートの種類。
    const char* EventIdAppletLife = "applet_life";
    const char* EventIdAppletForeground = "applet_fg";
    const char* EventIdAccountOpen = "account_open";
    const char* EventIdAppletForegroundWithAccount = "applet_fg_account";
    const char* EventIdPowerState = "power_state";

    // プレイレポートに使用するキーの種類。
    const char* KeyStartUserClock = "lc_started_at";
    const char* KeyStartNetworkClock = "nc_started_at";
    const char* KeyDuration = "duration";
    const char* KeyAppletApplicationId = "application_id";
    const char* KeyAppletAppletId = "applet_id";
    const char* KeyAppletVersion = "version";
    const char* KeyAppletStorageId = "storage_id";
    const char* KeyStartPowerState = "power_state_start";
    const char* KeyEndPowerState = "power_state_end";

    const int DateTimeStringSize = sizeof("YYYY-MM-DD hh:mm:ss");

    nn::time::TimeZoneRule& GetTimeZoneRule() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(nn::time::TimeZoneRule, s_TimeZoneRule);
        NN_FUNCTION_LOCAL_STATIC(nn::Result, s_Result, = nn::time::LoadTimeZoneRule(&s_TimeZoneRule, { "UTC" }));
        NN_ABORT_UNLESS_RESULT_SUCCESS(s_Result);
        return s_TimeZoneRule;
    }

    bool GetDateTimeString(char* buffer, size_t bufferSize, const nn::time::PosixTime& posixTime) NN_NOEXCEPT
    {
        nn::time::CalendarTime calendar;
        auto result = nn::time::ToCalendarTime(&calendar, nullptr, posixTime, GetTimeZoneRule());
        if( result.IsSuccess() )
        {
            int l = nn::util::SNPrintf(buffer, bufferSize, "%04d-%02d-%02d %02d:%02d:%02d",
                calendar.year, calendar.month, calendar.day,
                calendar.hour, calendar.minute, calendar.second);
            NN_SDK_ASSERT(static_cast<size_t>(l) < bufferSize);
            NN_UNUSED(l);
            buffer[bufferSize - 1] = '\0';
            return true;
        }
        else
        {
            NN_DETAIL_PDM_TRACE("[PlayLogGenerator] Failed to GetDateTimeString : %03d-%04d (%08x).\n", result.GetModule(), result.GetModule(), result.GetInnerValueForDebug());
            return false;
        }
    }

// ncm_StorageUtil.h を利用することが可能
    const char* GetStorageIdStr(const ncm::StorageId& storageId) NN_NOEXCEPT
    {
        switch( storageId )
        {
        case ncm::StorageId::BuildInSystem:
            return "BuildInSystem";
        case ncm::StorageId::BuildInUser:
            return "BuildInUser";
        case ncm::StorageId::Card:
            return "Card";
        case ncm::StorageId::Host:
            return "Host";
        case ncm::StorageId::None:
            return "None";
        case ncm::StorageId::SdCard:
            return "SdCard";
        default:
            return "(unknown)";
        }
    }

    const char* GetPowerStateStr(const PowerStateChangeEventType& eventType) NN_NOEXCEPT
    {
        switch( eventType )
        {
        case PowerStateChangeEventType::On:
            return "On";
        case PowerStateChangeEventType::Off:
            return "Off";
        case PowerStateChangeEventType::Sleep:
            return "Sleep";
        case PowerStateChangeEventType::Awake:
            return "Awake";
        case PowerStateChangeEventType::BackgroundServicesAwake:
            return "BackgroundServicesAwake";
        case PowerStateChangeEventType::Terminate:
            return "Terminate";
        default:
            return "(unknown)";
        }
    }

    void AddAppletInfo(prepo::SystemPlayReport* report, const AppletData& data) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(report->Add(KeyAppletAppletId, static_cast<int64_t>(data.appletId)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(report->Add(KeyAppletApplicationId, prepo::Any64BitId{ data.applicationId.value }));
        NN_ABORT_UNLESS_RESULT_SUCCESS(report->Add(KeyAppletVersion, static_cast<int64_t>(data.version)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(report->Add(KeyAppletStorageId, GetStorageIdStr(data.storageId)));
    }

    void AddClockTimeInfo(prepo::SystemPlayReport* report, const char* key, const nn::time::PosixTime& time) NN_NOEXCEPT
    {
        char timeString[DateTimeStringSize];
        if( GetDateTimeString(timeString, DateTimeStringSize, time) )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(report->Add(key, timeString));
        }
        else
        {
            NN_DETAIL_PDM_TRACE("[PlayLogGenerator] Failed to add %s.\n", key);
        }
    }

    bool AddCommonInfo(prepo::SystemPlayReport* report, const PlayLogEntryPointTime& startPoint, const PlayLogEntryPointTime& endPoint) NN_NOEXCEPT
    {
        if( startPoint.steadyClockTimePointValue > endPoint.steadyClockTimePointValue )
        {
            NN_DETAIL_PDM_WARN("[PlayLogGenerator] startPoint's SteadyClockTimePointValue(%lld) is larger than endPoint's SteadyClockTimePointValue(%lld). This is unexpected.\n",
                startPoint.steadyClockTimePointValue, endPoint.steadyClockTimePointValue);
            return false;
        }

        auto seconds = endPoint.steadyClockTimePointValue - startPoint.steadyClockTimePointValue;
        NN_ABORT_UNLESS_RESULT_SUCCESS(report->Add(KeyDuration, seconds));

        AddClockTimeInfo(report, KeyStartUserClock, startPoint.userClockTime);
        if( startPoint.networkClockTime )
        {
            AddClockTimeInfo(report, KeyStartNetworkClock, *startPoint.networkClockTime);
        }
        else
        {
            // ネットワーク時計が無効な場合、キーバリュー自体を含めない。
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(report->SetApplicationId(PdmPlayReportApplicationId));
        return true;
    }
}

PlayLogEntryPointTime PlayLogEntryPointTime::MakeWithCurrentTime() NN_NOEXCEPT
{
    PlayLogEntryPointTime time;
    NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardUserSystemClock::GetCurrentTime(&time.userClockTime));
    time::PosixTime networkClockTime;
    if( time::StandardNetworkSystemClock::GetCurrentTime(&networkClockTime).IsSuccess() )
    {
        time.networkClockTime = networkClockTime;
    }
    else
    {
        time.networkClockTime = nullptr;
    }
    time.steadyClockTimePointValue = detail::GetSteadyClockTimeForPlayEvent();
    return time;
}

void AppletData::Set(const ncm::ApplicationId& _applicationId, applet::AppletId _appletId, uint32_t _version, const ncm::StorageId& _storageId) NN_NOEXCEPT
{
    applicationId = _applicationId;
    appletId = _appletId;
    version = _version;
    storageId = _storageId;
}

// -- PlayLogGenerator --

PlayLogGenerator::PlayLogGenerator() NN_NOEXCEPT
    : m_IsEnabled(false)
{
    Clear();
}

void PlayLogGenerator::ProcessAppletEvent(AppletEventType eventType, const ncm::ApplicationId& applicationId, applet::AppletId appletId, uint32_t version, const ncm::StorageId& storageId, const PlayLogEntryPointTime& time) NN_NOEXCEPT
{
    switch( eventType )
    {
    case AppletEventType::Launch:
        {
            ProcessAppletLaunchEvent(applicationId, appletId, version, storageId, time);
        }
        break;
    case AppletEventType::InFocus:
        {
            ProcessAppletInFocusEvent(applicationId, appletId, version, storageId, time);
        }
        break;
    case AppletEventType::OutOfFocus:
    case AppletEventType::Background:
        {
            ProcessAppletLostFocusEvent(applicationId, appletId, version, storageId, time);
        }
        break;
    case AppletEventType::Exit:
    case AppletEventType::AbnormalExit:
    case AppletEventType::Terminate:
        {
            ProcessAppletExitEvent(applicationId, appletId, version, storageId, time);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void PlayLogGenerator::Clear() NN_NOEXCEPT
{
    for( int i = 0; i < AliveAppletCountMax; i++ )
    {
        m_AppletLaunchPoint[i].Invalidate();
    }
    for( int i = 0; i < FocusedAppletCountMax; i++ )
    {
        m_AppletInFocusPoint[i].Invalidate();
    }
    for( int i = 0; i < account::UserCountMax; i++ )
    {
        m_AccountOpenPoint[i].Invalidate();
    }
    m_PowerStateChangePoint.Invalidate();
    m_LastApplicationLaunchEvent.Invalidate();
}

void PlayLogGenerator::SetEnabledFromFwdbgSettings() NN_NOEXCEPT
{
    bool enabled;
    if(nn::settings::fwdbg::GetSettingsItemValue(&enabled, sizeof(enabled), "pdm", "save_playlog") == sizeof(enabled))
    {
        m_IsEnabled = enabled;
        NN_DETAIL_PDM_TRACE("[PlayLogGenerator] PlayLog is %s.\n", enabled ? "enabled" : "disabled");
    }
    else
    {
        m_IsEnabled = false;
        NN_DETAIL_PDM_TRACE("[PlayLogGenerator] Failed to load fwdbg settings. PlayLog is disabled.\n");
    }
}

bool PlayLogGenerator::IsEnabled() const NN_NOEXCEPT
{
    return m_IsEnabled;
}

void PlayLogGenerator::ProcessAppletLaunchEvent(const ncm::ApplicationId applicationId, applet::AppletId appletId, uint32_t version, const ncm::StorageId storageId, const PlayLogEntryPointTime& time) NN_NOEXCEPT
{
    if( !m_IsEnabled )
    {
        return;
    }
    NN_UTIL_LOCK_GUARD(m_ProcessEventMutex);

    if( appletId == applet::AppletId_Application )
    {
        m_LastApplicationLaunchEvent.data.Set(applicationId, appletId, version, storageId);
        m_LastApplicationLaunchEvent.time = time;
    }
    for( int i = 0; i < AliveAppletCountMax; i++ )
    {
        if( !m_AppletLaunchPoint[i].IsValid() )
        {
            m_AppletLaunchPoint[i].data.Set(applicationId, appletId, version, storageId);
            m_AppletLaunchPoint[i].time = time;
            return;
        }
    }
    NN_SDK_ASSERT(false, "[PlayLogGenerator] There is no space left to store applet launch event.\n");
}

void PlayLogGenerator::ProcessAppletInFocusEvent(const ncm::ApplicationId applicationId, applet::AppletId appletId, uint32_t version, const ncm::StorageId storageId, const PlayLogEntryPointTime& time) NN_NOEXCEPT
{
    if( !m_IsEnabled )
    {
        return;
    }
    NN_UTIL_LOCK_GUARD(m_ProcessEventMutex);

    for( auto& inFocusPoint : m_AppletInFocusPoint )
    {
        if( !inFocusPoint.IsValid() )
        {
            inFocusPoint.data.Set(applicationId, appletId, version, storageId);
            inFocusPoint.time = time;
            return;
        }
    }
    NN_SDK_ASSERT(false, "[PlayLogGenerator] There is no space left to store applet InFocus event.\n");
}

void PlayLogGenerator::ProcessAppletLostFocusEvent(const ncm::ApplicationId applicationId, applet::AppletId appletId, uint32_t version, const ncm::StorageId storageId, const PlayLogEntryPointTime& time) NN_NOEXCEPT
{
    NN_UNUSED(appletId);
    NN_UNUSED(version);
    NN_UNUSED(storageId);

    if( !m_IsEnabled )
    {
        return;
    }
    NN_UTIL_LOCK_GUARD(m_ProcessEventMutex);

    for( auto& inFocusPoint : m_AppletInFocusPoint )
    {
        if( inFocusPoint.IsValid() && applicationId == inFocusPoint.data.applicationId )
        {
            SaveAppletForegroundPlayLog(inFocusPoint, time);
            for( int i = 0; i < account::UserCountMax; i++ )
            {
                if( m_AccountOpenPoint[i].IsValid() )
                {
                    SaveAppletForegroundWithAccountPlayLog(m_AccountOpenPoint[i], inFocusPoint, time);
                }
            }
            inFocusPoint.Invalidate();
        }
    }
}

void PlayLogGenerator::ProcessAppletExitEvent(const ncm::ApplicationId applicationId, applet::AppletId appletId, uint32_t version, const ncm::StorageId storageId, const PlayLogEntryPointTime& time) NN_NOEXCEPT
{
    NN_UNUSED(appletId);
    NN_UNUSED(version);
    NN_UNUSED(storageId);

    if( !m_IsEnabled )
    {
        return;
    }
    NN_UTIL_LOCK_GUARD(m_ProcessEventMutex);

    for( int i = 0; i < AliveAppletCountMax; i++ )
    {
        if( m_AppletLaunchPoint[i].IsValid() && m_AppletLaunchPoint[i].data.applicationId == applicationId )
        {
            SaveAppletLifePlayLog(m_AppletLaunchPoint[i], time);
            m_AppletLaunchPoint[i].Invalidate();
            return;
        }
    }
    NN_DETAIL_PDM_WARN("[PlayLogGenerator] No matching applet launch point is stored.\n");
}

void PlayLogGenerator::ProcessAccountOpenEvent(const account::Uid & uid, const PlayLogEntryPointTime & time) NN_NOEXCEPT
{
    if( !m_IsEnabled )
    {
        return;
    }
    NN_UTIL_LOCK_GUARD(m_ProcessEventMutex);

    for( int i = 0; i < account::UserCountMax; i++ )
    {
        if( !m_AccountOpenPoint[i].IsValid() )
        {
            m_AccountOpenPoint[i].data = uid;
            m_AccountOpenPoint[i].time = time;
            return;
        }
    }
    NN_SDK_ASSERT(false, "[PlayLogGenerator] There is no space left to store account open point.\n");
}

void PlayLogGenerator::ProcessAccountCloseEvent(const account::Uid& uid, const PlayLogEntryPointTime& time) NN_NOEXCEPT
{
    if( !m_IsEnabled )
    {
        return;
    }
    NN_UTIL_LOCK_GUARD(m_ProcessEventMutex);

    util::optional<int> index;
    for( int i = 0; i < account::UserCountMax; i++ )
    {
        if( m_AccountOpenPoint[i].IsValid() && m_AccountOpenPoint[i].data == uid )
        {
            index = i;
            break;
        }
    }
    if( !index )
    {
        NN_DETAIL_PDM_WARN("[PlayLogGenerator] No matching account open point is stored. Closed account is %016llx-%016llx\n",
                uid._data[0], uid._data[1]);
        return;
    }
    NN_UTIL_SCOPE_EXIT{ m_AccountOpenPoint[*index].Invalidate(); };

    if( !m_LastApplicationLaunchEvent.IsValid() )
    {
        NN_DETAIL_PDM_WARN("[PlayLogGenerator] No application launch event is stored to supply applet info for account report.\n");
        return;
    }

    SaveAccountOpenPlayLog(m_AccountOpenPoint[*index], m_LastApplicationLaunchEvent.data, time);

    for( auto& inFocusPoint : m_AppletInFocusPoint )
    {
        if( inFocusPoint.IsValid() )
        {
            SaveAppletForegroundWithAccountPlayLog(m_AccountOpenPoint[*index], inFocusPoint, time);
        }
    }
}

void PlayLogGenerator::ProcessPowerStateChangeEvent(PowerStateChangeEventType eventType, const PlayLogEntryPointTime& time) NN_NOEXCEPT
{
    if( !m_IsEnabled )
    {
        return;
    }
    NN_UTIL_LOCK_GUARD(m_ProcessEventMutex);

    if( m_PowerStateChangePoint.IsValid() )
    {
        SavePowerStatePlayLog(m_PowerStateChangePoint, eventType, time);
    }
    m_PowerStateChangePoint.data = eventType;
    m_PowerStateChangePoint.time = time;

    if( eventType == PowerStateChangeEventType::Off )
    {
        // 電源OFF のイベントが来たらその後のイベントの受け付けを止め、
        // 閉じられていない始点の登録のあるプレイログを生成する。
        m_IsEnabled = false;
        // アプレットの生存期間
        for( int i = 0; i < AliveAppletCountMax; i++ )
        {
            if( m_AppletLaunchPoint[i].IsValid() )
            {
                SaveAppletLifePlayLog(m_AppletLaunchPoint[i], time);
            }
        }
        // アプレットのFG期間
        for( auto& inFocusPoint : m_AppletInFocusPoint )
        {
            if( inFocusPoint.IsValid() )
            {
                SaveAppletForegroundPlayLog(inFocusPoint, time);
            }
        }

        // アカウントのオープン期間
        if( m_LastApplicationLaunchEvent.IsValid() )
        {
            for( int i = 0; i < account::UserCountMax; i++ )
            {
                if( m_AccountOpenPoint[i].IsValid() )
                {
                    SaveAccountOpenPlayLog(m_AccountOpenPoint[i], m_LastApplicationLaunchEvent.data, time);
                }
            }
        }
        // アプレットのFG期間（アカウント）
        for( auto& inFocusPoint : m_AppletInFocusPoint )
        {
            if( inFocusPoint.IsValid() )
            {
                for( int i = 0; i < account::UserCountMax; i++ )
                {
                    if( m_AccountOpenPoint[i].IsValid() )
                    {
                        SaveAppletForegroundWithAccountPlayLog(m_AccountOpenPoint[i], inFocusPoint, time);
                    }
                }
            }
        }
        Clear();
    }
}

void PlayLogGenerator::SetLastPowerOffTime(const PlayEvent& powerOffEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(powerOffEvent.eventCategory == PlayEventCategory::PowerStateChange);
    NN_SDK_REQUIRES(powerOffEvent.powerStateChangeEventData.eventType == PowerStateChangeEventType::Off);

    m_PowerStateChangePoint.data = PowerStateChangeEventType::Off;
    m_PowerStateChangePoint.time.userClockTime = powerOffEvent.userTime;
    if( powerOffEvent.networkTime.value != 0 )
    {
        m_PowerStateChangePoint.time.networkClockTime = powerOffEvent.networkTime;
    }
    else
    {
        m_PowerStateChangePoint.time.networkClockTime = nullptr;
    }
    m_PowerStateChangePoint.time.steadyClockTimePointValue = powerOffEvent.steadyTime;
}

void PlayLogGenerator::SavePlayLog(const char* eventId, const PlayLogEntryPointTime& startPointTime, const PlayLogEntryPointTime& endPointTime, std::function<void(prepo::SystemPlayReport* report)> addKeyValueFunc) const NN_NOEXCEPT
{
    prepo::SystemPlayReport report(eventId);
    report.SetBuffer(m_PrepoBuffer, PrepoBufferSize);
    if( !AddCommonInfo(&report, startPointTime, endPointTime) )
    {
        NN_DETAIL_PDM_TRACE("[PlayLogGenerator] Failed to add common info for PlayLog. Skip saving PlayReport.\n");
        return;
    }

    addKeyValueFunc(&report);

    auto saveResult = report.Save();
    if( saveResult.IsFailure() )
    {
        NN_DETAIL_PDM_TRACE("[PlayLogGenerator] Failed to save %s : %03d-%04d (%08x).\n", eventId, saveResult.GetModule(), saveResult.GetDescription(), saveResult.GetInnerValueForDebug());
    }
}

void PlayLogGenerator::SavePlayLog(const char* eventId, const PlayLogEntryPointTime& startPointTime, const PlayLogEntryPointTime& endPointTime, std::function<void(prepo::SystemPlayReport* report)> addKeyValueFunc, const account::Uid& uid) const NN_NOEXCEPT
{
    prepo::SystemPlayReport report(eventId);
    report.SetBuffer(m_PrepoBuffer, PrepoBufferSize);
    if( !AddCommonInfo(&report, startPointTime, endPointTime) )
    {
        NN_DETAIL_PDM_TRACE("[PlayLogGenerator] Failed to add common info for PlayLog. Skip saving PlayReport.\n");
        return;
    }

    addKeyValueFunc(&report);

    auto saveResult = report.Save(uid);
    if( saveResult.IsFailure() )
    {
        NN_DETAIL_PDM_TRACE("[PlayLogGenerator] Failed to save %s : %03d-%04d (%08x).\n", eventId, saveResult.GetModule(), saveResult.GetDescription(), saveResult.GetInnerValueForDebug());
    }
}

void PlayLogGenerator::SaveAppletLifePlayLog(const PlayLogEntryStartPoint<AppletData>& appletLaunchPoint, const PlayLogEntryPointTime& endPointTime) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(appletLaunchPoint.IsValid());
    SavePlayLog(EventIdAppletLife, appletLaunchPoint.time, endPointTime,
        [&](prepo::SystemPlayReport* report)
        {
            AddAppletInfo(report, appletLaunchPoint.data);
        }
    );
}

void PlayLogGenerator::SaveAppletForegroundPlayLog(const PlayLogEntryStartPoint<AppletData>& appletResumePoint, const PlayLogEntryPointTime& endPointTime) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(appletResumePoint.IsValid());
    SavePlayLog(EventIdAppletForeground, appletResumePoint.time, endPointTime,
        [&](prepo::SystemPlayReport* report)
        {
            AddAppletInfo(report, appletResumePoint.data);
        }
    );
}

void PlayLogGenerator::SaveAccountOpenPlayLog(const PlayLogEntryStartPoint<account::Uid>& accountOpenPoint, const AppletData& appletData, const PlayLogEntryPointTime& endPointTime) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(accountOpenPoint.IsValid());
    SavePlayLog(EventIdAccountOpen, accountOpenPoint.time, endPointTime,
        [&](prepo::SystemPlayReport* report)
        {
            AddAppletInfo(report, appletData);
        },
        accountOpenPoint.data
    );
}

void PlayLogGenerator::SaveAppletForegroundWithAccountPlayLog(const PlayLogEntryStartPoint<account::Uid>& accountOpenPoint, const PlayLogEntryStartPoint<AppletData>& appletResumePoint, const PlayLogEntryPointTime& endPointTime) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(accountOpenPoint.IsValid());
    NN_SDK_REQUIRES(appletResumePoint.IsValid());
    const auto& startPointTime = (appletResumePoint.time.steadyClockTimePointValue > accountOpenPoint.time.steadyClockTimePointValue) ? appletResumePoint.time : accountOpenPoint.time;
    SavePlayLog(EventIdAppletForegroundWithAccount, startPointTime, endPointTime,
        [&](prepo::SystemPlayReport* report)
        {
            AddAppletInfo(report, appletResumePoint.data);
        },
        accountOpenPoint.data
    );
}

void PlayLogGenerator::SavePowerStatePlayLog(const PlayLogEntryStartPoint<PowerStateChangeEventType>& lastPowerStateChangePoint, PowerStateChangeEventType nextPowerState, const PlayLogEntryPointTime& endPointTime) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(lastPowerStateChangePoint.IsValid());
    SavePlayLog(EventIdPowerState, lastPowerStateChangePoint.time, endPointTime,
        [&](prepo::SystemPlayReport* report)
        {
            report->Add(KeyStartPowerState, GetPowerStateStr(lastPowerStateChangePoint.data));
            report->Add(KeyEndPowerState, GetPowerStateStr(nextPowerState));
        }
    );
}

}}}
