﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/account.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/pdm/detail/pdm_Log.h>
#include <nn/pdm/detail/pdm_Util.h>
#include <nn/pdm/detail/pdm_PlayEventBuffer.h>
#include <nn/pdm/detail/pdm_PlayLogGenerator.h>
#include <nn/pdm/detail/pdm_PlayEventFactory.h>
#include <nn/pdm/detail/pdm_AccountPlayEventBuffer.h>
#include <nn/pdm/detail/pdm_AccountEventWatcherThread.h>
#include <nn/account/account_ApiForSystemServices.h>

namespace nn { namespace pdm { namespace detail {

namespace {

    detail::InitializationManager g_Initialization = NN_PDM_INITIALIZATION_INITIALIZER;

    os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK Bit8 g_ThreadStack[8 * 1024];

    os::Event g_StopEvent(os::EventClearMode_ManualClear);

    class OpenUserTable
    {
    public:
        OpenUserTable() NN_NOEXCEPT
            : m_OpenUserCount(0) {};
        void Update() NN_NOEXCEPT;
    private:
        static bool Contains(const account::Uid uids[], int count, const account::Uid& uid) NN_NOEXCEPT;
        account::Uid m_OpenUsers[account::UserCountMax];
        int m_OpenUserCount;
    };

    bool OpenUserTable::Contains(const account::Uid uids[], int count, const account::Uid& uid) NN_NOEXCEPT
    {
        for( int i = 0; i < count; i++ )
        {
            if( uids[i] == uid )
            {
                return true;
            }
        }
        return false;
    }

    void OpenUserTable::Update() NN_NOEXCEPT
    {
        NN_DETAIL_PDM_TRACE("[AccountEventWatcher] OpenUserTable::Update()\n");
        int count;
        account::Uid uids[account::UserCountMax];

        NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListOpenUsers(&count, uids, account::UserCountMax));

        for( int i = 0; i < count; i++ )
        {
            if( !Contains(m_OpenUsers, m_OpenUserCount, uids[i]) )
            {
                NN_DETAIL_PDM_TRACE("Account %016llx-%016llx opened.\n", uids[i]._data[0], uids[i]._data[1]);
                PlayEventBuffer::GetInstance().Add(detail::MakeUserAccountEvent(UserAccountEventType::Open, uids[i]));
                PlayLogGenerator::GetInstance().ProcessAccountOpenEvent(uids[i], PlayLogEntryPointTime::MakeWithCurrentTime());
            }
        }
        for( int i = 0; i < m_OpenUserCount; i++ )
        {
            if( !Contains(uids, count, m_OpenUsers[i]) )
            {
                NN_DETAIL_PDM_TRACE("Account %016llx-%016llx closed.\n", m_OpenUsers[i]._data[0], m_OpenUsers[i]._data[1]);
                PlayEventBuffer::GetInstance().Add(detail::MakeUserAccountEvent(UserAccountEventType::Close, m_OpenUsers[i]));
                PlayLogGenerator::GetInstance().ProcessAccountCloseEvent(m_OpenUsers[i], PlayLogEntryPointTime::MakeWithCurrentTime());
            }
        }

        if( count > 0 )
        {
            std::memcpy(m_OpenUsers, uids, sizeof(account::Uid) * count);
        }

        m_OpenUserCount = count;
    }

    class NetworkServiceAccountAvailabilityTable
    {
    public:
        void Initialize() NN_NOEXCEPT;
        void Update() NN_NOEXCEPT;
    private:
        bool IsNetworkServiceAccountAvailable(const account::Uid& uid) NN_NOEXCEPT;
        int             m_AccountCount;
        account::Uid    m_Uids[account::UserCountMax];
        bool            m_IsNetworkServiceAccountAvailable[account::UserCountMax];
    };

    void NetworkServiceAccountAvailabilityTable::Initialize() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&m_AccountCount, m_Uids, account::UserCountMax));
        for( int i = 0; i < m_AccountCount; i++ )
        {
            account::NetworkServiceAccountManager manager;
            NN_ABORT_UNLESS_RESULT_SUCCESS(account::GetNetworkServiceAccountManager(&manager, m_Uids[i]));
            NN_ABORT_UNLESS_RESULT_SUCCESS(manager.IsNetworkServiceAccountAvailable(&m_IsNetworkServiceAccountAvailable[i]));
        }
    }

    void NetworkServiceAccountAvailabilityTable::Update() NN_NOEXCEPT
    {
        NN_DETAIL_PDM_TRACE("[AccountEventWatcher] NetworkServiceAccountAvailabilityTable::Update()\n");
        int count;
        account::Uid uids[account::UserCountMax];
        bool isAvailable[account::UserCountMax] = { false };
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&count, uids, account::UserCountMax));

        for( int i = 0; i < count; i++ )
        {
            account::NetworkServiceAccountManager manager;
            auto getManagerResult = account::GetNetworkServiceAccountManager(&manager, uids[i]);
            if( getManagerResult.IsFailure() )
            {
                NN_DETAIL_PDM_TRACE("Failed to get NetworkServiceAccountManager for %016llx-%016llx.\n", uids[i]._data[0], uids[i]._data[1]);
                isAvailable[i] = IsNetworkServiceAccountAvailable(uids[i]);
                continue;
            }

            account::NetworkServiceAccountId networkServiceAccountID;
            isAvailable[i] = manager.GetNetworkServiceAccountId(&networkServiceAccountID).IsSuccess();

            if( isAvailable[i] && !IsNetworkServiceAccountAvailable(uids[i]) )
            {
                NN_DETAIL_PDM_TRACE("Account %016llx-%016llx's NetworkServiceAccount(%016llx) became available.\n", uids[i]._data[0], uids[i]._data[1], networkServiceAccountID.id);
                PlayEventBuffer::GetInstance().Add(detail::MakeNetworkServiceAccountAvailableEvent(uids[i], networkServiceAccountID));
            }
            else if( !isAvailable[i] && IsNetworkServiceAccountAvailable(uids[i]) )
            {
                NN_DETAIL_PDM_TRACE("Account %016llx-%016llx's NetworkServiceAccount became unavailable.\n", uids[i]._data[0], uids[i]._data[1]);
                PlayEventBuffer::GetInstance().Add(detail::MakeUserAccountEvent(UserAccountEventType::NetworkServiceAccountUnavailable, uids[i]));
            }
        }

        if( count > 0 )
        {
            std::memcpy(m_Uids, uids, sizeof(account::Uid) * count);
            std::memcpy(m_IsNetworkServiceAccountAvailable, isAvailable, sizeof(bool) * count);
        }

        m_AccountCount = count;
    }

    bool NetworkServiceAccountAvailabilityTable::IsNetworkServiceAccountAvailable(const account::Uid& uid) NN_NOEXCEPT
    {
        for( int i = 0; i < m_AccountCount; i++ )
        {
            if( m_Uids[i] == uid )
            {
                return m_IsNetworkServiceAccountAvailable[i];
            }
        }
        return false;
    }

    class UserRegistryWatcher
    {
        NN_DISALLOW_COPY(UserRegistryWatcher);
        NN_DISALLOW_MOVE(UserRegistryWatcher);

    public:
        UserRegistryWatcher() NN_NOEXCEPT {}

        void Initialize() NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(account::GetUserRegistrationNotifier(&m_Notifier));
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Notifier.GetSystemEvent(&m_Receiver));
        }

        void Update() NN_NOEXCEPT
        {
            m_Receiver.Clear();
            int count;
            account::Uid uids[account::UserCountMax];
            NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&count, uids, account::UserCountMax));
            AccountPlayEventProvider::GetInstance().UpdateUserRegistry(uids, count);
        }

        inline os::SystemEvent* GetEventReceiver() NN_NOEXCEPT
        {
            return &m_Receiver;
        }

    private:
        account::Notifier   m_Notifier;
        os::SystemEvent     m_Receiver;
    };

    void ThreadFunc(void*) NN_NOEXCEPT
    {
        OpenUserTable openUserTable;
        NetworkServiceAccountAvailabilityTable networkServiceAccountAvailabilityTable;
        networkServiceAccountAvailabilityTable.Initialize();
        UserRegistryWatcher userRegistryWatcher;
        userRegistryWatcher.Initialize();
        os::SystemEvent& userRegistryChangeReceiver = *(userRegistryWatcher.GetEventReceiver());

        account::Notifier userStateChangeNotifier;
        os::SystemEvent userStateChangeEvent;
        account::Notifier networkServiceAccountAvailabilityChangeNotifier;
        os::SystemEvent networkServiceAccountAvailabilityChangeEvent;

        NN_ABORT_UNLESS_RESULT_SUCCESS(account::GetUserStateChangeNotifier(&userStateChangeNotifier));
        NN_ABORT_UNLESS_RESULT_SUCCESS(userStateChangeNotifier.GetSystemEvent(&userStateChangeEvent));
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::GetNetworkServiceAccountAvailabilityChangeNotifier(&networkServiceAccountAvailabilityChangeNotifier));
        NN_ABORT_UNLESS_RESULT_SUCCESS(networkServiceAccountAvailabilityChangeNotifier.GetSystemEvent(&networkServiceAccountAvailabilityChangeEvent));

        while( NN_STATIC_CONDITION(true) )
        {
            int index = os::WaitAny(
                g_StopEvent.GetBase(),
                userStateChangeEvent.GetBase(),
                networkServiceAccountAvailabilityChangeEvent.GetBase(),
                userRegistryChangeReceiver.GetBase()
            );

            if( index == 0 )
            {
                break;
            }

            switch( index )
            {
            case 1: // UserAccount : Open/Close
                {
                    userStateChangeEvent.Clear();
                    openUserTable.Update();
                }
                break;
            case 2: // NetworkServiceAccount : Available/Unavailable
                {
                    networkServiceAccountAvailabilityChangeEvent.Clear();
                    networkServiceAccountAvailabilityTable.Update();
                }
                break;
            case 3: // UserRegistration : Registration / Delete
                {
                    userRegistryWatcher.Update();
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    }
}

void AccountEventWatcherThread::Start() NN_NOEXCEPT
{
    g_Initialization.Initialize([]
    {
        NN_DETAIL_PDM_TRACE("[AccountEventWatcherThread] Start()\n");
        os::CreateThread(&g_Thread, ThreadFunc, nullptr,
            g_ThreadStack, sizeof(g_ThreadStack), NN_SYSTEM_THREAD_PRIORITY(pdmSrv, AccountEventWatcher));
        os::SetThreadName(&g_Thread, NN_SYSTEM_THREAD_NAME(pdmSrv, AccountEventWatcher));
        os::StartThread(&g_Thread);
    });
}

void AccountEventWatcherThread::Stop() NN_NOEXCEPT
{
    g_Initialization.Finalize([]
    {
        NN_DETAIL_PDM_TRACE("[AccountEventWatcherThread] Stop()\n");
        g_StopEvent.Signal();
        os::WaitThread(&g_Thread);
        os::DestroyThread(&g_Thread);
    });
}

}}}
