﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/sf/impl/sf_ExpHeapAllocator.h>

#include <nn/pcv/detail/pcv_IPcvService.h>

#include <nn/pcv/driver/pcv.h>

#include "pcv_ArbitrationServer.h"
#include "pcv_PcvServiceImpl.h"

namespace nn {
namespace pcv {
namespace server {

nn::Result PcvServiceImpl::SetPowerEnabled(std::int32_t moduleId, bool enabled) NN_NOEXCEPT
{
    return driver::SetPowerEnabled(static_cast<Module>(moduleId), enabled);
}

nn::Result PcvServiceImpl::SetClockEnabled(std::int32_t moduleId, bool enabled) NN_NOEXCEPT
{
    return driver::SetClockEnabled(static_cast<Module>(moduleId), enabled);
}

nn::Result PcvServiceImpl::SetClockRate(std::int32_t moduleId, std::uint32_t clockRateHz) NN_NOEXCEPT
{
    return driver::SetClockRate(static_cast<Module>(moduleId),
                                static_cast<ClockHz>(clockRateHz));
}

nn::Result PcvServiceImpl::GetClockRate(nn::sf::Out<std::uint32_t> pClockRateHz, std::int32_t moduleId) NN_NOEXCEPT
{
    ClockHz clockRate;
    nn::Result result = driver::GetClockRate(&clockRate, static_cast<Module>(moduleId));
    *pClockRateHz = clockRate;
    return result;
}

nn::Result PcvServiceImpl::GetState(nn::sf::Out<nn::pcv::ModuleState> pState, std::int32_t moduleId) NN_NOEXCEPT
{
    ModuleState state = {0};

    nn::Result result = driver::GetState(&state, static_cast<Module>(moduleId));

    *pState = state;

    return result;
}

nn::Result PcvServiceImpl::GetPossibleClockRates(nn::sf::Out<std::int32_t> pOutType,
                                                 const nn::sf::OutArray<std::uint32_t>& pOutRates,
                                                 nn::sf::Out<int> pOutCount, std::int32_t moduleId,
                                                 int maxCount) NN_NOEXCEPT
{
    ClockRatesListType type;
    ClockHz rates[MaxNumClockRates];
    int count = 0; // if GetPossibleClockRates fails, count is not set
    int limitedMaxCount = (MaxNumClockRates < maxCount) ? MaxNumClockRates : maxCount;

    nn::Result result = driver::GetPossibleClockRates(&type, rates, &count,
                                                      static_cast<Module>(moduleId),
                                                      limitedMaxCount);

    *pOutType = static_cast<std::int32_t>(type);
    for ( int index = 0; index < count; index++ )
    {
        pOutRates[index] = rates[index];
    }
    *pOutCount = count;

    return result;
}

nn::Result PcvServiceImpl::SetMinVClockRate(std::int32_t moduleId, std::uint32_t clockRateHz) NN_NOEXCEPT
{
    return driver::SetMinVClockRate(static_cast<Module>(moduleId), static_cast<ClockHz>(clockRateHz));
}

nn::Result PcvServiceImpl::SetReset(std::int32_t moduleId, bool asserted) NN_NOEXCEPT
{
    return driver::SetReset(static_cast<Module>(moduleId), asserted);
}

nn::Result PcvServiceImpl::SetVoltageEnabled(std::int32_t powerDomain, bool enabled) NN_NOEXCEPT
{
    return driver::SetVoltageEnabled(static_cast<PowerDomain>(powerDomain), enabled);
}

nn::Result PcvServiceImpl::GetVoltageEnabled(nn::sf::Out<bool> pEnabled, std::int32_t powerDomain) NN_NOEXCEPT
{
    bool enabled;
    nn::Result result = driver::GetVoltageEnabled(&enabled, static_cast<PowerDomain>(powerDomain));
    *pEnabled = enabled;
    return result;
}

nn::Result PcvServiceImpl::GetVoltageRange(nn::sf::Out<std::int32_t> pMinVolt, nn::sf::Out<std::int32_t> pMaxVolt, nn::sf::Out<std::int32_t> pStepVolt, std::int32_t powerDomain) NN_NOEXCEPT
{
    MicroVolt vMin, vMax, vStep;
    nn::Result result = driver::GetVoltageRange(&vMin, &vMax, &vStep, static_cast<PowerDomain>(powerDomain));
    *pMinVolt  = vMin;
    *pMaxVolt  = vMax;
    *pStepVolt = vStep;
    return result;
}

nn::Result PcvServiceImpl::SetVoltageValue(std::int32_t powerDomain, std::int32_t microVolt) NN_NOEXCEPT
{
    return driver::SetVoltageValue(static_cast<PowerDomain>(powerDomain), microVolt);
}

nn::Result PcvServiceImpl::GetVoltageValue(nn::sf::Out<std::int32_t> pMicroVolt, std::int32_t powerDomain) NN_NOEXCEPT
{
    MicroVolt voltValue;
    nn::Result result = driver::GetVoltageValue(&voltValue, static_cast<PowerDomain>(powerDomain));
    *pMicroVolt = voltValue;
    return result;
}

nn::Result PcvServiceImpl::GetTemperatureThresholds(const nn::sf::OutArray<TemperatureThreshold>& pThresholds,
                                nn::sf::Out<int> pOutCount, int maxCount) NN_NOEXCEPT
{
    TemperatureThreshold ranges[MaxNumTemperatureThresholds];
    int count = 0; // if GetPossibleClockRates fails, count is not set
    int limitedMaxCount = (MaxNumTemperatureThresholds < maxCount) ? MaxNumTemperatureThresholds : maxCount;

    nn::Result result = driver::GetTemperatureThresholds(ranges, &count, limitedMaxCount);

    for ( int index = 0; index < count; index++ )
    {
        pThresholds[index] = ranges[index];
    }
    *pOutCount = count;

    return result;
}

nn::Result PcvServiceImpl::SetTemperature(std::int32_t temperatureMilliC) NN_NOEXCEPT
{
    return driver::SetTemperature(static_cast<MilliC>(temperatureMilliC));
}



nn::Result PcvServiceImpl::Initialize() NN_NOEXCEPT
{
    driver::Initialize();

    NN_RESULT_SUCCESS;
}

nn::Result PcvServiceImpl::IsInitialized(nn::sf::Out<bool> pOutInitialized) NN_NOEXCEPT
{
    *pOutInitialized = driver::IsInitialized();

    NN_RESULT_SUCCESS;
}

nn::Result PcvServiceImpl::Finalize() NN_NOEXCEPT
{
    driver::Finalize();

    NN_RESULT_SUCCESS;
}

nn::Result PcvServiceImpl::PowerOn(PowerControlTarget powerControlTarget, MicroVolt microVolt) NN_NOEXCEPT
{
    return driver::PowerOn(powerControlTarget, microVolt);
}

nn::Result PcvServiceImpl::PowerOff(PowerControlTarget powerControlTarget) NN_NOEXCEPT
{
    return driver::PowerOff(powerControlTarget);
}

nn::Result PcvServiceImpl::ChangeVoltage(PowerControlTarget powerControlTarget, MicroVolt microVolt) NN_NOEXCEPT
{
    return driver::ChangeVoltage(powerControlTarget, microVolt);
}

nn::Result PcvServiceImpl::IsPoweredOn(nn::sf::Out<bool> pOutIsPoweredOn, PowerControlTarget powerControlTarget) NN_NOEXCEPT
{
    bool isPoweredOn;
    auto result = driver::IsPoweredOn(&isPoweredOn, powerControlTarget);
    *pOutIsPoweredOn = isPoweredOn;
    return result;
}

::nn::Result PcvServiceImpl::GetPowerClockInfoEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle) NN_NOEXCEPT
{
    nn::os::SystemEventType* pSystemEvent = driver::GetPowerClockInfoEvent();
    nn::os::NativeHandle internalHandle = nn::os::GetReadableHandleOfSystemEvent(pSystemEvent);

    // 寿命管理しない
    outHandle.Set(::nn::sf::NativeHandle(internalHandle, false));

    NN_RESULT_SUCCESS;
}

nn::Result PcvServiceImpl::GetOscillatorClock(nn::sf::Out<std::uint32_t> pClockRateHz) NN_NOEXCEPT
{
    *pClockRateHz = driver::GetOscillatorClock();
    NN_RESULT_SUCCESS;
}

nn::Result PcvServiceImpl::GetDvfsTable(const nn::sf::OutArray<std::uint32_t>& pOutClocks,
                                        const nn::sf::OutArray<std::int32_t>& pOutVoltages,
                                        nn::sf::Out<int32_t> pOutCount, std::int32_t moduleId, int32_t maxCount) NN_NOEXCEPT
{
    return driver::GetDvfsTable(pOutClocks.GetData(), pOutVoltages.GetData(), pOutCount.GetPointer(),
                                static_cast<Module>(moduleId), maxCount);
}

nn::Result PcvServiceImpl::GetModuleStateTable(const nn::sf::OutArray<nn::pcv::ModuleState>& pOutModuleStates,
                                nn::sf::Out<int32_t> pOutCount, int32_t maxCount) NN_NOEXCEPT
{
    driver::GetModuleStateTable(pOutModuleStates.GetData(), pOutCount.GetPointer(), maxCount);
    NN_RESULT_SUCCESS;
}

nn::Result PcvServiceImpl::GetPowerDomainStateTable(const nn::sf::OutArray<nn::pcv::PowerDomainState>& pOutPowerDomainStates,
                                    nn::sf::Out<int32_t> pOutCount, int32_t maxCount) NN_NOEXCEPT
{
    driver::GetPowerDomainStateTable(pOutPowerDomainStates.GetData(), pOutCount.GetPointer(), maxCount);
    NN_RESULT_SUCCESS;
}

nn::Result PcvServiceImpl::GetFuseInfo(const nn::sf::OutArray<uint32_t>& pOutValues,
                                    nn::sf::Out<int32_t> pOutCount, int32_t maxCount) NN_NOEXCEPT
{
    driver::GetFuseInfo(pOutValues.GetData(), pOutCount.GetPointer(), maxCount);
    NN_RESULT_SUCCESS;
}

nn::Result PcvServiceImpl::GetDramId(const nn::sf::Out<uint32_t>& pOutDramId) NN_NOEXCEPT
{
    driver::GetDramId(pOutDramId.GetPointer());
    NN_RESULT_SUCCESS;
}

} // namespace server

nn::Result ImmediateManagerImpl::SetClockRate(std::int32_t moduleId, std::uint32_t clockRateHz) NN_NOEXCEPT
{
    return driver::SetClockRate(static_cast<Module>(moduleId),
                                static_cast<ClockHz>(clockRateHz));
}

nn::Result ArbitrationManagerImpl::ReleaseControl(std::int32_t moduleId) NN_NOEXCEPT
{
    server::ReleaseControl(static_cast<Module>(moduleId));
    NN_RESULT_SUCCESS;
}

} // namespace pcv
} // namespace nn
