﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <type_traits>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>

#include <nn/pcv/detail/pcv_IPcvService.h>
#include <nn/pcv/detail/pcv_Log.h>
#include <nn/pcv/driver/pcv_Suspend.h>
#include <nn/pcv/server/pcv_PcvHipcServer.h>
#include <nn/pcv/pcv_IImmediateManager.sfdl.h>
#include <nn/psc.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>

#include "../detail/pcv_ServiceName.h"
#include "pcv_PcvServiceImpl.h"

namespace nn {
namespace pcv {
namespace server {

namespace {

struct MyServerManagerOption
{
    static const size_t PointerTransferBufferSize = 4096;
};

// HipcSimpleAllInOneServerManager は HIPC のサーバのポートとセッションを一元管理する。
class MyServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<30, 1, MyServerManagerOption>
{};

// 繰り返しのサーバの起動と終了が可能となるように placement new で初期化を行う。
// 繰り返しの起動と終了が必要ない場合には MyServerManager は直接配置してもよい。
std::aligned_storage<sizeof(MyServerManager), NN_ALIGNOF(MyServerManager)>::type g_MyServerManagerStorage;
MyServerManager* g_pMyServerManager;

nn::sf::UnmanagedServiceObject<detail::IPcvService, PcvServiceImpl> g_PcvService;

// スリープ等の通知を受け取るオブジェクト
nn::psc::PmModule g_PmModulePcvClock;
nn::psc::PmModule g_PmModulePcvVoltage;

// Immediate サーバ用オブジェクト

const int ImmediateSessionCountMax = 2;

const int NumberOfManagers = 1;

class MyImmediateServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<ImmediateSessionCountMax, NumberOfManagers>
{};

std::aligned_storage<sizeof(MyImmediateServerManager),
    NN_ALIGNOF(MyImmediateServerManager)>::type g_MyImmediateServerManagerStorage;
MyImmediateServerManager* g_pMyImmediateServerManager;

nn::sf::UnmanagedServiceObject<IImmediateManager, ImmediateManagerImpl> g_ImmediateManager;

}

void InitializePcvServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pMyServerManager);
    NN_SDK_ASSERT(!g_pMyImmediateServerManager);

    // MyServerManager オブジェクトのコンストラクト
    g_pMyServerManager = new (&g_MyServerManagerStorage) MyServerManager;

    // サービス名の登録とポートの初期化
    // sessionCountMax は、現時点では十分に大きな値を指定しておけばよい。
    // 現在はどのプロセスからアクセスしても 1つのマネージャーにアクセスされる
    auto sessionCountMax = 30;
    g_pMyServerManager->RegisterObjectForPort(g_PcvService.GetShared(), sessionCountMax, detail::PcvServiceName);

    // PSC へのモジュール登録
    auto result = g_PmModulePcvClock.Initialize(
            nn::psc::PmModuleId_PcvClock,
            nullptr, // No dependency to others
            0,
            nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    const nn::psc::PmModuleId   pcvVoltageDependencies[] =
    {
        nn::psc::PmModuleId_I2cPowerBus,
        nn::psc::PmModuleId_Gpio,
    };
    result = g_PmModulePcvVoltage.Initialize(
            nn::psc::PmModuleId_PcvVoltage,
            pcvVoltageDependencies,
            sizeof(pcvVoltageDependencies) / sizeof(pcvVoltageDependencies[0]),
            nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // サーバマネージャの開始
    // ただし、実際のサーバ動作は、LoopAuto 関数等を呼び出すことで行う必要がある。
    g_pMyServerManager->Start();

    // Immediate サーバの初期化

    g_pMyImmediateServerManager = new (&g_MyImmediateServerManagerStorage) MyImmediateServerManager;

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_pMyImmediateServerManager->RegisterObjectForPort(g_ImmediateManager.GetShared(), ImmediateSessionCountMax, detail::ImmediateServiceName));

    g_pMyImmediateServerManager->Start();
}

void LoopPcvServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    // PcvClock の PSC イベントを同時待ち対象に加える
    nn::os::MultiWaitHolderType pmEventHolderClock;
    nn::os::InitializeMultiWaitHolder(
                &pmEventHolderClock,
                g_PmModulePcvClock.GetEventPointer()->GetBase());
    nn::os::SetMultiWaitHolderUserData(
                &pmEventHolderClock,
                nn::psc::PmModuleId_PcvClock); // PmModuleId は SF のタグと衝突しないことを保証している
    g_pMyServerManager->AddUserWaitHolder(&pmEventHolderClock);

    // PcvVoltage の PSC イベントを同時待ち対象に加える
    nn::os::MultiWaitHolderType pmEventHolderVoltage;
    nn::os::InitializeMultiWaitHolder(
                &pmEventHolderVoltage,
                g_PmModulePcvVoltage.GetEventPointer()->GetBase());
    nn::os::SetMultiWaitHolderUserData(
                &pmEventHolderVoltage,
                nn::psc::PmModuleId_PcvVoltage); // PmModuleId は SF のタグと衝突しないことを保証している
    g_pMyServerManager->AddUserWaitHolder(&pmEventHolderVoltage);

    // For workaround for NSBG-6205, SIGLO-41037
    bool isPcvClockSuspendedForHardwareSleep = false;

    while (NN_STATIC_CONDITION(true))
    {
        nn::os::MultiWaitHolderType* p = g_pMyServerManager->Wait();

        switch (nn::os::GetMultiWaitHolderUserData(p))
        {
        // SF processing
        case MyServerManager::HipcSimpleAllInOneServerManager::AcceptTag:
        case MyServerManager::HipcSimpleAllInOneServerManager::InvokeTag:
            g_pMyServerManager->ProcessAuto(p);
            break;
        case nn::psc::PmModuleId_PcvClock:
        case nn::psc::PmModuleId_PcvVoltage:
            {
                auto& pmModule = (nn::os::GetMultiWaitHolderUserData(p) == nn::psc::PmModuleId_PcvClock) ?
                                 g_PmModulePcvClock : g_PmModulePcvVoltage;

                bool tryWaitResult = pmModule.GetEventPointer()->TryWait();
                NN_SDK_ASSERT(tryWaitResult);
                NN_UNUSED(tryWaitResult);
                pmModule.GetEventPointer()->Clear();
                g_pMyServerManager->AddUserWaitHolder(p);

                nn::psc::PmState    state;
                nn::psc::PmFlagSet  flags;
                auto result = pmModule.GetRequest(&state, &flags);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                switch (state)
                {
                case nn::psc::PmState_EssentialServicesSleepReady:
                {
                    if (nn::os::GetMultiWaitHolderUserData(p) == nn::psc::PmModuleId_PcvClock)
                    {
                        // FIXME: Workaround for NSBG-6205, SIGLO-41037
                        if (!flags.Test<nn::psc::PmFlag::SkippingHardwareSleepAfterEssentialServicesSleepReady>())
                        {
                            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::driver::SuspendClocks());
                            NN_DETAIL_PCV_TRACE("Finished SuspendClocks().\n");
                            isPcvClockSuspendedForHardwareSleep = true;
                        }
                        else
                        {
                            NN_DETAIL_PCV_TRACE("Skipped SuspendClocks().\n");
                        }
                    }
                    else // nn::psc::PmModuleId_PcvVoltage
                    {
                        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::driver::SuspendVoltage());
                        NN_DETAIL_PCV_TRACE("Finished SuspendVoltage().\n");
                    }
                    break;
                }
                case nn::psc::PmState_EssentialServicesAwake:
                {
                    if (nn::os::GetMultiWaitHolderUserData(p) == nn::psc::PmModuleId_PcvClock)
                    {
                        // FIXME: Workaround for NSBG-6205, SIGLO-41037
                        if (isPcvClockSuspendedForHardwareSleep)
                        {
                            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::driver::ResumeClocks());
                            NN_DETAIL_PCV_TRACE("Finished ResumeClocks().\n");
                            isPcvClockSuspendedForHardwareSleep = false;
                        }
                        else
                        {
                            NN_DETAIL_PCV_TRACE("Skipped ResumeClocks().\n");
                        }
                    }
                    else // nn::psc::PmModuleId_PcvVoltage
                    {
                        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::driver::ResumeVoltage());
                        NN_DETAIL_PCV_TRACE("Finished ResumeVoltage().\n");
                    }
                    break;
                }
                default:
                    break;
                }
                pmModule.Acknowledge(state, ResultSuccess());
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

void LoopPcvImmediateServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyImmediateServerManager);

    g_pMyImmediateServerManager->LoopAuto();
}

void RequestStopPcvServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);
    NN_SDK_ASSERT(g_pMyImmediateServerManager);

    // 処理ループの停止リクエストを送り、LoopAuto 関数を返す
    g_pMyServerManager->RequestStop();
    g_pMyImmediateServerManager->RequestStop();
}

void FinalizePcvServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);
    NN_SDK_ASSERT(g_pMyImmediateServerManager);

    // MyServerManager のデストラクト
    // 登録したサービスなどはここで登録が解除される。
    g_pMyServerManager->~MyServerManager();
    g_pMyServerManager = nullptr;

    g_pMyImmediateServerManager->~MyImmediateServerManager();
    g_pMyImmediateServerManager = nullptr;
}

} // namespace server
} // namespace pcv
} // namespace nn
