﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/dd.h>
#include <nn/os.h>
#include <nn/pcv/pcv.h>
#include <nn/pcv/pcv_Result.private.h>
#include <nn/pcv/detail/pcv_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/spl/spl_Types.h>

#include <nne/max7762x/max7762x_results.h>
#include <nne/max7762x/max7762x_regulator_Types.h>
#include <nne/max7762x/max7762x_regulator_api.h>
#include <nne/max7762x/max7762x_regulator_strings.h>
#include <nne/vcc/vcc_API.h>

#include "pcv_Conversion-hardware.nx.h"
#include "pcv_Driver.h"
#include "pcv_ErrorReport-hardware.nx.h"

// TODO: odin2 上のレギュレータドライバのヘッダが先に参照されるが宣言が見つからない為にエラーになる
// いずれ odin2 上からレギュレータドライバが取り除かれたら除去するべきワークアラウンド
namespace nne { namespace max7762x {

nne::max7762x::Voltage get_last_voltage(const nne::max7762x::RegulatorSession* pSession);
nne::max7762x::Result is_last_enabled(const nne::max7762x::RegulatorSession* pSession, bool* is_enabled);

}} // namespace nne::max7762x

namespace nn {
namespace pcv {
namespace driver {
namespace detail {

namespace {

const nn::dd::PhysicalAddress DVFS_BASE = 0x70110000;
const size_t DVFS_SIZE = 0x400;
const size_t DVFS_CL_DVFS_I2C_STS_0_OFFSET = 0x48;
const uint32_t DVFS_CL_DVFS_I2C_STS_0_I2C_LAST_VALUE_MASK = 0x0000007e;
const uint32_t DVFS_CL_DVFS_I2C_STS_0_I2C_LAST_VALUE_SHIFT = 1;
const size_t DVFS_CL_DVFS_OUTPUT_LUT_0_OFFSET = 0x200;

const size_t ClDvfsOutputLut0IndexMax = 33;

const uint32_t MAX77621_VOUT_DVS_VOUT_DVS_MASK = 0x7f;
const uint32_t MAX77812_M4_VOUT_M4_VOUT_MASK = 0xff;

NN_FORCEINLINE uint32_t ReadRegister(uintptr_t addresss) NN_NOEXCEPT
{
    return *(reinterpret_cast<uint32_t*>(addresss));
}

}

void ModuleStateTable::Initialize() NN_NOEXCEPT
{
    // m_ModuleStates の初期化
    for (int i = 0; i < ModuleStateTableSize; i++)
    {
        detail::GetState(&m_ModuleStates[i], ModuleListForErrorReport[i]);
    }
}

void ModuleStateTable::GetModuleStates(ModuleState* pOutModuleStates, int32_t* pOutCount, int32_t maxCount) NN_NOEXCEPT
{
    int count = std::min(maxCount, ModuleStateTableSize);

    for (int i = 0; i < count; i++)
    {
        pOutModuleStates[i] = m_ModuleStates[i];
    }

    *pOutCount = count;
}

void ModuleStateTable::SetPowerEnabled(Module moduleId, bool enabled) NN_NOEXCEPT
{
    for (int i = 0; i < ModuleStateTableSize; i++)
    {
        if (ModuleListForErrorReport[i] == moduleId)
        {
            m_ModuleStates[i].clockEnabled = enabled;
            break;
        }
    }
}

void ModuleStateTable::SetClockEnabled(Module moduleId, bool enabled) NN_NOEXCEPT
{
    for (int i = 0; i < ModuleStateTableSize; i++)
    {
        if (ModuleListForErrorReport[i] == moduleId)
        {
            m_ModuleStates[i].powerEnabled = enabled;
            break;
        }
    }
}

void ModuleStateTable::SetClockRate(Module moduleId, ClockHz clockRateHz) NN_NOEXCEPT
{
    for (int i = 0; i < ModuleStateTableSize; i++)
    {
        if (ModuleListForErrorReport[i] == moduleId)
        {
            m_ModuleStates[i].clockFrequency = clockRateHz;
            break;
        }
    }
}

void ModuleStateTable::SetMinVClockRate(Module moduleId, ClockHz clockRateHz) NN_NOEXCEPT
{
    for (int i = 0; i < ModuleStateTableSize; i++)
    {
        if (ModuleListForErrorReport[i] == moduleId)
        {
            m_ModuleStates[i].minVClockRate = clockRateHz;
            break;
        }
    }
}

void ModuleStateTable::SetReset(Module moduleId, bool asserted) NN_NOEXCEPT
{
    for (int i = 0; i < ModuleStateTableSize; i++)
    {
        if (ModuleListForErrorReport[i] == moduleId)
        {
            m_ModuleStates[i].resetAsserted = asserted;
            break;
        }
    }
}

void PowerDomainStateTable::Initialize(nn::spl::Regulator regulator) NN_NOEXCEPT
{
    switch ( regulator )
    {
    case nn::spl::Regulator_Max77621:
        m_pPowerDomainListForErrorReport = &(PowerDomainListForErrorReportForMax77621[0]);
        m_PowerDomainStateTableSize = PowerDomainStateTableSizeForMax77621;
        break;
    case nn::spl::Regulator_Max77812PhaseConfiguration31:
    case nn::spl::Regulator_Max77812PhaseConfiguration211:
        m_pPowerDomainListForErrorReport = &(PowerDomainListForErrorReportForMax77812[0]);
        m_PowerDomainStateTableSize = PowerDomainStateTableSizeForMax77812;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // m_PowerDomainStates の初期化
    for (int i = 0; i < m_PowerDomainStateTableSize; i++)
    {
        auto& state = m_PowerDomainStates[i];
        detail::GetVoltageEnabled(&state.enabled, *(m_pPowerDomainListForErrorReport + i));
        detail::GetVoltageValue(&state.voltage, *(m_pPowerDomainListForErrorReport + i));
    }

    m_DvfsBase = nn::dd::QueryIoMappingAddress(DVFS_BASE, DVFS_SIZE);
}

Result PowerDomainStateTable::GetRegisterValueFromLookUpTable(uint32_t* pOutReigsterValue) NN_NOEXCEPT
{
    size_t index = (ReadRegister(m_DvfsBase + DVFS_CL_DVFS_I2C_STS_0_OFFSET)
        & DVFS_CL_DVFS_I2C_STS_0_I2C_LAST_VALUE_MASK)
        >> DVFS_CL_DVFS_I2C_STS_0_I2C_LAST_VALUE_SHIFT;
    NN_SDK_ASSERT(index <= ClDvfsOutputLut0IndexMax);
    if ( index > ClDvfsOutputLut0IndexMax )
    {
        return ResultInvalidIndexOfLookUpTable();
    }
    *pOutReigsterValue = ReadRegister(m_DvfsBase + DVFS_CL_DVFS_OUTPUT_LUT_0_OFFSET + index * sizeof(uint32_t));
    NN_RESULT_SUCCESS;
}

uint32_t PowerDomainStateTable::GetMax77621CpuVoltageMicroVolt() NN_NOEXCEPT
{
    const uint32_t BaseMicroVolt = 606250;
    const uint32_t StepMicroVolt = 6250;

    uint32_t registerValue = 0;
    if ( GetRegisterValueFromLookUpTable(&registerValue).IsFailure() )
    {
        // index が不正な値であれば 0V とします
        return 0;
    }

    return (registerValue & MAX77621_VOUT_DVS_VOUT_DVS_MASK) * StepMicroVolt + BaseMicroVolt;
}

uint32_t PowerDomainStateTable::GetMax77812CpuVoltageMicroVolt() NN_NOEXCEPT
{
    const uint32_t BaseMicroVolt = 250000;
    const uint32_t StepMicroVolt = 5000;

    uint32_t registerValue = 0;
    if ( GetRegisterValueFromLookUpTable(&registerValue).IsFailure() )
    {
        // index が不正な値であれば 0V とします
        return 0;
    }

    return (registerValue & MAX77812_M4_VOUT_M4_VOUT_MASK) * StepMicroVolt + BaseMicroVolt;
}

void PowerDomainStateTable::GetPowerDomainStates(PowerDomainState* pOutPowerDomainStates, int32_t* pOutCount, int32_t maxCount) NN_NOEXCEPT
{
    int count = std::min(maxCount, m_PowerDomainStateTableSize);

    for (int i = 0; i < count; i++)
    {
        auto powerDomain = *(m_pPowerDomainListForErrorReport + i);

        switch ( powerDomain )
        {
        case PowerDomain_Max77621_Cpu:
            m_PowerDomainStates[i].voltage = GetMax77621CpuVoltageMicroVolt();
            m_PowerDomainStates[i].enabled = true;
            break;
        case PowerDomain_Max77812_Cpu:
            m_PowerDomainStates[i].voltage = GetMax77812CpuVoltageMicroVolt();
            m_PowerDomainStates[i].enabled = true;
            break;
        case PowerDomain_Max77621_Gpu:
        case PowerDomain_Max77812_Gpu:
            {
                nne::max7762x::RegulatorSession session;

                // 製品環境では ErrorReport を送ることを優先して ABORT しない
                auto result = MakeResult(nne::max7762x::OpenSession(&session, nne::max7762x::str_regulator_name[ConvertPowerDomainToRegulatorHandle(powerDomain)]));
                NN_SDK_ASSERT(result.IsSuccess());

                if ( result.IsSuccess() )
                {
                    m_PowerDomainStates[i].voltage = nne::max7762x::get_last_voltage(&session);

                    result = MakeResult(nne::max7762x::is_last_enabled(&session, &m_PowerDomainStates[i].enabled));
                    NN_SDK_ASSERT(result.IsSuccess());
                    if ( result.IsFailure() )
                    {
                        // 失敗時の値は false に固定
                        m_PowerDomainStates[i].enabled = false;
                    }

                    result = MakeResult(nne::max7762x::CloseSession(&session));
                    NN_SDK_ASSERT(result.IsSuccess());

                    NN_UNUSED(result);
                }
            }

            break;
        default:
            break;
        }

        pOutPowerDomainStates[i] = m_PowerDomainStates[i];
    }

    *pOutCount = count;
}

void PowerDomainStateTable::SetVoltageEnabled(PowerDomain powerDomain, bool enabled) NN_NOEXCEPT
{
    for (int i = 0; i < m_PowerDomainStateTableSize; i++)
    {
        if (*(m_pPowerDomainListForErrorReport + i) == powerDomain)
        {
            m_PowerDomainStates[i].enabled = enabled;
            break;
        }
    }
}

void PowerDomainStateTable::SetVoltageValue(PowerDomain powerDomain, MicroVolt microVolt) NN_NOEXCEPT
{
    for (int i = 0; i < m_PowerDomainStateTableSize; i++)
    {
        if (*(m_pPowerDomainListForErrorReport + i) == powerDomain)
        {
            m_PowerDomainStates[i].voltage = microVolt;
            break;
        }
    }
}

void DvfsTable::Initialize() NN_NOEXCEPT
{
    m_Cpu.count = DvfsTableMaxDataCount;
    nne::vcc::QueryDvfsCurve(IP_CPU, m_Cpu.clocks, m_Cpu.voltages, &m_Cpu.count);
    m_Gpu.count = DvfsTableMaxDataCount;
    nne::vcc::QueryDvfsCurve(IP_GPU, m_Gpu.clocks, m_Gpu.voltages, &m_Gpu.count);
    m_Emc.count = DvfsTableMaxDataCount;
    nne::vcc::QueryDvfsCurve(IP_EMC, m_Emc.clocks, m_Emc.voltages, &m_Emc.count);

    // 取得した DVFS テーブルの要素数が最大値と同じだったら、データがあふれたかもしれないので警告
    if (m_Cpu.count == DvfsTableMaxDataCount ||
        m_Gpu.count == DvfsTableMaxDataCount ||
        m_Emc.count == DvfsTableMaxDataCount)
    {
        NN_DETAIL_PCV_TRACE("DVFS table for error report may have overflowed (CPU:%d, GPU:%d, EMC:%d).\n", m_Cpu.count, m_Gpu.count, m_Emc.count);
    }
}

Result DvfsTable::GetDvfsTable(uint32_t* pOutClocks, int32_t* pOutVoltages, int32_t* pOutCount, Module moduleId, int32_t maxCount) NN_NOEXCEPT
{
    DvfsTableModule* pTable;

    switch (moduleId)
    {
    case Module_Cpu:
        pTable = &m_Cpu;
        break;
    case Module_Gpu:
        pTable = &m_Gpu;
        break;
    case Module_Emc:
        pTable = &m_Emc;
        break;
    default:
        return nn::pcv::ResultInvalidModuleId();
    }

    int32_t count = maxCount < pTable->count ? maxCount : pTable->count;
    for (int32_t i = 0; i < count; i++)
    {
        pOutClocks[i] = pTable->clocks[i];
        pOutVoltages[i] = pTable->voltages[i];
    }
    *pOutCount = count;

    return nn::ResultSuccess();
}

void FuseInfo::Initialize() NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(sizeof(m_Values) / sizeof(m_Values[0]), 6);

    uintptr_t fuseAddress = nn::dd::QueryIoMappingAddress(FusePhysicalAddress, FuseAddressSpaceSize);

    // NSBG-5801 で要求されたフォーマットに従い、データを格納
    m_Values[0] = ReadRegister(fuseAddress + 0x114);
    m_Values[1] = ReadRegister(fuseAddress + 0x130);
    m_Values[2] = ReadRegister(fuseAddress + 0x134);
    m_Values[3] = ReadRegister(fuseAddress + 0x118) * 4;
    m_Values[4] = ReadRegister(fuseAddress + 0x228) * 5;
    m_Values[5] = ReadRegister(fuseAddress + 0x140) * 4;
}

void FuseInfo::GetFuseInfo(uint32_t* pOutValues, int32_t* pOutCount, int32_t maxCount) NN_NOEXCEPT
{
    const int valueCount = sizeof(m_Values) / sizeof(m_Values[0]);
    const int32_t count = maxCount < valueCount ? maxCount : valueCount;

    for (int32_t i = 0; i < count; i++)
    {
        pOutValues[i] = m_Values[i];
    }
    *pOutCount = count;
}

void DramInfo::Initialize(uint32_t dramId) NN_NOEXCEPT
{
    m_DramId = dramId;
}

void DramInfo::GetDramId(uint32_t* pOutDramId) NN_NOEXCEPT
{
    *pOutDramId = m_DramId;
}

} // namespace detail
} // namespace driver
} // namespace pcv
} // namespace nn
