﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_SystemThreadDefinition.h>

#include <nn/dd.h>
#include <nn/os.h>

#include <nn/pcv/detail/pcv_Log.h>
#include <nn/pcv/pcv.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/spl/spl_Api.h>
#include <nn/spl/spl_Types.h>

#include <nne/vcc/vcc_API.h>
#include <nne/vcc/vcc_dram_ids.h>
#include <nne/vcc/vcc_results.h>
#include <nne/max7762x/max7762x_results.h>
#include <nne/max7762x/max7762x_regulator_Types.h>
#include <nne/max7762x/max7762x_regulator_api.h>
#include <nne/max7762x/max7762x_regulator_strings.h>

#include "pcv_Conversion-hardware.nx.h"
#include "pcv_Driver.h"
#include "pcv_GetModuleName.h"
#include "pcv_ErrorReport-hardware.nx.h"

#define NN_PCV_USB2_HACK

namespace {

// Initialize の参照カウント
int g_InitializeCount = 0;
int g_NumVccThresholds = 0;
nne::vcc::TemperatureThreshold* g_pVccThresholds = NULL;

// 参照カウントを守る Mutex
struct StaticMutex
{
    nn::os::MutexType mutex;
    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex(&mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex(&mutex);
    }
} g_InitializeCountMutex = { NN_OS_MUTEX_INITIALIZER(false) };

bool IsModuleValid(nn::pcv::Module moduleId) NN_NOEXCEPT
{
    return moduleId < nn::pcv::Module_NumModule;
}

bool IsPowerDomainValid(nn::pcv::PowerDomain powerDomain) NN_NOEXCEPT
{
    return ((0 <= powerDomain) && (powerDomain < nn::pcv::PowerDomain_NumPowerDomain));
}

nne::vcc::VccThreadInfo g_EmcThreadInfo =
{
    "EmcDvfsPeriodicCompHandler",
    NN_SYSTEM_THREAD_NAME(pcv, EmcDvfsPeriodicCompHandler),
    NN_SYSTEM_THREAD_PRIORITY(pcv, EmcDvfsPeriodicCompHandler),
};

nne::vcc::ClockHz g_OscRateHz = 0;

// Develop ビルドでの CPU 負荷低減の為、アクセス回数が多くなり得るモジュールはログ出力しない。
const nn::pcv::Module NoLogModuleList[] =
{
    nn::pcv::Module_Nvenc,
    nn::pcv::Module_Nvdec,
    nn::pcv::Module_Nvjpg,

    // SIGLO-78029: EDEV で INA が存在しないため I2C で上限までリトライが行われて大量のログが出る問題の回避
    nn::pcv::Module_I2c2,
};

bool IsLogEnabled(nn::pcv::Module moduleId) NN_NOEXCEPT
{
    for ( auto& noLogModule : NoLogModuleList )
    {
        if ( noLogModule == moduleId )
        {
            return false;
        }
    }

    return true;
}

// エラーレポート用
nn::os::SystemEvent g_ErrorReportEvent(nn::os::EventClearMode_ManualClear, true);
nn::pcv::driver::detail::ModuleStateTable g_ModuleStateTable;
nn::pcv::driver::detail::PowerDomainStateTable g_PowerDomainStateTable;
nn::pcv::driver::detail::DvfsTable g_DvfsTable;
nn::pcv::driver::detail::FuseInfo g_FuseInfo;
nn::pcv::driver::detail::DramInfo g_DramInfo;

} // namespace

namespace nn {
namespace pcv {
namespace driver {
namespace detail {

void Initialize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    nn::spl::Initialize();

    auto dramId = nn::spl::GetDramId();

    if ( g_InitializeCount == 0 )
    {
        nne::vcc::Initialize(&g_OscRateHz, ConvertDramId(dramId), &g_EmcThreadInfo);

        NN_DETAIL_PCV_TRACE("PCV detects OSC rate %u Hz.\n", g_OscRateHz);
        if ( g_OscRateHz == 0 )
        {
            // Nothing to do now.
        }
    }

    g_InitializeCount++;

    // We cannot include Raptor's VCC header in our pcv_Types.h (to export to others). Thus, we have
    // to manually hand match/copy the values from Raptor's vcc_limits.h (included by vcc_API.h).
    // We can verify here that we hand copied the values correctly.
    NN_SDK_ASSERT_EQUAL(nne::vcc::MaxRatesFromVcc, nn::pcv::MaxNumClockRates);
    NN_SDK_ASSERT_EQUAL(nne::vcc::MaxNumThresholds, nn::pcv::MaxNumTemperatureThresholds);

    // The temperature thresholds that VCC library is interested in is static.  We will obtain
    // the data and check if the call fails here.  We prefer to abort here at initializaton,
    // rather than letting a dynamic client request abort the pcv process.
    g_pVccThresholds = nne::vcc::QueryTemperatureThresholds(&g_NumVccThresholds);
    NN_ABORT_UNLESS(g_NumVccThresholds > 0, "[pcv] Vcc returns invalid number of temperature thresholds.");
    NN_ABORT_UNLESS(g_pVccThresholds != nullptr, "[pcv] Vcc returns a NULL temperature threshold array.");

    auto regulator = nn::spl::GetRegulator();
    nn::spl::Finalize();

    // 初期化時のエラーレポート用処理
    g_DramInfo.Initialize(static_cast<uint32_t>(dramId));
    g_FuseInfo.Initialize();
    g_ModuleStateTable.Initialize();
    g_PowerDomainStateTable.Initialize(regulator);
    g_DvfsTable.Initialize();
    g_ErrorReportEvent.Signal();
}

bool IsInitialized() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    return (g_InitializeCount > 0) ? true : false;
}

void Finalize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    g_InitializeCount--;
}

Result SetPowerEnabled(Module moduleId, bool enabled) NN_NOEXCEPT
{
    if ( !IsModuleValid(moduleId))
    {
        return ResultInvalidModuleId();
    }

    auto result = MakeResult(nne::vcc::SetPowerEnabled(ConvertModuleToIp(moduleId), enabled));

    g_ModuleStateTable.SetPowerEnabled(moduleId, enabled);
    g_ErrorReportEvent.Signal();

    return result;
}

Result SetClockEnabled(Module moduleId, bool enabled) NN_NOEXCEPT
{
    if ( !IsModuleValid(moduleId) )
    {
        return ResultInvalidModuleId();
    }

    auto result = MakeResult(nne::vcc::SetClockEnabled(ConvertModuleToIp(moduleId), enabled));

    g_ModuleStateTable.SetClockEnabled(moduleId, enabled);
    g_ErrorReportEvent.Signal();

    return result;
}

Result SetClockRate(Module moduleId, ClockHz clockRateHz) NN_NOEXCEPT
{
    if ( IsLogEnabled(moduleId) )
    {
        NN_DETAIL_PCV_TRACE("%s(%s, %u).\n", __FUNCTION__, GetModuleName(moduleId), clockRateHz);
    }

    if ( !IsModuleValid(moduleId) )
    {
        return ResultInvalidModuleId();
    }

    auto result = MakeResult(nne::vcc::SetClockRate(ConvertModuleToIp(moduleId), clockRateHz));

    g_ModuleStateTable.SetClockRate(moduleId, clockRateHz);
    g_ErrorReportEvent.Signal();

    return result;
}

Result GetClockRate(ClockHz* pClockRateHz, Module moduleId) NN_NOEXCEPT
{
    if ( !IsModuleValid(moduleId) )
    {
        return ResultInvalidModuleId();
    }

    nne::vcc::IPSettingsQuery vccQuery;
    vccQuery.id = ConvertModuleToIp(moduleId);
    nn::Result rc = MakeResult(nne::vcc::QuerySettings(&vccQuery));
    *pClockRateHz = vccQuery.f_Hz;

    return rc;
}

Result GetState(ModuleState *pState, Module moduleId) NN_NOEXCEPT
{
    if ( !IsModuleValid(moduleId) )
    {
        return ResultInvalidModuleId();
    }

    nn::Result rc;

    nne::vcc::IPSettingsQuery vccQuery;
    vccQuery.id = ConvertModuleToIp(moduleId);
    vccQuery.clock_enabled = false;
    vccQuery.power_enabled = false;
    vccQuery.reset_asserted = false;
    vccQuery.f_Hz = 0;
    vccQuery.minV_Hz = 0;
    rc = MakeResult(nne::vcc::QuerySettings(&vccQuery));

    pState->clockEnabled = vccQuery.clock_enabled;
    pState->powerEnabled = vccQuery.power_enabled;
    pState->resetAsserted = vccQuery.reset_asserted;
    pState->clockFrequency = vccQuery.f_Hz;
    pState->minVClockRate = vccQuery.minV_Hz;

    return rc;
}

Result GetPossibleClockRates(ClockRatesListType* pOutType,
                             ClockHz* pOutRates,
                             int* pOutCount,
                             Module moduleId,
                             int maxCount) NN_NOEXCEPT
{
    if ( !IsModuleValid(moduleId) )
    {
        return ResultInvalidModuleId();
    }

    ClockHz rates[MaxNumClockRates];
    int numRates = MaxNumClockRates;
    nn::Result rc = MakeResult(nne::vcc::QueryPossibleRates(ConvertModuleToIp(moduleId), rates, &numRates));

    if ( rc.IsSuccess() )
    {
        int outCount = 0;
        for ( int i = 0; i < numRates && outCount < maxCount; ++i )
        {
            pOutRates[outCount++] = rates[i];
        }
        *pOutCount = outCount;

        if ( pOutRates[0] == 0 )
        {
            *pOutType = ClockRatesListType_Range;
        }
        else
        {
            *pOutType = ClockRatesListType_Discrete;
        }
    }
    else
    {
        *pOutType = ClockRatesListType_Invalid;
        *pOutCount = 0;
    }

    return rc;
}

Result SetMinVClockRate(Module moduleId, ClockHz clockRateHz) NN_NOEXCEPT
{
    if ( !IsModuleValid(moduleId) )
    {
         return ResultInvalidModuleId();
    }

    auto result = MakeResult(nne::vcc::SetMinVClockRate(ConvertModuleToIp(moduleId), clockRateHz));

    g_ModuleStateTable.SetMinVClockRate(moduleId, clockRateHz);
    g_ErrorReportEvent.Signal();

    return result;
}

Result SetReset(Module moduleId, bool asserted) NN_NOEXCEPT
{
    if ( !IsModuleValid(moduleId) )
    {
         return ResultInvalidModuleId();
    }

    auto result = MakeResult(nne::vcc::SetResetAsserted(ConvertModuleToIp(moduleId), asserted));

    g_ModuleStateTable.SetReset(moduleId, asserted);
    g_ErrorReportEvent.Signal();

    return result;
}

Result SetVoltageEnabled(PowerDomain powerDomain, bool enabled) NN_NOEXCEPT
{
    nn::Result rc;

    if (!IsPowerDomainValid(powerDomain))
    {
        rc = ResultInvalidPowerDomain();
    }
    else
    {
        nne::max7762x::Result           rc2;
        nne::max7762x::RegulatorSession session;
        rc2 = nne::max7762x::OpenSession(&session, nne::max7762x::str_regulator_name[ConvertPowerDomainToRegulatorHandle(powerDomain)]);
        if (rc2 != nne::max7762x::Result_Success)
        {
            NN_DETAIL_PCV_WARN("<<< %s: cannot open session for domain %d.\n", __FUNCTION__, powerDomain);
            return ResultPMICCannotOpenSession();
        }
        rc = enabled ? MakeResult(nne::max7762x::enable(&session)) : MakeResult(nne::max7762x::disable(&session));
        rc2 = nne::max7762x::CloseSession(&session);
    }

    g_PowerDomainStateTable.SetVoltageEnabled(powerDomain, enabled);
    g_ErrorReportEvent.Signal();

    return rc;
}

Result GetVoltageEnabled(bool* pEnabled, PowerDomain powerDomain) NN_NOEXCEPT
{
    nn::Result rc;

    if (!IsPowerDomainValid(powerDomain))
    {
        rc = ResultInvalidPowerDomain();
    }
    else
    {
        nne::max7762x::Result           rc2;
        nne::max7762x::RegulatorSession session;
        rc2 = nne::max7762x::OpenSession(&session, nne::max7762x::str_regulator_name[ConvertPowerDomainToRegulatorHandle(powerDomain)]);
        if (rc2 != nne::max7762x::Result_Success)
        {
            NN_DETAIL_PCV_WARN("<<< %s: cannot open session for domain %d.\n", __FUNCTION__, powerDomain);
            return ResultPMICCannotOpenSession();
        }
        rc = MakeResult(nne::max7762x::is_enabled(&session, pEnabled));
        rc2 = nne::max7762x::CloseSession(&session);
    }
    return rc;
}

Result GetVoltageRange(MicroVolt* pMinVolt, MicroVolt* pMaxVolt, MicroVolt* pStepVolt, PowerDomain powerDomain) NN_NOEXCEPT
{
    NN_DETAIL_PCV_TRACE("PCV %s() not yet fully implemented by VCC.\n", __FUNCTION__);
    if ((NULL == pMinVolt) || (NULL == pMaxVolt) || (NULL == pStepVolt))
    {
        NN_DETAIL_PCV_WARN("PCV %s(): invalid params.\n", __FUNCTION__);
        return ResultInvalidArg();
    }

    if (!IsPowerDomainValid(powerDomain))
    {
        NN_DETAIL_PCV_WARN("PCV %s(): invalid powerdomain.\n", __FUNCTION__);
        return ResultInvalidPowerDomain();
    }

    nn::Result                      rc;
    nne::max7762x::Result           rc2;
    nne::max7762x::RegulatorSession session;
    rc2 = nne::max7762x::OpenSession(&session, nne::max7762x::str_regulator_name[ConvertPowerDomainToRegulatorHandle(powerDomain)]);
    if (rc2 != nne::max7762x::Result_Success)
    {
        NN_DETAIL_PCV_WARN("<<< %s: cannot open session for domain %d.\n", __FUNCTION__, powerDomain);
        return ResultPMICCannotOpenSession();
    }
    rc = MakeResult(nne::max7762x::get_voltage_range(&session,
                                                     (nne::max7762x::Voltage*)pMinVolt,
                                                     (nne::max7762x::Voltage*)pMaxVolt,
                                                     (nne::max7762x::Voltage*)pStepVolt));
    rc2 = nne::max7762x::CloseSession(&session);
    return rc;
}

Result SetVoltageValue(PowerDomain powerDomain, MicroVolt microVolt) NN_NOEXCEPT
{
    if (!IsPowerDomainValid(powerDomain))
    {
        return ResultInvalidPowerDomain();
    }

    nn::Result                      rc;
    nne::max7762x::Result           rc2;
    nne::max7762x::RegulatorSession session;
    rc2 = nne::max7762x::OpenSession(&session, nne::max7762x::str_regulator_name[ConvertPowerDomainToRegulatorHandle(powerDomain)]);
    if (rc2 != nne::max7762x::Result_Success)
    {
        NN_DETAIL_PCV_WARN("<<< %s: cannot open session for domain %d.\n", __FUNCTION__, powerDomain);
        return ResultPMICCannotOpenSession();
    }
    rc = MakeResult(set_voltage(&session, microVolt));
    rc2 = nne::max7762x::CloseSession(&session);

    g_PowerDomainStateTable.SetVoltageValue(powerDomain, microVolt);
    g_ErrorReportEvent.Signal();

    return rc;
}

Result GetVoltageValue(MicroVolt* pMicroVolt, PowerDomain powerDomain) NN_NOEXCEPT
{
    if (NULL == pMicroVolt)
    {
        return ResultInvalidArg();
    }
    if (!IsPowerDomainValid(powerDomain))
    {
        return ResultInvalidPowerDomain();
    }

    nne::max7762x::Result           rc2;
    nne::max7762x::RegulatorSession session;
    rc2 = nne::max7762x::OpenSession(&session, nne::max7762x::str_regulator_name[ConvertPowerDomainToRegulatorHandle(powerDomain)]);
    if (rc2 != nne::max7762x::Result_Success)
    {
        NN_DETAIL_PCV_WARN("<<< %s: cannot open session for domain %d.\n", __FUNCTION__, powerDomain);
        return ResultPMICCannotOpenSession();
    }
    *pMicroVolt = (MicroVolt) nne::max7762x::get_voltage(&session);
    rc2 = nne::max7762x::CloseSession(&session);
    return ResultSuccess();
}

Result GetTemperatureThresholds(TemperatureThreshold* pThresholds, int* pOutCount, int maxCount) NN_NOEXCEPT
{
    if ((NULL == pThresholds) || (NULL == pOutCount) || (maxCount <= 0))
    {
        return ResultInvalidArg();
    }

    // Copy the thresholds from the array provided by VCC (obtained in our Initialize() above)
    int count = ( maxCount < g_NumVccThresholds ) ? maxCount : g_NumVccThresholds;
    TemperatureThreshold*           dst = pThresholds;
    nne::vcc::TemperatureThreshold* src = g_pVccThresholds;
    for ( int index = 0; index < count; index++ )
    {
        dst->minMilliC = static_cast<MilliC>(src->min_mC);
        dst->maxMilliC = static_cast<MilliC>(src->max_mC);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)

        // 温度センサの分解能を考慮して最少ヒステリシスは 1 度（0 は不可）。
        const nn::pcv::MilliC TemperatureRangeHisterisis = 1000;

        // ヒステリシスが x 度以上ある。
        NN_ABORT_UNLESS((dst->maxMilliC - dst->minMilliC) >= TemperatureRangeHisterisis);

        // 下限値間の関係と上限値間の関係が妥当である。
        if ( index > 0 )
        {
            NN_ABORT_UNLESS((dst - 1)->minMilliC < dst->minMilliC);
            NN_ABORT_UNLESS((dst - 1)->maxMilliC < dst->maxMilliC);
        }

#endif

        dst++;
        src++;
    }
    *pOutCount = count;

    NN_RESULT_SUCCESS;
}

Result SetTemperature(MilliC temperatureMilliC) NN_NOEXCEPT
{
    auto result = MakeResult(nne::vcc::SetTemperature(static_cast<nne::vcc::Temp_mC>(temperatureMilliC)));

    // 温度変更による電圧変更を想定したエラーコンテキスト再取得の通知
    g_ErrorReportEvent.Signal();

    return result;
}

Result SuspendClocks() NN_NOEXCEPT
{
    return MakeResult(nne::vcc::SuspendClocks());
}

Result ResumeClocks() NN_NOEXCEPT
{
    return MakeResult(nne::vcc::ResumeClocks());
}

Result SuspendVoltage() NN_NOEXCEPT
{
    NN_RESULT_DO(MakeResult(nne::vcc::SuspendVoltage()));
    NN_RESULT_DO(MakeResult(nne::max7762x::Suspend()));
    NN_RESULT_SUCCESS;
}

Result ResumeVoltage() NN_NOEXCEPT
{
    NN_RESULT_DO(MakeResult(nne::max7762x::Resume()));
    NN_RESULT_DO(MakeResult(nne::vcc::ResumeVoltage()));
    NN_RESULT_SUCCESS;
}

nn::os::SystemEventType* GetPowerClockInfoEvent() NN_NOEXCEPT
{
    return g_ErrorReportEvent.GetBase();
}

uint32_t GetOscillatorClock() NN_NOEXCEPT
{
    return static_cast<uint32_t>(g_OscRateHz);
}

Result GetDvfsTable(uint32_t* pOutClocks, int32_t* pOutVoltages, int32_t* pOutCount, Module moduleId, int32_t maxCount) NN_NOEXCEPT
{
    return g_DvfsTable.GetDvfsTable(pOutClocks, pOutVoltages, pOutCount, moduleId, maxCount);
}

void GetModuleStateTable(ModuleState* pOutModuleStates, int32_t* pOutCount, int32_t maxCount) NN_NOEXCEPT
{
    g_ModuleStateTable.GetModuleStates(pOutModuleStates, pOutCount, maxCount);
}

void GetPowerDomainStateTable(PowerDomainState* pOutPowerDomainStates, int32_t* pOutCount, int32_t maxCount) NN_NOEXCEPT
{
    g_PowerDomainStateTable.GetPowerDomainStates(pOutPowerDomainStates, pOutCount, maxCount);
}

void GetFuseInfo(uint32_t* pOutValues, int32_t* pOutCount, int32_t maxCount) NN_NOEXCEPT
{
    g_FuseInfo.GetFuseInfo(pOutValues, pOutCount, maxCount);
}

void GetDramId(uint32_t* pOutDramId) NN_NOEXCEPT
{
    g_DramInfo.GetDramId(pOutDramId);
}

} // namespace detail
} // namespace driver
} // namespace pcv
} // namespace nn
