﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/pctl_UnlinkDevice.h>

#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ServiceConfig.h>
#include <nn/pctl/detail/service/pctl_ServiceMemoryManagement.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/watcher/dispatcher/pctl_RetrieveOwnersDispatcher.h>
#include <nn/pctl/detail/service/watcher/dispatcher/pctl_UnlinkDeviceDispatcher.h>

#include <nn/result/result_HandlingUtility.h>

#include <nn/util/util_ScopeExit.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

nn::Result UnlinkDeviceExecutor::ExecuteInner(common::NetworkBuffer& bufferInfo) NN_NOEXCEPT
{
    TokenHolder tokenHolder;
    NN_RESULT_DO(g_pWatcher->GetNetworkManager().AcquireAuthenticationToken(tokenHolder, bufferInfo, m_pCancelable));

    size_t actualOwnerCount = 0;
    ServerDeviceOwner* pOwners = static_cast<ServerDeviceOwner*>(
        AllocateMemoryBlock(sizeof(ServerDeviceOwner) * MaxDeviceOwnerCount)
        );
    NN_RESULT_THROW_UNLESS(pOwners != nullptr, nn::pctl::ResultHttpErrorOutOfMemory());
    NN_UTIL_SCOPE_EXIT { FreeMemoryBlock(pOwners); };

    NN_RESULT_DO(
        dispatcher::RetrieveDeviceOwnersDispatcher::Execute(&actualOwnerCount, pOwners, MaxDeviceOwnerCount,
            bufferInfo, m_pCancelable, tokenHolder.GetToken(), m_DeviceId)
        );

    // actualOwnerCount == 0 であっても特にエラーにはしない
    for (size_t i = 0; i < actualOwnerCount; ++i)
    {
        NN_RESULT_DO(
            dispatcher::UnlinkDeviceDispatcher::Execute(bufferInfo, m_pCancelable, tokenHolder.GetToken(),
                m_DeviceId, pOwners[i].user.accountId)
            );
    }

    NN_RESULT_SUCCESS;
}

nn::Result UnlinkDeviceExecutor::Execute(common::NetworkBuffer& bufferInfo) NN_NOEXCEPT
{
    auto result = ExecuteInner(bufferInfo);
    if (result.IsSuccess())
    {
        m_IsAlreadyFailedByInvalidToken = false;
    }
    // 最初の Invalid token はそのまま返す
    else if (nn::pctl::ResultServerInvalidToken::Includes(result))
    {
        if (!m_IsAlreadyFailedByInvalidToken)
        {
            m_IsAlreadyFailedByInvalidToken = true;
            return result;
        }
    }
    if (result.IsSuccess() || m_IsAlwaysClearPairingInfo)
    {
        g_pWatcher->GetNetworkManager().ClearPairingInfo(true);
    }
    return result;
}

void UnlinkDeviceExecutor::Cancel() NN_NOEXCEPT
{
    if (m_pCancelable != nullptr)
    {
        m_pCancelable->Cancel();
    }
}

}}}}}
