﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/pctl_Pairing.h>

#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ServiceConfig.h>
#include <nn/pctl/detail/service/pctl_ServiceMain.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/common/pctl_SystemInfo.h>
#include <nn/pctl/detail/service/watcher/pctl_Authentication.h>
#include <nn/pctl/detail/service/watcher/pctl_Notifications.h>
#include <nn/pctl/detail/service/watcher/pctl_WatcherErrorHandler.h>
#include <nn/pctl/detail/service/watcher/dispatcher/pctl_RetrieveOwnersDispatcher.h>
#include <nn/pctl/detail/service/watcher/dispatcher/pctl_UpdateDeviceDispatcher.h>

#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

////////////////////////////////////////////////////////////////////////////////

nn::Result RequestPairing::Execute(common::NetworkBuffer& bufferInfo) NN_NOEXCEPT
{
    auto& manager = g_pWatcher->GetNetworkManager();
    if (!manager.IsNotificationTokenRetrieved())
    {
        m_NotificationToken.data[0] = 0;
        NN_RESULT_DO(
            CreateNotificationToken::Execute(m_NotificationToken, m_pCancelable)
            );
        // Generic版では取得されないのでそのチェックを行う
        if (m_NotificationToken.data[0] != 0)
        {
            manager.StoreNotificationToken(m_NotificationToken);
        }
    }

    TokenHolder tokenHolder;
    NN_RESULT_DO(manager.AcquireAuthenticationToken(tokenHolder, bufferInfo, m_pCancelable));

    ServerDeviceSimple device;
    NN_RESULT_DO(
        dispatcher::FindDeviceWithPairingCodeDispatcher::Execute(device, bufferInfo, m_pCancelable, tokenHolder.GetToken(), m_PairingCode)
        );
    NN_RESULT_DO(
        dispatcher::RetrieveDeviceOwnersDispatcher::Execute(m_ActualOwnerCountPointer, m_pOwners, m_MaxOwnerCount,
            bufferInfo, m_pCancelable, tokenHolder.GetToken(), device.id)
        );
    *m_pDeviceId = device.id;
    NN_RESULT_SUCCESS;
}

void RequestPairing::Cancel() NN_NOEXCEPT
{
    if (m_pCancelable != nullptr)
    {
        m_pCancelable->Cancel();
    }
}

////////////////////////////////////////////////////////////////////////////////

nn::Result AuthorizePairing::Execute(common::NetworkBuffer& bufferInfo) NN_NOEXCEPT
{
    // 先にネットワーク時計から時刻を取得する
    // (取得に失敗した場合にすぐエラーを返せるようにする)
    nn::time::PosixTime timeStart = nn::time::PosixTime();
    NN_RESULT_THROW_UNLESS(common::GetNetworkTime(&timeStart), nn::pctl::ResultFailedToGetNetworkTime());
    auto tickStart = nn::os::GetSystemTick();

    TokenHolder tokenHolder;
    NN_RESULT_DO(g_pWatcher->GetNetworkManager().AcquireAuthenticationToken(tokenHolder, bufferInfo, m_pCancelable));

    system::Settings settings;
    g_pMain->GetSettingsManager().GetSettings(settings);
    // 未設定だった場合は解除コードを生成して改めて設定値を取得する
    if (settings.current.pinCode[0] == 0)
    {
        g_pMain->GetSettingsManager().GenerateRandomPinCodeAndNoRestriction(settings);
    }
    // ActivateDevice を実行する前にデバイス情報を更新しておく
    g_pWatcher->GetNetworkManager().RefreshDeviceStatus();

    NN_RESULT_DO(
        dispatcher::ActivateDeviceDispatcher::Execute(&m_Etag, bufferInfo, m_pCancelable, tokenHolder.GetToken(), settings, m_DeviceId)
        );

    // 先に経過時間をリセットする(イベントのデータに使われるため、連携済み状態にする前にクリア)
    g_pWatcher->GetWatcherEventManager().ClearSpentTime();

    // 開始時に取得したネットワーク時計の時刻に、tickから算出した経過時間を加える
    auto timeNow = timeStart + (nn::os::GetSystemTick() - tickStart).ToTimeSpan();
    g_pMain->GetSettingsManager().SetSettings(settings);
    g_pWatcher->GetNetworkManager().StoreEtagForSettings(&m_Etag);
    // プレイタイマーはサーバーから取得するまで存在しないのでクリアする
    g_pWatcher->GetWatcherEventManager().ClearPlayTimerSettings();
    // デバイスユーザー情報は初期状態ではすべて送信する必要があるので、
    // プロセス内にキープしているデータをクリアする
    g_pWatcher->GetWatcherEventManager().ClearDeviceUserData();
    // 連携確定時にアップロードしているので更新済みとする
    g_pWatcher->GetNetworkManager().SetLastUpdateDeviceSuccess(true);

    // 連携状態を確定
    g_pWatcher->GetNetworkManager().StorePairingInfo(m_DeviceId);

    // 連携情報設定後に呼び出す
    g_pWatcher->GetWatcherEventManager().ProcessPairingActivated(timeNow);

    NN_RESULT_SUCCESS;
}

void AuthorizePairing::Cancel() NN_NOEXCEPT
{
    if (m_pCancelable != nullptr)
    {
        m_pCancelable->Cancel();
    }
}

}}}}}
