﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/pctl_IntermittentOperation.h>

#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ApplicationStateManager.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/overlay/pctl_OverlaySender.h>
#include <nn/pctl/detail/service/watcher/pctl_DeviceEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_LockGuard.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

namespace
{
    bool g_IsSchedulingNecessary = false;
    nn::os::SdkMutexType g_MutexForSchedulingNecessary = NN_OS_SDK_MUTEX_INITIALIZER();
}

IntermittentContextBase::IntermittentContextBase() NN_NOEXCEPT
{
}

IntermittentContextBase::~IntermittentContextBase() NN_NOEXCEPT
{
    // 保存イベントを処理
    g_pWatcher->GetWatcherEventManager().ProcessSerializeData();

    // 間欠起動終了の宣言
    IntermittentOperation::NotifyTaskFinished();

    NN_DETAIL_PCTL_TRACE("[pctl] IntermittentContextBase: finished\n");
}

//static
nn::Result IntermittentOperation::CreateInstance(IntermittentOperation** ppOutInstance, ServerDeviceId deviceId, common::Cancelable* pCancelable) NN_NOEXCEPT
{
    auto instance = new IntermittentContext<IntermittentOperation, ServerDeviceId, common::Cancelable*>(deviceId, pCancelable);
    NN_RESULT_THROW_UNLESS(instance != nullptr, nn::pctl::ResultOutOfMemory());

    *ppOutInstance = instance;

    NN_RESULT_SUCCESS;
}

IntermittentOperation::~IntermittentOperation() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_TRACE("[pctl] Intermittent operation finished.\n");
}

nn::Result IntermittentOperation::Execute(common::NetworkBuffer& bufferInfo) NN_NOEXCEPT
{
    m_RetrieveParam.pBufferInfo = &bufferInfo;

    // 処理が動いていることを示すためにイベントを追加
    // (何かしら1つ以上のイベントを送信できるようにする)
    g_pWatcher->GetWatcherEventStorage().AddIdleEvent();

    // (設定反映に失敗したら続行しない)
    NN_RESULT_DO(ProcessRetrieveSettings());
    NN_RESULT_DO(ProcessUploadDeviceEvents());

    NN_RESULT_SUCCESS;
}

void IntermittentOperation::Cancel() NN_NOEXCEPT
{
    if (m_RetrieveParam.pCancelable != nullptr)
    {
        m_RetrieveParam.pCancelable->Cancel();
    }
}

nn::Result IntermittentOperation::ProcessRetrieveSettings() NN_NOEXCEPT
{
    m_RetrieveParam.changedTypeFlags.Reset();
    m_RetrieveParam.isNotModified = false;
    // 設定変更なしも検知してハンドルするので Etag を設定
    g_pWatcher->GetNetworkManager().GetEtagForSettings(&m_RetrieveParam.etagInfo);

    NN_RESULT_TRY(RetrieveSettingsExecutor::ExecuteRetrieveAndStoreSettings(m_RetrieveParam))
        // このエラーの場合は連携情報が削除されていることを表すのでローカルデータの削除処理も行う
        NN_RESULT_CATCH(nn::pctl::ResultServerResourceIsNotFound)
        {
            g_pWatcher->GetNetworkManager().HandlePairingDeletedStatus();
            NN_RESULT_THROW(nn::pctl::ResultPairingDeleted());
        }
        // ResultServerResourceIsNotFound と同様のハンドリングをする
        // (ただしResult自体はそのまま返す)
        NN_RESULT_CATCH(nn::pctl::ResultServerResourceNotAvailable)
        {
            g_pWatcher->GetNetworkManager().HandlePairingDeletedStatus();
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    // 成功した場合はバックグラウンド取得待ちを解除
    g_pWatcher->GetWatcherEventManager().ClearWaitingForRetrieveSettingsBackground();

    // 「Not Modified」ではない場合は通知を準備
    if (!m_RetrieveParam.isNotModified)
    {
        g_pWatcher->GetWatcherEventManager().GetSynchronizationEvent()->Signal();
        // 解除コードの変更があればそのイベントも通知する
        if (m_RetrieveParam.changedTypeFlags.Test<nn::ovln::format::PctlSettingTypeFlag::PinCode>())
        {
            g_pWatcher->GetWatcherEventManager().GetPinCodeChangedEvent()->Signal();
        }
        auto r = overlay::NotifySettingChanged(m_RetrieveParam.changedTypeFlags);
        if (r.IsFailure())
        {
            NN_DETAIL_PCTL_INFO("[pctl] Failed to send overlay notification (0x%08lX)\n", r.GetInnerValueForDebug());
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result IntermittentOperation::ProcessUploadDeviceEvents() NN_NOEXCEPT
{
    // 確認できていない中断・再開イベントがある可能性もあるのでチェックを行う
    ApplicationStateManager::RecordPlayDataBasedEvents();

    // (スリープ中の扱い、IDLE等のイベントは発行しない)
    return UploadDeviceEventsExecutor::ExecuteUploadDeviceEvents(
        UploadDeviceEventsExecutor::UploadMode_PostAllImmediately,
        *m_RetrieveParam.pBufferInfo,
        m_RetrieveParam.deviceId,
        m_RetrieveParam.pCancelable,
        &m_RetrieveParam.etagInfo, // (既に有効な値が入っており上書きされるが、結局同じ値に上書きされることになる)
        &m_NotificationTokenBuffer
        );
}

bool IntermittentOperation::IsSchedulingNecessary() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(g_MutexForSchedulingNecessary);
    auto ret = g_IsSchedulingNecessary;
    return ret;
}

void IntermittentOperation::SetSchedulingNecessary(bool value) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(g_MutexForSchedulingNecessary);
    g_IsSchedulingNecessary = value;
}

}}}}}
