﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/pctl_Authentication.h>

#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ServiceConfig.h>
#include <nn/pctl/detail/service/pctl_ServiceMemoryManagement.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/common/pctl_SystemInfo.h>
#include <nn/pctl/detail/service/json/pctl_JsonWebApi.h>
#include <nn/pctl/detail/service/json/pctl_JsonErrorHandler.h>
#include <nn/pctl/detail/service/json/pctl_JsonHttpInputStream.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/time/time_StandardSteadyClock.h>
#include <nn/http/http_Result.h>

#include <nn/os/os_Tick.h>
#include <nn/util/util_Execution.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_IntUtil.h>

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/dauth/dauth_Api.h>
#include <nn/dauth/dauth_Result.h>
#endif

#if defined( NN_BUILD_CONFIG_TOOLCHAIN_GCC )
#define STRTOULL strtoull
#elif defined( NN_BUILD_CONFIG_OS_WIN )
#define STRTOULL _strtoui64
#else
#define STRTOULL std::strtoull
#endif

/*
device_auth_token:

{
  "expires_in" : ${有効期限のUnixtime},
  "device_auth_token" : "${device_auth_token}"
}
*/

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

namespace
{
class CancelListener
    : public common::Cancelable::Listener
{
private:
    common::Cancelable& m_Cancellable;
    nn::util::Cancelable m_CancelableInDauth;
public:
    explicit CancelListener(common::Cancelable& cancellable) NN_NOEXCEPT
        : m_Cancellable(cancellable)
    {
        m_Cancellable.AttachListener(this);
    }
    ~CancelListener() NN_NOEXCEPT
    {
        m_Cancellable.DetachListener();
    }
    virtual void SignalFromPctl() NN_NOEXCEPT NN_OVERRIDE
    {
        m_CancelableInDauth.Cancel();   // pctl側からキャンセルをdauthに伝える
    }
    nn::util::Cancelable* GetCancellable() NN_NOEXCEPT
    {
        return &m_CancelableInDauth;
    }
};
}

nn::Result Authentication::RetrieveDeviceAuthenticationToken(common::NetworkBuffer& bufferInfo, common::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_UNUSED(bufferInfo);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_STATIC_ASSERT(sizeof(m_TokenData.authToken) >= (nn::dauth::RequiredBufferSizeForDeviceAuthenticationToken));

    util::optional<CancelListener> listener;
    if (pCancelable)
    {
        listener.emplace(*pCancelable);
    }

    auto result = nn::dauth::AcquireDeviceAuthenticationToken(
        &m_Expiration,
        &m_AcquiredTokenSize,
        m_TokenData.authToken,
        sizeof(m_TokenData.authToken),
        PctlClientId,
        false,          // true を指定するとキャッシュを無視します。
        listener ? listener->GetCancellable() : nullptr );

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(dauth::ResultCancelled)
        {
            NN_RESULT_THROW(pctl::ResultCanceled());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardSteadyClock::GetCurrentTimePoint(&m_TokenData.tokenRetrieved));
    m_TokenData.tokenExpiresAt = (m_Expiration - nn::os::GetSystemTick().ToTimeSpan()).GetSeconds();
    g_pWatcher->GetNetworkManager().StoreDeviceAuthenticationToken(m_TokenData);

    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(pCancelable);
    NN_RESULT_THROW(pctl::ResultNotImplemented());
#endif
}

}}}}}
