﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/dispatcher/pctl_UpdateDeviceDispatcher.h>

#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ServiceConfig.h>
#include <nn/pctl/detail/service/pctl_ServiceMain.h>
#include <nn/pctl/detail/service/pctl_ServiceMemoryManagement.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/common/pctl_SystemInfo.h>
#include <nn/pctl/detail/service/json/pctl_JsonWebApi.h>
#include <nn/pctl/detail/service/json/pctl_JsonErrorHandler.h>
#include <nn/pctl/detail/service/json/pctl_JsonHttpInputStream.h>
#include <nn/pctl/detail/service/json/pctl_JsonStructuredWriter.h>
#include <nn/pctl/detail/service/watcher/pctl_WatcherErrorHandler.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/time/time_TimeZoneApi.h>

#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher { namespace dispatcher {

namespace
{
    // WriteJsonFormatForDeviceResource / WriteJsonFormatForActivateDevice の両方で利用
    struct WriteDeviceResourceParam
    {
        nn::npns::NotificationToken notificationToken;
        bool hasNotificationToken;
        NN_PADDING3;
        DeviceStatus deviceStatus;
        nn::time::LocationName timeZoneName;
        union
        {
            // WriteJsonFormatForDeviceResource 専用
            struct
            {
                const EtagInfo* etagSynchronized;
                char pinCode[system::MaxPinCodeLength];
            } deviceResource;
            // WriteJsonFormatForActivateDevice 専用
            struct
            {
                nn::time::PosixTime lastOnlineCheckAt;
                const system::Settings* pSettings;
                FreeCommunicationApplicationInfo currentAppInfo;
                char applicationIdText[17];
            } activateDevice;
        };
    };

    static bool WriteNotificationTokenKey(const char** outKeyName, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        if (!p->hasNotificationToken)
        {
            *outKeyName = nullptr;
        }
        else
        {
            *outKeyName = "notificationToken";
        }
        return true;
    };

    static bool WriteNotificationTokenValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        *outValueLength = nn::util::Strnlen(p->notificationToken.data,
            std::extent<decltype(p->notificationToken.data)>::value);
        *outValue = p->notificationToken.data;
        return true;
    };

    static bool WriteTimeZoneValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        *outValueLength = nn::util::Strnlen(p->timeZoneName._value, nn::time::LocationName::Size);
        *outValue = p->timeZoneName._value;
        return true;
    };

    static bool WriteLanguageValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        *outValueLength = nn::util::Strnlen(p->deviceStatus.language, std::extent<decltype(p->deviceStatus.language)>::value);
        *outValue = p->deviceStatus.language;
        return true;
    };

    static bool WriteRegionValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        auto& value = RegionValues[p->deviceStatus.regionCode];
        *outValueLength = value.length;
        *outValue = value.value;
        return true;
    };

    static bool WriteSerialNumberValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        *outValueLength = nn::util::Strnlen(p->deviceStatus.serialNumber.string, std::extent<decltype(p->deviceStatus.serialNumber.string)>::value);
        *outValue = p->deviceStatus.serialNumber.string;
        return true;
    };

    static bool WriteSystemVersionStringValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        *outValueLength = nn::util::Strnlen(p->deviceStatus.systemVersionText, std::extent<decltype(p->deviceStatus.systemVersionText)>::value);
        *outValue = p->deviceStatus.systemVersionText;
        return true;
    };

    static bool WriteSystemVersionUintValue(nn::util::optional<uint64_t>* outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        *outValue = static_cast<uint64_t>(p->deviceStatus.systemVersion);
        return true;
    };

    static bool WriteSynchronizedUnlockCodeKey(const char** outKeyName, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        // Etagが無い場合は synchronizedUnlockCode も書き込まない
        if (p->deviceResource.etagSynchronized == nullptr)
        {
            *outKeyName = nullptr;
        }
        else
        {
            *outKeyName = "synchronizedUnlockCode";
        }
        return true;
    }

    static bool WriteSynchronizedUnlockCodeValueForDeviceResource(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        *outValueLength = nn::util::Strnlen(p->deviceResource.pinCode, std::extent<decltype(p->deviceResource.pinCode)>::value);
        *outValue = p->deviceResource.pinCode;
        return true;
    };

    static bool WriteSynchronizedObjectKey(const char** outKeyName, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        if (p->deviceResource.etagSynchronized == nullptr)
        {
            *outKeyName = nullptr;
        }
        else
        {
            *outKeyName = "synchronizedParentalControlSetting";
        }
        return true;
    }

    static bool WriteSynchronizedEtagValueForDeviceResource(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        *outValueLength = nn::util::Strnlen(p->deviceResource.etagSynchronized->etag, std::extent<decltype(p->deviceResource.etagSynchronized->etag)>::value);
        *outValue = p->deviceResource.etagSynchronized->etag;
        return true;
    };

    static bool WriteSynchronizedAtValueForDeviceResource(nn::util::optional<int64_t>* outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        *outValue = p->deviceResource.etagSynchronized->lastUpdatedAt;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(static const, WriteJsonFormatForDeviceResource)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
            NN_DETAIL_PCTL_JSON_WRITE_KEY_OPTIONAL(WriteNotificationTokenKey) NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteNotificationTokenValue)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("timeZone")                 NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteTimeZoneValue)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("language")                 NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteLanguageValue)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("region")                   NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteRegionValue)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("serialNumber")             NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteSerialNumberValue)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("firmwareVersion")          NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
                NN_DETAIL_PCTL_JSON_WRITE_KEY("displayedVersion")         NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteSystemVersionStringValue)
                NN_DETAIL_PCTL_JSON_WRITE_KEY("internalVersion")          NN_DETAIL_PCTL_JSON_WRITE_VALUE_UINT64(WriteSystemVersionUintValue)
            NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
            NN_DETAIL_PCTL_JSON_WRITE_KEY_OPTIONAL(WriteSynchronizedUnlockCodeKey)    NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteSynchronizedUnlockCodeValueForDeviceResource)
            NN_DETAIL_PCTL_JSON_WRITE_KEY_OPTIONAL(WriteSynchronizedObjectKey)        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
                NN_DETAIL_PCTL_JSON_WRITE_KEY("synchronizedEtag")         NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteSynchronizedEtagValueForDeviceResource)
                NN_DETAIL_PCTL_JSON_WRITE_KEY("synchronizedAt")           NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(WriteSynchronizedAtValueForDeviceResource)
            NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_WRITE()

    ////////////////////////////////////////////////////////////////////////////

    static bool WriteActivatedValue(nn::util::optional<bool>* outValue, void*, int) NN_NOEXCEPT
    {
        *outValue = true;
        return true;
    };

    static bool WriteLastOnlineCheckAtValueForActivate(nn::util::optional<int64_t>* outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        *outValue = p->activateDevice.lastOnlineCheckAt.value;
        return true;
    };

    static bool WriteUnlockCodeValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        *outValueLength = static_cast<size_t>(nn::util::Strnlen(p->activateDevice.pSettings->current.pinCode, system::MaxPinCodeLength));
        *outValue = p->activateDevice.pSettings->current.pinCode;
        return true;
    };

    static bool WriteFunctionalRestrictionLevelValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        auto& value = SafetyLevelValues[p->activateDevice.pSettings->current.safetyLevel];
        *outValueLength = value.length;
        *outValue = value.value;
        return true;
    };

    static bool WriteCustomRatingRestrictionValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        auto& value = RestrictionValues[p->activateDevice.pSettings->custom.ratingAge != 0 ? 1 : 0];
        *outValueLength = value.length;
        *outValue = value.value;
        return true;
    };

    static bool WriteCustomRatingAgeValue(nn::util::optional<int64_t>* outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        auto value = p->activateDevice.pSettings->custom.ratingAge;
        if (value == 0)
        {
            // 制限なしの場合は最低値を指定しておく(使用されないはず)
            value = 3;
        }
        *outValue = static_cast<int64_t>(value);
        return true;
    };

    static bool WriteCustomSnsPostRestrictionValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        auto& value = RestrictionValues[p->activateDevice.pSettings->custom.isSnsPostRestricted ? 1 : 0];
        *outValueLength = value.length;
        *outValue = value.value;
        return true;
    };

    static bool WriteCustomUgcRestrictionValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        auto& value = RestrictionValues[p->activateDevice.pSettings->custom.isFreeCommunicationRestrictedByDefault ? 1 : 0];
        *outValueLength = value.length;
        *outValue = value.value;
        return true;
    };

    static bool WriteCustomRatingOrganizationValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        auto& value = RatingOrganizationValues[static_cast<int>(p->activateDevice.pSettings->current.ratingOrganization)];
        *outValueLength = value.length;
        *outValue = value.value;
        return true;
    };

    static bool WriteApplicationIdValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        *outValueLength = nn::util::Strnlen(p->activateDevice.applicationIdText, 17);
        *outValue = p->activateDevice.applicationIdText;
        return true;
    };

    static bool WriteApplicationRestrictionValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);
        auto& value = RestrictionValues[p->activateDevice.currentAppInfo.isRestricted ? 1 : 0];
        *outValueLength = value.length;
        *outValue = value.value;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(static const, WriteJsonFormatForApplicationsSettings)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("applicationId")         NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteApplicationIdValue)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("restriction")           NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteApplicationRestrictionValue)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_WRITE()

    static bool WriteApplicationsObject(nn::util::optional<bool>* outHasItem, const char** outKeyName, int* outDefinitionLength, const json::WriteDataDefinition** outDefinition, void* param, int index, int itemIndex) NN_NOEXCEPT
    {
        NN_UNUSED(index);

        auto p = reinterpret_cast<WriteDeviceResourceParam*>(param);

        bool hasItem = (itemIndex >= 0 && static_cast<size_t>(itemIndex) < g_pMain->GetSettingsManager().GetFreeCommunicationApplicationListCount());
        if (hasItem)
        {
            // FIXME: 高速化？
            g_pMain->GetSettingsManager().GetFreeCommunicationApplicationList(&p->activateDevice.currentAppInfo, static_cast<size_t>(itemIndex), 1);

            nn::util::SNPrintf(p->activateDevice.applicationIdText, std::extent<decltype(p->activateDevice.applicationIdText)>::value,
                "%016llX", p->activateDevice.currentAppInfo.applicationId);
            *outKeyName = p->activateDevice.applicationIdText;
        }
        *outHasItem = hasItem;
        *outDefinitionLength = static_cast<int>(std::extent<decltype(WriteJsonFormatForApplicationsSettings)>::value);
        *outDefinition = WriteJsonFormatForApplicationsSettings;
        return true;
    }

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(static const, WriteJsonFormatForActivateDevice)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
            NN_DETAIL_PCTL_JSON_WRITE_KEY_OPTIONAL(WriteNotificationTokenKey) NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteNotificationTokenValue)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("timeZone")                 NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteTimeZoneValue)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("language")                 NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteLanguageValue)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("region")                   NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteRegionValue)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("serialNumber")             NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteSerialNumberValue)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("firmwareVersion")          NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
                NN_DETAIL_PCTL_JSON_WRITE_KEY("displayedVersion")         NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteSystemVersionStringValue)
                NN_DETAIL_PCTL_JSON_WRITE_KEY("internalVersion")          NN_DETAIL_PCTL_JSON_WRITE_VALUE_UINT64(WriteSystemVersionUintValue)
            NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
            NN_DETAIL_PCTL_JSON_WRITE_KEY("activated")                NN_DETAIL_PCTL_JSON_WRITE_VALUE_BOOLEAN(WriteActivatedValue)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("lastOnlineCheckedAt")      NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(WriteLastOnlineCheckAtValueForActivate)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("parentalControlSetting")   NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
                NN_DETAIL_PCTL_JSON_WRITE_KEY("unlockCode")                  NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteUnlockCodeValue)
                NN_DETAIL_PCTL_JSON_WRITE_KEY("functionalRestrictionLevel")  NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteFunctionalRestrictionLevelValue)
                NN_DETAIL_PCTL_JSON_WRITE_KEY("customSettings")              NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
                    NN_DETAIL_PCTL_JSON_WRITE_KEY("ageRestriction")          NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
                        NN_DETAIL_PCTL_JSON_WRITE_KEY("restriction")         NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteCustomRatingRestrictionValue)
                        NN_DETAIL_PCTL_JSON_WRITE_KEY("ratingAge")           NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(WriteCustomRatingAgeValue)
                    NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
                    NN_DETAIL_PCTL_JSON_WRITE_KEY("ugcRestriction")          NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
                        NN_DETAIL_PCTL_JSON_WRITE_KEY("restriction")         NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteCustomUgcRestrictionValue)
                        NN_DETAIL_PCTL_JSON_WRITE_KEY("applications")        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_REPEAT(WriteApplicationsObject)
                    NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
                    NN_DETAIL_PCTL_JSON_WRITE_KEY("snsPostRestriction")      NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteCustomSnsPostRestrictionValue)
                    NN_DETAIL_PCTL_JSON_WRITE_KEY("ratingOrganization")      NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteCustomRatingOrganizationValue)
                NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
            NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_WRITE()

    struct RetrieveResultParamForActivateDevice
    {
        EtagInfo* etag;
    };

    static bool HandleSynchronizedEtagFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveResultParamForActivateDevice*>(param);
        if (data->etag != nullptr)
        {
            size_t count = valueLength;
            if (count >= MaxEtagLength)
            {
                NN_DETAIL_PCTL_WARN("Etag is too long (length = %lu)\n", static_cast<uint32_t>(count));
                count = MaxEtagLength - 1;
            }
            std::memcpy(data->etag->etag, value, sizeof(char) * count);
            data->etag->etag[count] = 0;
        }
        return true;
    }

    static bool HandleSynchronizedUpdatedAtValueFunction(void* param, int /*index*/, uint64_t value) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveResultParamForActivateDevice*>(param);
        if (data->etag != nullptr)
        {
            data->etag->lastUpdatedAt = value;
        }
        return true;
    }

    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(static const, ExpectJsonFormatForActivateDevice)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("synchronizedParentalControlSetting") NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
                NN_DETAIL_PCTL_JSON_EXPECT_KEY("synchronizedEtag")   NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleSynchronizedEtagFunction)
                NN_DETAIL_PCTL_JSON_EXPECT_KEY("synchronizedAt")     NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(HandleSynchronizedUpdatedAtValueFunction)
            NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    ////////////////////////////////////////////////////////////////////////////

    struct WriteAlarmSettingParam
    {
        bool isDisabled;
        nn::time::PosixTime invisibleUntil;
    };

    static bool WriteAlarmVisibilityValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteAlarmSettingParam*>(param);
        // disabled == invisible
        auto& value = DeviceAlarmSettingVisibilityName[p->isDisabled ? 1 : 0];
        *outValueLength = value.length;
        *outValue = value.value;
        return true;
    };

    static bool WriteAlarmInvisibleUntilKey(const char** outKeyName, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteAlarmSettingParam*>(param);
        // enabled (visible) の場合は不要なデータ
        if (!p->isDisabled)
        {
            *outKeyName = nullptr;
        }
        else
        {
            *outKeyName = "invisibleUntil";
        }
        return true;
    };

    static bool WriteAlarmInvisibleUntilValue(nn::util::optional<int64_t>* outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteAlarmSettingParam*>(param);
        NN_SDK_ASSERT(p->isDisabled, "Must not be called when isDisabled == false");
        *outValue = p->invisibleUntil.value;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(static const, WriteJsonFormatForAlarmSetting)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("alarmSetting")   NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
                NN_DETAIL_PCTL_JSON_WRITE_KEY("visibility")     NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteAlarmVisibilityValue)
                NN_DETAIL_PCTL_JSON_WRITE_KEY_OPTIONAL(WriteAlarmInvisibleUntilKey)  NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(WriteAlarmInvisibleUntilValue)
            NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_WRITE()

    ////////////////////////////////////////////////////////////////////////////

    struct WriteOnlineCheckParam
    {
        nn::time::PosixTime lastOnlineCheckAt;
    };

    static bool WriteLastOnlineCheckAtValue(nn::util::optional<int64_t>* outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteOnlineCheckParam*>(param);
        *outValue = p->lastOnlineCheckAt.value;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(static const, WriteJsonFormatForOnlineCheck)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("lastOnlineCheckedAt")           NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(WriteLastOnlineCheckAtValue)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_WRITE()

    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(static const, ExpectJsonFormatWithAnyObject)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    struct WriteSynchronizationParam
    {
        const EtagInfo* etagData;
        char pinCode[system::MaxPinCodeLength];
    };

    static bool WriteSynchronizedUnlockCodeValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteSynchronizationParam*>(param);
        *outValueLength = nn::util::Strnlen(p->pinCode, std::extent<decltype(p->pinCode)>::value);
        *outValue = p->pinCode;
        return true;
    };

    static bool WriteSynchronizedEtagValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteSynchronizationParam*>(param);
        *outValueLength = nn::util::Strnlen(p->etagData->etag, std::extent<decltype(p->etagData->etag)>::value);
        *outValue = p->etagData->etag;
        return true;
    };

    static bool WriteSynchronizedAtValue(nn::util::optional<int64_t>* outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WriteSynchronizationParam*>(param);
        *outValue = p->etagData->lastUpdatedAt;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(static const, WriteJsonFormatForSynchronization)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("synchronizedUnlockCode")   NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteSynchronizedUnlockCodeValue)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("synchronizedParentalControlSetting")   NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
                NN_DETAIL_PCTL_JSON_WRITE_KEY("synchronizedEtag")         NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteSynchronizedEtagValue)
                NN_DETAIL_PCTL_JSON_WRITE_KEY("synchronizedAt")           NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(WriteSynchronizedAtValue)
            NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_WRITE()

} // namespace `anonymous'

////////////////////////////////////////////////////////////////////////////////

nn::Result UpdateDeviceDispatcher::Execute(common::NetworkBuffer& bufferInfo, common::Cancelable* pCancelable,
    const char* token, ServerDeviceId deviceId, const EtagInfo* etagSynchronized) NN_NOEXCEPT
{
    WatcherErrorHandler* pErrorHandler = new WatcherErrorHandler();
    NN_RESULT_THROW_UNLESS(pErrorHandler != nullptr, nn::pctl::ResultOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        delete pErrorHandler;
    };

    // tokenヘッダーの作成
    // 22 == strlen("Authorization: Bearer ")
    size_t tokenHeaderLength = std::strlen(token) + 23;
    char* tokenHeader = reinterpret_cast<char*>(AllocateMemoryBlock(sizeof(char) * tokenHeaderLength));
    NN_RESULT_THROW_UNLESS(tokenHeader != nullptr, nn::pctl::ResultHttpErrorOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        FreeMemoryBlock(tokenHeader);
    };
    nn::util::SNPrintf(tokenHeader, tokenHeaderLength, "Authorization: Bearer %s", token);

    // URLの作成
    char url[UrlBufferLength_UpdateDevice];

    nn::util::SNPrintf(url, std::extent<decltype(url)>::value, UrlFormat_UpdateDevice,
        ServerEndpoint, deviceId);

    // システムデータの取得
    auto& manager = g_pWatcher->GetNetworkManager();
    WriteDeviceResourceParam param;
    param.hasNotificationToken = manager.IsNotificationTokenRetrieved();
    if (param.hasNotificationToken)
    {
        manager.GetNotificationToken(param.notificationToken);
    }
    manager.GetDeviceStatus(param.deviceStatus);
    // タイムゾーン
    nn::time::GetDeviceLocationName(&param.timeZoneName);
    // 解除コード
    if (etagSynchronized != nullptr && etagSynchronized->etag[0] == 0)
    {
        etagSynchronized = nullptr;
    }
    if (etagSynchronized != nullptr) // Etagが無い場合は pinCode は使用しない
    {
        nn::pctl::detail::service::system::Settings settings;
        g_pMain->GetSettingsManager().GetSettings(settings);
        std::memcpy(param.deviceResource.pinCode, settings.current.pinCode, sizeof(param.deviceResource.pinCode));
    }
    param.deviceResource.etagSynchronized = etagSynchronized;

    json::JsonStructuredWriter writer(&param, WriteJsonFormatForDeviceResource);

    nn::pctl::detail::service::json::JsonHttpInputStream stream;
    json::JsonDataHandler handler(nullptr, ExpectJsonFormatWithAnyObject);

    NN_RESULT_DO(stream.GetRequest().Open("POST", url));
    NN_RESULT_DO(stream.GetRequest().AddRequestHeader("Content-Type: application/json"));
    NN_RESULT_DO(stream.GetRequest().AddRequestHeader(tokenHeader));

    NN_RESULT_DO(
        json::ParseWebStreamWithPostJSON(&writer, &handler, pErrorHandler, &stream,
            bufferInfo.GetBufferForJsonValue(), common::NetworkBuffer::JsonValueBufferMemorySize,
            bufferInfo.GetBufferForStream(), common::NetworkBuffer::StreamBufferMemorySize,
            pCancelable)
        );

    NN_RESULT_SUCCESS;
}

////////////////////////////////////////////////////////////////////////////////

nn::Result ActivateDeviceDispatcher::Execute(EtagInfo* outEtagSynchronized, common::NetworkBuffer& bufferInfo, common::Cancelable* pCancelable,
    const char* token, const system::Settings& settings, ServerDeviceId deviceId) NN_NOEXCEPT
{
    WatcherErrorHandler* pErrorHandler = new WatcherErrorHandler();
    NN_RESULT_THROW_UNLESS(pErrorHandler != nullptr, nn::pctl::ResultOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        delete pErrorHandler;
    };

    // tokenヘッダーの作成
    // 22 == strlen("Authorization: Bearer ")
    size_t tokenHeaderLength = std::strlen(token) + 23;
    char* tokenHeader = reinterpret_cast<char*>(AllocateMemoryBlock(sizeof(char) * tokenHeaderLength));
    NN_RESULT_THROW_UNLESS(tokenHeader != nullptr, nn::pctl::ResultHttpErrorOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        FreeMemoryBlock(tokenHeader);
    };
    nn::util::SNPrintf(tokenHeader, tokenHeaderLength, "Authorization: Bearer %s", token);

    // URLの作成
    char url[UrlBufferLength_ActivateDevice];

    nn::util::SNPrintf(url, std::extent<decltype(url)>::value, UrlFormat_ActivateDevice,
        ServerEndpoint, deviceId);

    // システムデータの取得
    auto& manager = g_pWatcher->GetNetworkManager();
    WriteDeviceResourceParam param;
    param.hasNotificationToken = manager.IsNotificationTokenRetrieved();
    if (param.hasNotificationToken)
    {
        manager.GetNotificationToken(param.notificationToken);
    }
    manager.GetDeviceStatus(param.deviceStatus);
    // タイムゾーン
    nn::time::GetDeviceLocationName(&param.timeZoneName);
    // 時刻
    NN_RESULT_THROW_UNLESS(common::GetNetworkTime(&param.activateDevice.lastOnlineCheckAt),
        nn::pctl::ResultInvalidState());
    // ペアコン設定
    param.activateDevice.pSettings = &settings;

    json::JsonStructuredWriter writer(&param, WriteJsonFormatForActivateDevice);

    RetrieveResultParamForActivateDevice resultParam;
    resultParam.etag = outEtagSynchronized;
    resultParam.etag->etag[0] = 0;

    nn::pctl::detail::service::json::JsonHttpInputStream stream;
    json::JsonDataHandler handler(&resultParam, ExpectJsonFormatForActivateDevice);

    NN_RESULT_DO(stream.GetRequest().Open("POST", url));
    NN_RESULT_DO(stream.GetRequest().AddRequestHeader("Content-Type: application/json"));
    NN_RESULT_DO(stream.GetRequest().AddRequestHeader(tokenHeader));

    NN_RESULT_DO(
        json::ParseWebStreamWithPostJSON(&writer, &handler, pErrorHandler, &stream,
            bufferInfo.GetBufferForJsonValue(), common::NetworkBuffer::JsonValueBufferMemorySize,
            bufferInfo.GetBufferForStream(), common::NetworkBuffer::StreamBufferMemorySize,
            pCancelable)
        );

    NN_RESULT_SUCCESS;
}

////////////////////////////////////////////////////////////////////////////////

nn::Result OnlineCheckDispatcher::Execute(common::NetworkBuffer& bufferInfo, common::Cancelable* pCancelable, const char* token, ServerDeviceId deviceId) NN_NOEXCEPT
{
    WatcherErrorHandler* pErrorHandler = new WatcherErrorHandler();
    NN_RESULT_THROW_UNLESS(pErrorHandler != nullptr, nn::pctl::ResultOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        delete pErrorHandler;
    };

    // tokenヘッダーの作成
    // 22 == strlen("Authorization: Bearer ")
    size_t tokenHeaderLength = std::strlen(token) + 23;
    char* tokenHeader = reinterpret_cast<char*>(AllocateMemoryBlock(sizeof(char) * tokenHeaderLength));
    NN_RESULT_THROW_UNLESS(tokenHeader != nullptr, nn::pctl::ResultHttpErrorOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        FreeMemoryBlock(tokenHeader);
    };
    nn::util::SNPrintf(tokenHeader, tokenHeaderLength, "Authorization: Bearer %s", token);

    // URLの作成
    char url[UrlBufferLength_OnlineCheck];

    nn::util::SNPrintf(url, std::extent<decltype(url)>::value, UrlFormat_OnlineCheck,
        ServerEndpoint, deviceId);

    WriteOnlineCheckParam param;
    NN_RESULT_THROW_UNLESS(common::GetNetworkTime(&param.lastOnlineCheckAt),
        nn::pctl::ResultInvalidState());

    json::JsonStructuredWriter writer(&param, WriteJsonFormatForOnlineCheck);

    nn::pctl::detail::service::json::JsonHttpInputStream stream;
    json::JsonDataHandler handler(nullptr, ExpectJsonFormatWithAnyObject);

    NN_RESULT_DO(stream.GetRequest().Open("POST", url));
    NN_RESULT_DO(stream.GetRequest().AddRequestHeader("Content-Type: application/json"));
    NN_RESULT_DO(stream.GetRequest().AddRequestHeader(tokenHeader));

    NN_RESULT_DO(
        json::ParseWebStreamWithPostJSON(&writer, &handler, pErrorHandler, &stream,
            bufferInfo.GetBufferForJsonValue(), common::NetworkBuffer::JsonValueBufferMemorySize,
            bufferInfo.GetBufferForStream(), common::NetworkBuffer::StreamBufferMemorySize,
            pCancelable)
        );

    NN_RESULT_SUCCESS;
}

////////////////////////////////////////////////////////////////////////////////

nn::Result UpdateDeviceSynchronizationDispatcher::Execute(common::NetworkBuffer& bufferInfo, common::Cancelable* pCancelable, const char* token,
    ServerDeviceId deviceId, const char* pinCode, const EtagInfo* etagSynchronized) NN_NOEXCEPT
{
    WatcherErrorHandler* pErrorHandler = new WatcherErrorHandler();
    NN_RESULT_THROW_UNLESS(pErrorHandler != nullptr, nn::pctl::ResultOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        delete pErrorHandler;
    };

    // tokenヘッダーの作成
    // 22 == strlen("Authorization: Bearer ")
    size_t tokenHeaderLength = std::strlen(token) + 23;
    char* tokenHeader = reinterpret_cast<char*>(AllocateMemoryBlock(sizeof(char) * tokenHeaderLength));
    NN_RESULT_THROW_UNLESS(tokenHeader != nullptr, nn::pctl::ResultHttpErrorOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        FreeMemoryBlock(tokenHeader);
    };
    nn::util::SNPrintf(tokenHeader, tokenHeaderLength, "Authorization: Bearer %s", token);

    // URLの作成
    char url[UrlBufferLength_UpdateDevice];

    nn::util::SNPrintf(url, std::extent<decltype(url)>::value, UrlFormat_UpdateDevice,
        ServerEndpoint, deviceId);

    WriteSynchronizationParam param;
    param.etagData = etagSynchronized;
    std::memcpy(param.pinCode, pinCode, sizeof(param.pinCode));

    json::JsonStructuredWriter writer(&param, WriteJsonFormatForSynchronization);

    nn::pctl::detail::service::json::JsonHttpInputStream stream;
    json::JsonDataHandler handler(nullptr, ExpectJsonFormatWithAnyObject);

    NN_RESULT_DO(stream.GetRequest().Open("POST", url));
    NN_RESULT_DO(stream.GetRequest().AddRequestHeader("Content-Type: application/json"));
    NN_RESULT_DO(stream.GetRequest().AddRequestHeader(tokenHeader));

    NN_RESULT_DO(
        json::ParseWebStreamWithPostJSON(&writer, &handler, pErrorHandler, &stream,
            bufferInfo.GetBufferForJsonValue(), common::NetworkBuffer::JsonValueBufferMemorySize,
            bufferInfo.GetBufferForStream(), common::NetworkBuffer::StreamBufferMemorySize,
            pCancelable)
        );

    NN_RESULT_SUCCESS;
}

////////////////////////////////////////////////////////////////////////////////

nn::Result UpdateDeviceAlarmSettingDispatcher::Execute(common::NetworkBuffer& bufferInfo, common::Cancelable* pCancelable, const char* token,
    ServerDeviceId deviceId, bool isDisabled, const nn::time::PosixTime& timeInvisibleUntil) NN_NOEXCEPT
{
    WatcherErrorHandler* pErrorHandler = new WatcherErrorHandler();
    NN_RESULT_THROW_UNLESS(pErrorHandler != nullptr, nn::pctl::ResultOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        delete pErrorHandler;
    };

    // tokenヘッダーの作成
    // 22 == strlen("Authorization: Bearer ")
    size_t tokenHeaderLength = std::strlen(token) + 23;
    char* tokenHeader = reinterpret_cast<char*>(AllocateMemoryBlock(sizeof(char) * tokenHeaderLength));
    NN_RESULT_THROW_UNLESS(tokenHeader != nullptr, nn::pctl::ResultHttpErrorOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        FreeMemoryBlock(tokenHeader);
    };
    nn::util::SNPrintf(tokenHeader, tokenHeaderLength, "Authorization: Bearer %s", token);

    // URLの作成
    char url[UrlBufferLength_UpdateDevice];

    // (POST先はUpdateDevice)
    nn::util::SNPrintf(url, std::extent<decltype(url)>::value, UrlFormat_UpdateDevice,
        ServerEndpoint, deviceId);

    WriteAlarmSettingParam param;
    param.isDisabled = isDisabled;
    param.invisibleUntil = timeInvisibleUntil;

    json::JsonStructuredWriter writer(&param, WriteJsonFormatForAlarmSetting);

    nn::pctl::detail::service::json::JsonHttpInputStream stream;
    json::JsonDataHandler handler(nullptr, ExpectJsonFormatWithAnyObject);

    NN_RESULT_DO(stream.GetRequest().Open("POST", url));
    NN_RESULT_DO(stream.GetRequest().AddRequestHeader("Content-Type: application/json"));
    NN_RESULT_DO(stream.GetRequest().AddRequestHeader(tokenHeader));

    NN_RESULT_DO(
        json::ParseWebStreamWithPostJSON(&writer, &handler, pErrorHandler, &stream,
            bufferInfo.GetBufferForJsonValue(), common::NetworkBuffer::JsonValueBufferMemorySize,
            bufferInfo.GetBufferForStream(), common::NetworkBuffer::StreamBufferMemorySize,
            pCancelable)
        );

    NN_RESULT_SUCCESS;
}

}}}}}}
