﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/dispatcher/pctl_DeviceUserDispatcher.h>

#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ServiceMemoryManagement.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/common/pctl_FileSystem.h>
#include <nn/pctl/detail/service/common/pctl_HttpRequest.h>
#include <nn/pctl/detail/service/json/pctl_JsonWebApi.h>
#include <nn/pctl/detail/service/json/pctl_JsonErrorHandler.h>
#include <nn/pctl/detail/service/json/pctl_JsonHttpInputStream.h>
#include <nn/pctl/detail/service/json/pctl_JsonStructuredWriter.h>
#include <nn/pctl/detail/service/watcher/pctl_Authentication.h>
#include <nn/pctl/detail/service/watcher/pctl_WatcherErrorHandler.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_FormatString.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher { namespace dispatcher {

namespace
{
    static const size_t MaxBase64DataLengthWithNull = 192 * 1024;
    static const size_t CacheBufferSize = MaxBase64DataLengthWithNull + 1024;

    int UidToStringInternal(char* outString, int maxLengthWithNull, const nn::account::Uid& uid) NN_NOEXCEPT
    {
        if (maxLengthWithNull <= 0)
        {
            return 0;
        }
        // NOTE: pctlプロセスと関連サーバー・アプリ内部でのみ使用する形式
        // (形式自体はDevMenuで表示している形式と同じ)
        static const char DummyUidString[] = "00000000_00000000_00000000_00000000";
        NN_STATIC_ASSERT((std::extent<decltype(DummyUidString)>::value) <= MaxUidStringLength);
        return nn::util::SNPrintf(
            outString, static_cast<size_t>(maxLengthWithNull), "%08x_%08x_%08x_%08x",
            static_cast<uint32_t>(uid._data[0] >> 32),
            static_cast<uint32_t>(uid._data[0] & 0xFFFFFFFFull),
            static_cast<uint32_t>(uid._data[1] >> 32),
            static_cast<uint32_t>(uid._data[1] & 0xFFFFFFFFull));
    }

    nn::Result GetImageDataAsBase64Encoded(char* outBuffer, size_t maxBufferLengthWithNull,
        const nn::account::Uid& uid, void* tempBuffer, size_t maxTempBufferSize) NN_NOEXCEPT
    {
        size_t size = 0;
        NN_RESULT_DO(nn::account::LoadProfileImage(&size, tempBuffer, maxTempBufferSize, uid));
        std::memset(outBuffer, 0, maxBufferLengthWithNull);
        auto status = nn::util::Base64::ToBase64String(outBuffer, maxBufferLengthWithNull,
            tempBuffer, size, nn::util::Base64::Mode_NormalNoLinefeed);
        NN_RESULT_THROW_UNLESS(status == nn::util::Base64::Status_Success, nn::pctl::ResultUnexpected());
        NN_RESULT_SUCCESS;
    }


    struct PutDevicePlayerData
    {
        nn::account::Nickname nickname;
        char* base64DataBuffer;
    };

    static bool WriteNicknameValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<PutDevicePlayerData*>(param);
        *outValueLength = nn::util::Strnlen(p->nickname.name, std::extent<decltype(p->nickname.name)>::value);
        *outValue = p->nickname.name;
        return true;
    };

    static bool WriteImageDataValue(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<PutDevicePlayerData*>(param);
        *outValueLength = nn::util::Strnlen(p->base64DataBuffer, MaxBase64DataLengthWithNull);
        *outValue = p->base64DataBuffer;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(static const, WriteJsonFormatForPutDevicePlayer)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("nickname")   NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteNicknameValue)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("imageData")  NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteImageDataValue)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_WRITE()

    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(static const, ExpectJsonFormatWithAnyObject)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()
}

////////////////////////////////////////////////////////////////////////////////

int UidToString(char* outString, int maxLengthWithNull, const nn::account::Uid& uid) NN_NOEXCEPT
{
    return UidToStringInternal(outString, maxLengthWithNull, uid);
}

////////////////////////////////////////////////////////////////////////////////

nn::Result DeviceUserDispatcher::SendPutUser(common::NetworkBuffer& bufferInfo, common::Cancelable* pCancelable,
    const char* token, ServerDeviceId deviceId, const nn::account::Uid& uid) NN_NOEXCEPT
{
    WatcherErrorHandler* pErrorHandler = new WatcherErrorHandler();
    NN_RESULT_THROW_UNLESS(pErrorHandler != nullptr, nn::pctl::ResultOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        delete pErrorHandler;
    };

    // tokenヘッダーの作成
    // 22 == strlen("Authorization: Bearer ")
    size_t tokenHeaderLength = std::strlen(token) + 23;
    char* tokenHeader = reinterpret_cast<char*>(AllocateMemoryBlock(sizeof(char) * tokenHeaderLength));
    NN_RESULT_THROW_UNLESS(tokenHeader != nullptr, nn::pctl::ResultHttpErrorOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        FreeMemoryBlock(tokenHeader);
    };
    nn::util::SNPrintf(tokenHeader, tokenHeaderLength, "Authorization: Bearer %s", token);

    // URLの作成
    char url[UrlBufferLength_DevicePlayers + MaxUidStringLength];

    {
        int r = nn::util::SNPrintf(url, std::extent<decltype(url)>::value, UrlFormat_DevicePlayers,
            ServerEndpoint, deviceId);
        NN_SDK_ASSERT_EQUAL(r, static_cast<int>(UrlBufferLength_DevicePlayers - 1)); // '-1': NULL文字
        UidToStringInternal(url + r, static_cast<int>(std::extent<decltype(url)>::value) - r, uid);
    }

    PutDevicePlayerData param;
    NN_RESULT_TRY(nn::account::GetNickname(&param.nickname, uid))
        NN_RESULT_CATCH(nn::account::ResultUserNotExist)
        {
            // 見つからなかった場合は何もせず終了させる
            // (このメソッドが呼び出されているのは uid のユーザーが存在していたためであるので
            // 次のチェック時に状態が更新されるはず)
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY

    param.base64DataBuffer = static_cast<char*>(AllocateMemoryBlock(MaxBase64DataLengthWithNull));
    NN_RESULT_THROW_UNLESS(param.base64DataBuffer != nullptr, nn::pctl::ResultOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        FreeMemoryBlock(param.base64DataBuffer);
    };

    void* cacheBuffer = static_cast<char*>(AllocateMemoryBlock(CacheBufferSize));
    NN_RESULT_THROW_UNLESS(cacheBuffer != nullptr, nn::pctl::ResultOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        FreeMemoryBlock(cacheBuffer);
    };

    // キャッシュバッファーを一時バッファーとして用いる
    // (JSON 出力前なのでまだ使ってない)
    NN_RESULT_TRY(GetImageDataAsBase64Encoded(
        param.base64DataBuffer, MaxBase64DataLengthWithNull,
        uid, cacheBuffer, CacheBufferSize
    ))
        NN_RESULT_CATCH(nn::account::ResultUserNotExist)
        {
            // 見つからなかった場合は何もせず終了させる
            // (このメソッドが呼び出されているのは uid のユーザーが存在していたためであるので
            // 次のチェック時に状態が更新されるはず)
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY

    json::JsonStructuredWriter writer(&param, WriteJsonFormatForPutDevicePlayer);
    writer.SetCacheBuffer(cacheBuffer, CacheBufferSize);

    nn::pctl::detail::service::json::JsonHttpInputStream stream;
    json::JsonDataHandler handler(nullptr, ExpectJsonFormatWithAnyObject);

    NN_RESULT_DO(stream.GetRequest().Open("PUT", url));
    NN_RESULT_DO(stream.GetRequest().AddRequestHeader("Content-Type: application/json"));
    NN_RESULT_DO(stream.GetRequest().AddRequestHeader(tokenHeader));

    NN_RESULT_DO(
        json::ParseWebStreamWithPostJSON(&writer, &handler, pErrorHandler, &stream,
            bufferInfo.GetBufferForJsonValue(), common::NetworkBuffer::JsonValueBufferMemorySize,
            bufferInfo.GetBufferForStream(), common::NetworkBuffer::StreamBufferMemorySize,
            pCancelable)
        );

    NN_RESULT_SUCCESS;
}

nn::Result DeviceUserDispatcher::SendDeleteUser(common::NetworkBuffer& bufferInfo, common::Cancelable* pCancelable,
    const char* token, ServerDeviceId deviceId, const nn::account::Uid& uid) NN_NOEXCEPT
{
    // tokenヘッダーの作成
    // 22 == strlen("Authorization: Bearer ")
    size_t tokenHeaderLength = std::strlen(token) + 23;
    char* tokenHeader = reinterpret_cast<char*>(AllocateMemoryBlock(sizeof(char) * tokenHeaderLength));
    NN_RESULT_THROW_UNLESS(tokenHeader != nullptr, nn::pctl::ResultHttpErrorOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        FreeMemoryBlock(tokenHeader);
    };
    nn::util::SNPrintf(tokenHeader, tokenHeaderLength, "Authorization: Bearer %s", token);

    // URLの作成
    char url[UrlBufferLength_DevicePlayers + MaxUidStringLength];

    {
        int r = nn::util::SNPrintf(url, std::extent<decltype(url)>::value, UrlFormat_DevicePlayers,
            ServerEndpoint, deviceId);
        NN_SDK_ASSERT_EQUAL(r, static_cast<int>(UrlBufferLength_DevicePlayers - 1)); // '-1': NULL文字
        UidToStringInternal(url + r, static_cast<int>(std::extent<decltype(url)>::value) - r, uid);
    }

    // 200 でレスポンスもあるが無視できる
    nn::pctl::detail::service::common::HttpRequest request;

    NN_RESULT_DO(request.Open("DELETE", url));
    NN_RESULT_DO(request.AddRequestHeader(tokenHeader));

    NN_RESULT_TRY(common::PerformRequest(&request, bufferInfo.GetEntireBuffer(), common::NetworkBuffer::EntireBufferMemorySize, pCancelable))
        NN_RESULT_CATCH(nn::pctl::ResultUnexpectedServerError404)
    {
        // 「見つからない」は成功とみなせる
    }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

}}}}}}
