﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/ipc/pctl_IpcConfig.h>
#include <nn/pctl/detail/service/pctl_ParentalControlServiceImpl.h>
#include <nn/pctl/detail/service/pctl_ServiceMain.h>
#include <nn/pctl/detail/service/pctl_ServiceMemoryManagement.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/common/pctl_Constants.h>
#include <nn/pctl/detail/service/common/pctl_SystemInfo.h>
#include <nn/pctl/pctl_TypesSystem.h>
#include <nn/pctl/pctl_ResultPrivate.h>

#include <nn/arp/arp_Api.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/os/os_Random.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace pctl { namespace detail { namespace service {

namespace
{
    static const size_t MaxServerDeviceOwnerCount = 5;

#if NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS == NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS_HIPC
    // nn::ns::ApplicationControlProperty はサイズが大きいのでグローバル変数に置く
    // (使用期間が短いので使用時は Mutex で排他制御を行う)
    nn::ns::ApplicationControlProperty g_AppControlProperty;
#endif

#if NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS == NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS_HIPC
    // 現在セッションを張っているプロセスと capability の対応
    // (同時利用できるプロセスの数だけで十分)
    struct
    {
        nn::Bit64 processId;
        ParentalControlServiceImpl* pImpl;
    } g_Sessions[ipc::ProcessCountMax] = {};

    ParentalControlServiceImpl* FindServiceInstanceFromProcessId(nn::Bit64 processId) NN_NOEXCEPT
    {
        for (auto& s : g_Sessions)
        {
            if (s.processId == processId)
            {
                return s.pImpl;
            }
        }
        return nullptr;
    }

    bool AddToSessionList(nn::Bit64 processId, ParentalControlServiceImpl* pImpl) NN_NOEXCEPT
    {
        for (auto& s : g_Sessions)
        {
            if (s.processId == processId)
            {
                return false;
            }
        }
        for (auto& s : g_Sessions)
        {
            if (s.pImpl == nullptr)
            {
                s.processId = processId;
                s.pImpl = pImpl;
                return true;
            }
        }
        return false;
    }

    void RemoveFromSessionList(ParentalControlServiceImpl* pImpl) NN_NOEXCEPT
    {
        for (auto& s : g_Sessions)
        {
            if (s.pImpl == pImpl)
            {
                s.processId = 0;
                s.pImpl = nullptr;
                break;
            }
        }
    }
#endif

    inline bool IsCapabilityToIgnoreEvents(int capability) NN_NOEXCEPT
    {
        return (capability & (ipc::Capability_System | ipc::Capability_Recovery)) != 0;
    }

    nn::Result WaitAsyncContext(common::AsyncContext* pContext) NN_NOEXCEPT
    {
        nn::Result result;
        while (!pContext->IsFinished(&result))
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
        }
        pContext->CloseContext();
        return result;
    }

    void CancelAndCloseAsyncContext(common::AsyncContext* pContext) NN_NOEXCEPT
    {
        pContext->Cancel();
        // キャンセル終了待ち
        int retryCount = 400; // 最大で 400 * 50 = 20000 msec. 待機する
        while (retryCount > 0)
        {
            if (pContext->IsFinished())
            {
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            --retryCount;
        }
        if (retryCount == 0)
        {
            // IsFinished() == false の場合は安全に CloseContext できない
            // (そもそも Cancel しているのに終わらないので想定外である可能性もある)
            NN_DETAIL_PCTL_WARN("[pctl] Still running async context; may block next execution\n");
        }
        else
        {
            pContext->CloseContext();
        }
    }
}

ParentalControlServiceImpl::ParentalControlServiceImpl(int capability) NN_NOEXCEPT :
    m_capability(capability)
{
    std::memset(&m_TargetApplicationInfo, 0, sizeof(m_TargetApplicationInfo));
    for (auto& d : m_AsyncDataInProgress.asyncContextInProgressArray)
    {
        d.id = 0;
        d.pContext = nullptr;
        d.pDataHolder = nullptr;
    }
    m_TargetApplicationInfo.applicationId = nn::ncm::ApplicationId::GetInvalidId();
}

nn::Result ParentalControlServiceImpl::InternalInitialize(nn::Bit64 processId) NN_NOEXCEPT
{
#if NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS == NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS_HIPC
    NN_DETAIL_PCTL_TRACE("[pctl] InternalInitialize: processId = %llu, this = 0x%p\n", processId, this);
    auto p = FindServiceInstanceFromProcessId(processId);
    if (p != nullptr)
    {
        // マージする
        NN_DETAIL_PCTL_TRACE("[pctl] Merge capability: %d <-- %d, %d\n",
            (m_capability | p->m_capability), m_capability, p->m_capability);
        p->m_capability |= m_capability;
        m_capability = p->m_capability;
    }
    else
    {
        // 存在しない場合はリストに追加
        NN_RESULT_THROW_UNLESS(AddToSessionList(processId, this), nn::pctl::ResultOutOfSessionResource());
    }
#endif

    m_TargetApplicationInfo.processId = processId;
    m_TargetApplicationInfo.capability = m_capability;
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::Initialize() NN_NOEXCEPT
{
    // Application と System のどちらも持っていない場合は許可しない
    // (pctl, pctl:s, pctl:a, pctl:r のいずれかがあれば持っているはず → pctl_IpcConfig.h で static assert チェック)
    if ((m_capability & (detail::ipc::Capability_Application | detail::ipc::Capability_System)) == 0)
    {
        NN_DETAIL_PCTL_WARN("[pctl] Unexpected capability: the caller only has partial permission(s).\n");
        NN_RESULT_THROW(nn::pctl::ResultNotPermitted());
    }

#if NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS == NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS_DFC
    // DFC(同一プロセス)版の場合は特に何もしない
#elif NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS == NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS_HIPC
    if (m_TargetApplicationInfo.processId == 0)
    {
        NN_DETAIL_PCTL_WARN("[pctl] Unexpected process id\n");
        NN_RESULT_THROW(nn::pctl::ResultInvalidValue());
    }

    // Recovery権限を持っている場合はアプリ設定のチェックや
    // 起動イベントの生成などを行わない
    if ((m_capability & detail::ipc::Capability_Recovery) == 0)
    {
        nn::arp::ApplicationLaunchProperty launchProperty;   // ApplicationId 取得用
        nn::os::ProcessId pid = { m_TargetApplicationInfo.processId };

        auto result = nn::arp::GetApplicationLaunchProperty(&launchProperty, pid);
        if (result.IsSuccess())
        {
            NN_DETAIL_PCTL_TRACE("[pctl] nn::arp::GetApplicationLaunchProperty succeeded: app-id = 0x%016llX\n",
                launchProperty.id.value);
            m_TargetApplicationInfo.applicationId = launchProperty.id;

            // g_AppControlProperty を利用
            NN_FUNCTION_LOCAL_STATIC(nn::os::SdkMutexType, s_MutexForAppControlProperty, =NN_OS_SDK_MUTEX_INITIALIZER());
            NN_UTIL_LOCK_GUARD(s_MutexForAppControlProperty);
            result = nn::arp::GetApplicationControlProperty(&g_AppControlProperty, pid);
            if (result.IsSuccess())
            {
                std::memcpy(m_TargetApplicationInfo.ratingAge, g_AppControlProperty.ratingAge, sizeof(m_TargetApplicationInfo.ratingAge));
                m_TargetApplicationInfo.parentalControlFlag = g_AppControlProperty.parentalControlFlag;
            }
        }
        else
        {
            // ApplicationId が取れなかった場合でも特に問題にはしない
            // (「取得できるかどうか」と「通常のアプリケーションかどうか」(＝「ここで記録したいアプリケーションかどうか」)は同じ)
            NN_DETAIL_PCTL_INFO("[pctl] nn::arp::GetApplicationLaunchProperty returns 0x%08X (ignored)\n", result.GetInnerValueForDebug());
        }
    }
#endif

    // アプリケーションIDが取得できていれば ApplicationStateManager に処理を投げる
    // (実際に起動終了を監視するかどうかは ApplicationStateManager が capability の値などをチェックして判断)
    if (m_TargetApplicationInfo.applicationId != nn::ncm::ApplicationId::GetInvalidId())
    {
        ApplicationStateManager::SetCurrentApplicationInfo(m_TargetApplicationInfo);

        if (!IsCapabilityToIgnoreEvents(m_capability))
        {
            g_pMain->GetSettingsManager().PostApplicationLaunched(
                m_TargetApplicationInfo.applicationId,
                (m_TargetApplicationInfo.parentalControlFlag & static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication)) != 0
            );
        }
    }
    NN_RESULT_SUCCESS;
}

ParentalControlServiceImpl::~ParentalControlServiceImpl() NN_NOEXCEPT
{
    // アプリケーション情報を取得できていなければ終了イベントの生成などを行わない
    if (m_TargetApplicationInfo.applicationId != nn::ncm::ApplicationId::GetInvalidId())
    {
        ApplicationStateManager::ClearCurrentApplicationInfo(m_TargetApplicationInfo);

        nn::time::PosixTime timestamp;
        if (common::GetNetworkTime(&timestamp))
        {
            g_pMain->GetSettingsManager().PostApplicationTerminated(m_TargetApplicationInfo.applicationId, timestamp);
        }
        else
        {
            g_pMain->GetSettingsManager().PostApplicationTerminated(m_TargetApplicationInfo.applicationId, nn::util::nullopt);
        }
    }

    for (auto& d : m_AsyncDataInProgress.asyncContextInProgressArray)
    {
        if (d.pContext != nullptr)
        {
            if (d.pDataHolder != nullptr)
            {
                delete d.pDataHolder;
            }
            CancelAndCloseAsyncContext(d.pContext);
        }
    }

#if NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS == NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS_HIPC
    NN_DETAIL_PCTL_TRACE("[pctl] Release session: this = 0x%p\n", this);
    RemoveFromSessionList(this);
#endif
}

// 本体機能向けペアコン状態チェック関数

nn::Result ParentalControlServiceImpl::CheckFreeCommunicationPermission() NN_NOEXCEPT
{
    // 特に権限チェックを行わない
    //NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_General);

    auto& states = g_pMain->GetStatesRef();
    NN_RESULT_DO(
        g_pMain->GetSettingsManager().ConfirmFreeCommunicationRestriction(
            states,
            m_TargetApplicationInfo.applicationId,
            (m_TargetApplicationInfo.parentalControlFlag & static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication)) != 0
            )
        );

    // 設定状態にかかわらず通過した状態を持たせる
    // (アプリを中断して再開したときの判定に用いるため)
    states.freeCommunicationUsed = true;
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::ConfirmLaunchApplicationPermission(nn::ncm::ApplicationId applicationId, const nn::sf::InArray<std::int8_t>& ratingAge, bool freeCommunicationFlag) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_RatingCheck);

    // 一旦値をリセットする
    auto& states = g_pMain->GetStatesRef();
    states.nextApplicationId = nn::ncm::ApplicationId::GetInvalidId();

    NN_RESULT_TRY(g_pMain->GetSettingsManager().ConfirmLaunchApplicationPermission(
        states,
        applicationId,
        ratingAge.GetData(),
        ratingAge.GetLength(),
        freeCommunicationFlag))
        NN_RESULT_CATCH(nn::pctl::ResultRestrictedByRating)
        {
            g_pWatcher->GetWatcherEventStorage().AddApplicationRejectedEvent(applicationId);
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    // これから起動しようとしているアプリのIDを持たせる
    states.nextApplicationId = applicationId;

#if NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS == NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS_DFC
    // 同一プロセス版ではここへの到達をもってアプリ起動済みと判断
    m_TargetApplicationInfo.processId = 0;
    m_TargetApplicationInfo.applicationId = applicationId;
    std::memcpy(m_TargetApplicationInfo.ratingAge, ratingAge.GetData(), sizeof(m_TargetApplicationInfo.ratingAge));
    m_TargetApplicationInfo.parentalControlFlag = (freeCommunicationFlag ? static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication) : 0);
    std::memcpy(&states.currentApplicationInfo, &m_TargetApplicationInfo, sizeof(m_TargetApplicationInfo));
#endif
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::ConfirmResumeApplicationPermission(nn::ncm::ApplicationId applicationId, const nn::sf::InArray<std::int8_t>& ratingAge, bool freeCommunicationFlag) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_RatingCheck);

    NN_RESULT_DO(g_pMain->GetSettingsManager().ConfirmResumeApplicationPermission(g_pMain->GetStates(),
        applicationId,
        ratingAge.GetData(),
        ratingAge.GetLength(),
        freeCommunicationFlag));

    // ※ ここでは g_pMain->GetStates().freeCommunicationUsed や g_pMain->GetStates().stereoVisionUsed はリセットしない
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::ConfirmPlayableApplicationVideo(nn::ncm::ApplicationId applicationId, const nn::sf::InArray<std::int8_t>& ratingAge) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_ReadStatus);

    auto& states = g_pMain->GetStatesRef();

    NN_RESULT_DO(g_pMain->GetSettingsManager().ConfirmPlayableApplicationVideo(
        states,
        applicationId,
        ratingAge.GetData(),
        ratingAge.GetLength()));

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::ConfirmSnsPostPermission() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_SnsCheck);

    NN_RESULT_DO(g_pMain->GetSettingsManager().ConfirmSnsPostPermission(g_pMain->GetStates()));

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::ConfirmSystemSettingsPermission() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_ReadStatus);

    NN_RESULT_DO(g_pMain->GetSettingsManager().ConfirmSystemSettingsPermission());

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::IsRestrictionTemporaryUnlocked(nn::sf::Out<bool> outValue) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_ReadStatus);

    *outValue = g_pMain->GetStates().temporaryUnlocked || g_pMain->GetSettingsManager().IsAllFeaturesDisabled();
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::RevertRestrictionTemporaryUnlocked() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    if (g_pMain->GetStates().temporaryUnlocked)
    {
        // 一時解除状態を戻す
        ApplicationStateManager::UnsetRestrictionTemporaryUnlocked();
        if (IsWatcherAvailable())
        {
            g_pWatcher->GetWatcherEventStorage().AddLockEvent();
            g_pWatcher->GetWatcherEventManager().UpdateUnlockRestrictionStatus();
        }
    }
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::EnterRestrictedSystemSettings() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    auto& states = g_pMain->GetStatesRef();
    if (states.restrictedSystemSettingsEnterCount < std::numeric_limits<int>::max())
    {
        ++states.restrictedSystemSettingsEnterCount;
    }
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::LeaveRestrictedSystemSettings() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    auto& states = g_pMain->GetStatesRef();
    if (states.restrictedSystemSettingsEnterCount > 0)
    {
        --states.restrictedSystemSettingsEnterCount;
    }
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::IsRestrictedSystemSettingsEntered(nn::sf::Out<bool> outValue) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_ReadStatus);

    *outValue = (g_pMain->GetSettingsManager().IsRestrictionEnabled() &&
        (g_pMain->GetStates().restrictedSystemSettingsEnterCount > 0));
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::RevertRestrictedSystemSettingsEntered() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    g_pMain->GetStatesRef().restrictedSystemSettingsEnterCount = 0;
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetRestrictedFeatures(nn::sf::Out<int> outFeatures) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_ReadStatus);

    *outFeatures = g_pMain->GetSettingsManager().GetRestrictedFeatures();
    NN_RESULT_SUCCESS;
}

// ペアコン設定変更関数

nn::Result ParentalControlServiceImpl::IsRestrictionEnabled(nn::sf::Out<bool> outValue) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, (ipc::Capability::Capability_ReadStatus | ipc::Capability::Capability_Recovery));

    *outValue = g_pMain->GetSettingsManager().IsRestrictionEnabled();
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetSafetyLevel(nn::sf::Out<int> outSafetyLevel) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    *outSafetyLevel = static_cast<int>(g_pMain->GetSettingsManager().GetSafetyLevel());
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::SetSafetyLevel(int level) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    g_pMain->GetSettingsManager().SetSafetyLevel(static_cast<SafetyLevel>(level));
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetSafetyLevelSettings(nn::sf::Out<SafetyLevelSettings> pSettings, int level) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);
    NN_RESULT_THROW_UNLESS(level >= 0 && level < SafetyLevel::SafetyLevel_Max, nn::pctl::ResultInvalidValue());

    if (level == SafetyLevel::SafetyLevel_Custom)
    {
        g_pMain->GetSettingsManager().GetCustomSettings(pSettings.GetPointer());
    }
    else
    {
        nn::settings::system::RegionCode region;
        nn::settings::system::GetRegionCode(&region);
        if (region == nn::settings::system::RegionCode::RegionCode_Japan)
        {
            *pSettings = common::SafetyLevelPresetSettingsForJapan[level];
        }
        else
        {
            *pSettings = common::SafetyLevelPresetSettings[level];
        }
    }
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetCurrentSettings(nn::sf::Out<SafetyLevelSettings> pSettings) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);
    g_pMain->GetSettingsManager().GetCurrentSettings(pSettings.GetPointer());

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::SetCustomSafetyLevelSettings(SafetyLevelSettings pSettings) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    g_pMain->GetSettingsManager().SetCustomSafetyLevelSettings(pSettings);
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetDefaultRatingOrganization(nn::sf::Out<int> outOrganization) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_RatingCheck);
    *outOrganization = static_cast<int>(g_pMain->GetSettingsManager().GetDefaultRatingOrganization());
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::SetDefaultRatingOrganization(int organization) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    g_pMain->GetSettingsManager().SetDefaultRatingOrganization(static_cast<nn::ns::RatingOrganization>(organization));
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::DeleteSettings() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, (ipc::Capability::Capability_Management | ipc::Capability::Capability_Recovery));

    g_pMain->GetSettingsManager().DeleteSettings();

    ApplicationStateManager::UnsetRestrictionTemporaryUnlocked();

    auto& states = g_pMain->GetStatesRef();
    states.freeCommunicationUsed = false; // MEMO: 設定削除時はアプリが終了されている
    states.stereoVisionUsed = false;
    states.restrictedSystemSettingsEnterCount = 0;

    // 通常は未連携時は影響がないが、デバッグ処理でプレイタイマーを稼働させている場合があることも考慮して対応
    watcher::WatcherEventManager::ClearPlayTimerSettingsFromSaveData();
    // 連携解除のシグナルも落とす
    watcher::NetworkManager::ClearUnlinkedFlag();

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetFreeCommunicationApplicationListCount(nn::sf::Out<int> outCount) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    *outCount = static_cast<int>(g_pMain->GetSettingsManager().GetFreeCommunicationApplicationListCount());
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetFreeCommunicationApplicationList(nn::sf::Out<int> outCount, const nn::sf::OutArray<nn::pctl::FreeCommunicationApplicationInfo>& pOutTitleInfoArray, int offset) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    *outCount = static_cast<int>(
        g_pMain->GetSettingsManager().GetFreeCommunicationApplicationList(
            pOutTitleInfoArray.GetData(), static_cast<size_t>(offset), pOutTitleInfoArray.GetLength())
        );
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::UpdateFreeCommunicationApplicationList(const nn::sf::InArray<nn::pctl::FreeCommunicationApplicationInfo>& pTitleInfoArray) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    g_pMain->GetSettingsManager().UpdateFreeCommunicationApplicationList(pTitleInfoArray.GetData(), pTitleInfoArray.GetLength());
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::AddToFreeCommunicationApplicationList(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    g_pMain->GetSettingsManager().AddFreeCommunicationApplicationList(applicationId);
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::DeleteFromFreeCommunicationApplicationListForDebug(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    g_pMain->GetSettingsManager().DeleteFreeCommunicationApplicationList(applicationId);
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::ClearFreeCommunicationApplicationListForDebug() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    g_pMain->GetSettingsManager().ClearFreeCommunicationApplicationList();
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::DisableFeaturesForReset() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    // 非Recoveryモードの無効化(→ 一時的な無効化)とする
    DisableAllFeaturesImpl(false);

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::NotifyApplicationDownloadStarted(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    // (Watcherの初期化チェックはイベント発行時に行う)
    g_pMain->GetSettingsManager().PostApplicationDownloadStarted(applicationId);

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetExemptApplicationListCountForDebug(nn::sf::Out<int> outCount) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    *outCount = static_cast<int>(g_pMain->GetSettingsManager().GetExemptApplicationListCount());
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetExemptApplicationListForDebug(nn::sf::Out<int> outCount, const nn::sf::OutArray<nn::pctl::ExemptApplicationInfo>& pOutTitleInfoArray, int offset) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    *outCount = static_cast<int>(
        g_pMain->GetSettingsManager().GetExemptApplicationList(
            pOutTitleInfoArray.GetData(), static_cast<size_t>(offset), pOutTitleInfoArray.GetLength())
        );
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::UpdateExemptApplicationListForDebug(const nn::sf::InArray<nn::pctl::ExemptApplicationInfo>& pTitleInfoArray) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    g_pMain->GetSettingsManager().UpdateExemptApplicationList(pTitleInfoArray.GetData(), pTitleInfoArray.GetLength());
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::AddToExemptApplicationListForDebug(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    g_pMain->GetSettingsManager().AddExemptApplicationList(applicationId);
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::DeleteFromExemptApplicationListForDebug(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    g_pMain->GetSettingsManager().DeleteExemptApplicationList(applicationId);
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::ClearExemptApplicationListForDebug() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Management);

    g_pMain->GetSettingsManager().ClearExemptApplicationList();
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::ConfirmStereoVisionPermission() NN_NOEXCEPT
{
    // 特に権限チェックを行わない
    //NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_General);

    auto& states = g_pMain->GetStatesRef();
    NN_RESULT_DO(g_pMain->GetSettingsManager().ConfirmStereoVisionPermission(states));

    // 設定状態にかかわらず通過した状態を持たせる
    // (アプリを中断して再開したときの判定に用いるため)
    states.stereoVisionUsed = true;
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::ResetConfirmedStereoVisionPermission() NN_NOEXCEPT
{
    // 特に権限チェックを行わない
    //NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_General);

    auto& states = g_pMain->GetStatesRef();
    states.stereoVisionUsed = false;

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::IsStereoVisionPermitted(nn::sf::Out<bool> outValue) NN_NOEXCEPT
{
    *outValue = false;
    auto& states = g_pMain->GetStatesRef();
    NN_RESULT_DO(g_pMain->GetSettingsManager().ConfirmStereoVisionPermission(states));

    *outValue = true;
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::ConfirmStereoVisionRestrictionConfigurable() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_StereoVision);

    NN_RESULT_DO(g_pMain->GetSettingsManager().ConfirmStereoVisionRestrictionConfigurable());

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetStereoVisionRestriction(nn::sf::Out<bool> outRestricted) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_StereoVision);

    *outRestricted = g_pMain->GetSettingsManager().GetStereoVisionRestriction();

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::SetStereoVisionRestriction(bool restricted) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_StereoVision);

    g_pMain->GetSettingsManager().SetStereoVisionRestriction(restricted);

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::UnlockRestrictionTemporarily(const nn::sf::InArray<char>& code) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_PinCode);

    NN_RESULT_THROW_UNLESS(code.GetData(), nn::pctl::ResultInvalidValue());

    NN_RESULT_DO(g_pMain->GetSettingsManager().CheckPinCode(code.GetData(), code.GetLength()));

    if (!g_pMain->GetStates().temporaryUnlocked)
    {
        // Unlockイベントは先に発行
        if (IsWatcherAvailable())
        {
            g_pWatcher->GetWatcherEventStorage().AddUnlockEvent();
        }
        // 一時解除状態を設定
        ApplicationStateManager::SetRestrictionTemporaryUnlocked();

        // 状態変更による更新処理は状態を書き換えてから実行
        if (IsWatcherAvailable())
        {
            g_pWatcher->GetWatcherEventManager().UpdateUnlockRestrictionStatus();
        }
    }
    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::UnlockSystemSettingsRestriction(const nn::sf::InArray<char>& code) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_PinCode);

    NN_RESULT_THROW_UNLESS(code.GetData(), nn::pctl::ResultInvalidValue());

    NN_RESULT_DO(g_pMain->GetSettingsManager().CheckPinCode(code.GetData(), code.GetLength()));

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::SetPinCode(const nn::sf::InArray<char>& code) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_PinCode);

    NN_RESULT_DO(g_pMain->GetSettingsManager().SetPinCode(code.GetData(), code.GetLength()));

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GenerateInquiryCode(nn::sf::Out<nn::pctl::InquiryCode> pOutCodeData) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_MasterKey);

    g_pMain->GetSettingsManager().GenerateInquiryCode(pOutCodeData.GetPointer());

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::CheckMasterKey(nn::sf::Out<bool> outValue, const nn::pctl::InquiryCode& codeData, const nn::sf::InArray<char>& masterKey) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_MasterKey);

    // (masterKey.GetLength() はNULL文字込みの長さ)
    *outValue = g_pMain->GetSettingsManager().CheckMasterKey(codeData, masterKey.GetData(), masterKey.GetLength());

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetPinCodeLength(nn::sf::Out<int> outValue) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_PinCode);

    *outValue = g_pMain->GetSettingsManager().GetPinCodeLength();

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetPinCodeChangedEvent(nn::sf::Out<nn::sf::NativeHandle> pEvent) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_PinCode);

    // Watcherが初期化されていない場合は InvalidNativeHandle を返す
    if (!IsWatcherAvailable())
    {
        *pEvent = nn::sf::NativeHandle();
    }
    else
    {
        *pEvent = nn::sf::NativeHandle(
            g_pWatcher->GetWatcherEventManager().GetPinCodeChangedEvent()->GetReadableHandle(),
            false
            );
    }

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetPinCode(nn::sf::Out<int> outLength, const nn::sf::OutArray<char>& outCode) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_PinCode);
    NN_RESULT_THROW_UNLESS(outCode.GetLength() > 0 && outCode.GetData() != nullptr, nn::pctl::ResultInvalidValue());

    auto r = g_pMain->GetSettingsManager().GetPinCode(outCode.GetData(), outCode.GetLength());
    *outLength = static_cast<int>(r);

    NN_RESULT_SUCCESS;
}

// 見守り機能サーバー連携処理関数

nn::Result ParentalControlServiceImpl::IsPairingActive(nn::sf::Out<bool> outValue) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);

    // Watcherが初期化されていない場合、直接設定値を読み込んでその結果を返す
    if (!IsWatcherAvailable())
    {
        watcher::Config config;
        auto result = watcher::NetworkManager::LoadConfig(config);
        NN_ABORT_UNLESS(result.IsSuccess(), "Unexpected result for file open: %08x", result.GetInnerValueForDebug());
        *outValue = (config.isPairingActive != 0);
    }
    else
    {
        *outValue = g_pWatcher->GetNetworkManager().IsPairingActive();
    }

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetSettingsLastUpdated(nn::sf::Out<nn::time::PosixTime> outValue) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);

    // Watcherが初期化されていない場合は ResultNotFound を返す
    // (未同期状態と同じ)
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultNotFound());

    NN_RESULT_THROW_UNLESS(
        g_pWatcher->GetNetworkManager().GetSettingsUpdatedTime(outValue.GetPointer()),
        nn::pctl::ResultNotFound()
        );

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::DeletePairing() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, (ipc::Capability::Capability_Watcher | ipc::Capability::Capability_Recovery));

    // 無効化モード、またはWatcherが初期化されていない場合、直接設定値を読み込んで情報を削除する
    if (g_pMain->GetSettingsManager().IsAllFeaturesDisabled() || !IsWatcherAvailable())
    {
        // 連携情報
        {
            watcher::Config config;
            auto result = watcher::NetworkManager::LoadConfig(config);
            NN_ABORT_UNLESS(result.IsSuccess(), "Unexpected result for file open: %08x", result.GetInnerValueForDebug());
            watcher::NetworkManager::ClearPairingInfoFromConfig(config);
            NN_ABORT_UNLESS_RESULT_SUCCESS(watcher::NetworkManager::SaveConfig(config));

            // 制限対象外リストのクリア
            g_pMain->GetSettingsManager().ClearExemptApplicationList();
        }
        // プレイタイマー情報
        watcher::WatcherEventManager::ClearPlayTimerSettingsFromSaveData();
    }
    // Watcher が利用できる場合は通常の連携情報削除を行う
    else
    {
        // このケースにおいては通知は行わない
        g_pWatcher->GetNetworkManager().ClearPairingInfo(false);
    }

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetPairingAccountInfo(nn::sf::Out<nn::pctl::detail::PairingAccountInfoBase> pAccountInfo, const nn::pctl::detail::PairingInfoBase& pairingInfo) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());
    NN_RESULT_THROW_UNLESS(pairingInfo.state != PairingState::PairingState_None, nn::pctl::ResultInvalidOperation());

    PairingInfoDataHolder data = g_pWatcher->AcquirePairingInfoData();
    NN_RESULT_THROW_UNLESS(pairingInfo.id == static_cast<uint64_t>(data->deviceId), nn::pctl::ResultNeedsRefresh());

    pAccountInfo->id = static_cast<uint64_t>(data->deviceId);

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetAccountNickname(nn::sf::Out<std::uint32_t> pActualSize, const nn::sf::OutArray<char>& pNickname, const nn::pctl::detail::PairingAccountInfoBase& accountInfo) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    PairingInfoDataHolder data = g_pWatcher->AcquirePairingInfoData();
    NN_RESULT_THROW_UNLESS(accountInfo.id == static_cast<uint64_t>(data->deviceId), nn::pctl::ResultNeedsRefresh());
    NN_RESULT_THROW_UNLESS(data->countOwners > 0, nn::pctl::ResultNeedsRefresh());

    // 現時点では常に 0 番目の要素を用いる
    auto& owner = data->arrayOwners[0];
    // 常に終端にNULL文字を入れるようにする
    size_t nicknameSize = nn::util::Strnlen(owner.user.nickname, nn::account::NicknameBytesMax) + 1;
    NN_SDK_ASSERT(nicknameSize > 0);
    size_t copySize = std::min<size_t>(sizeof(char) * pNickname.GetLength(), sizeof(char) * (nicknameSize - 1));
    auto ptr = pNickname.GetData();
    std::memcpy(ptr, owner.user.nickname, copySize);
    if (pNickname.GetLength() >= nicknameSize)
    {
        ptr[nicknameSize - 1] = 0;
    }
    *pActualSize = static_cast<uint32_t>(nicknameSize);

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetAccountState(nn::sf::Out<int> outState, const nn::pctl::detail::PairingAccountInfoBase& accountInfo) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);

    PairingInfoDataHolder data = g_pWatcher->AcquirePairingInfoData();
    NN_RESULT_THROW_UNLESS(accountInfo.id == static_cast<uint64_t>(data->deviceId), nn::pctl::ResultNeedsRefresh());
    NN_RESULT_THROW_UNLESS(data->countOwners > 0, nn::pctl::ResultNeedsRefresh());

    // 現時点では常に 0 番目の要素を用いる
    auto& owner = data->arrayOwners[0];
    size_t nicknameSize = nn::util::Strnlen(owner.user.nickname, nn::account::NicknameBytesMax);

    // nickname が空文字列の場合は削除されている(退会されている)とみなす
    if (nicknameSize == 0)
    {
        *outState = PairingAccountState::PairingAccountState_Deleted;
    }
    else
    {
        *outState = PairingAccountState::PairingAccountState_Active;
    }

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetSynchronizationEvent(nn::sf::Out<nn::sf::NativeHandle> pEvent) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    *pEvent = nn::sf::NativeHandle(
        g_pWatcher->GetWatcherEventManager().GetSynchronizationEvent()->GetReadableHandle(),
        false
    );

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::StartPlayTimer() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    g_pWatcher->GetWatcherEventManager().SetPlayTimerEnabled(true);

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::StopPlayTimer() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    g_pWatcher->GetWatcherEventManager().SetPlayTimerEnabled(false);

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::IsPlayTimerEnabled(nn::sf::Out<bool> outValue) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_ReadStatus | ipc::Capability::Capability_Watcher);

    // Watcherが初期化されていない == この処理が不要なモードで動作しているので
    // ここでは false を返す
    if (!IsWatcherAvailable())
    {
        *outValue = false;
    }
    else
    {
        *outValue = g_pWatcher->GetWatcherEventManager().IsPlayTimerEnabled();
    }

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetPlayTimerRemainingTime(nn::sf::Out<nn::TimeSpanType> outTime) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_ReadStatus | ipc::Capability::Capability_Watcher);

    // Watcherが初期化されていない == この処理が不要なモードで動作しているので
    // ここでは 0 を返す
    if (!IsWatcherAvailable())
    {
        *outTime = nn::TimeSpan(0);
    }
    else
    {
        *outTime = g_pWatcher->GetWatcherEventManager().GetRemainingTime();
    }

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::IsRestrictedByPlayTimer(nn::sf::Out<bool> outValue) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);

    // Watcherが初期化されていない == この処理が不要なモードで動作しているので
    // ここでは false を返す
    if (!IsWatcherAvailable())
    {
        *outValue = false;
    }
    else
    {
        *outValue = g_pWatcher->GetWatcherEventManager().IsRestrictedByPlayTimer();
    }

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetPlayTimerSettings(nn::sf::Out<nn::pctl::PlayTimerSettings> pSettings) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);

    // Watcherが初期化されていない == この処理が不要なモードで動作しているので
    // ここでは未設定として 0 初期化されたデータを返す
    if (!IsWatcherAvailable())
    {
        std::memset(pSettings.GetPointer(), 0, sizeof(nn::pctl::PlayTimerSettings));
    }
    else
    {
        g_pWatcher->GetWatcherEventManager().GetPlayTimerSettings(pSettings.GetPointer());
    }

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetPlayTimerEventToRequestSuspension(nn::sf::Out<nn::sf::NativeHandle> pEvent) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    *pEvent = nn::sf::NativeHandle(
        g_pWatcher->GetWatcherEventManager().GetPlayTimerEventToRequestSuspension()->GetReadableHandle(),
        false
        );

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::IsPlayTimerAlarmDisabled(nn::sf::Out<bool> outValue) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);

    // Watcherが初期化されていない == この処理が不要なモードで動作しているので
    // ここでは false を返す
    if (!IsWatcherAvailable())
    {
        *outValue = false;
    }
    else
    {
        *outValue = g_pWatcher->GetWatcherEventManager().IsAlarmDisabled();
    }

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::SetPlayTimerSettingsForDebug(const nn::pctl::PlayTimerSettings& settings) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    // 制限設定が有効な場合のみ設定を適用する
    if (g_pMain->GetSettingsManager().IsRestrictionEnabled())
    {
        g_pWatcher->GetWatcherEventManager().UpdatePlayTimerSettings(settings);
    }

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::SetPlayTimerAlarmDisabledForDebug(bool isDisabled) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    // 制限設定が有効な場合のみ設定を適用する
    if (g_pMain->GetSettingsManager().IsRestrictionEnabled())
    {
        nn::time::PosixTime now;
        if (common::GetNetworkTime(&now))
        {
            g_pWatcher->GetWatcherEventManager().SetAlarmDisabled(isDisabled, now);
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetPlayTimerSpentTimeForTest(nn::sf::Out<nn::TimeSpanType> outTime) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);

    // Watcherが初期化されていない == この処理が不要なモードで動作しているので
    // ここでは 0 を返す
    if (!IsWatcherAvailable())
    {
        *outTime = nn::TimeSpan(0);
    }
    else
    {
        *outTime = g_pWatcher->GetWatcherEventManager().GetSpentTime();
    }

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::NotifyWrongPinCodeInputManyTimes() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);

    // (Watcherの初期化チェックはイベント発行時に行う)
    g_pMain->GetSettingsManager().PostWrongPinCode();

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::CancelNetworkRequest() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    g_pWatcher->GetNetworkManager().Cancel();

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetUnlinkedEvent(nn::sf::Out<nn::sf::NativeHandle> pEvent) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    *pEvent = nn::sf::NativeHandle(
        g_pWatcher->GetNetworkManager().GetUnlinkedEvent()->GetReadableHandle(),
        false
    );

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::ClearUnlinkedEvent() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    watcher::NetworkManager::ClearUnlinkedFlag();

    NN_RESULT_SUCCESS;
}


nn::Result ParentalControlServiceImpl::DisableAllFeatures(nn::sf::Out<bool> outIsAlreadyDisabled) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Recovery);

    // Recoveryモードでの無効化(→ 永続化される無効化)とする
    *outIsAlreadyDisabled = DisableAllFeaturesImpl(true);

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::PostEnableAllFeatures(nn::sf::Out<bool> outIsAlreadyEnabled) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Recovery);

    *outIsAlreadyEnabled = g_pMain->GetSettingsManager().PostEnableAllFeatures();
    // NOTE: 再起動するまでは無効化のままを継続する

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::IsAllFeaturesDisabled(nn::sf::Out<bool> outValue, nn::sf::Out<bool> outIsEnabledOnNextBoot) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Recovery);

    *outValue = g_pMain->GetSettingsManager().IsAllFeaturesDisabled(outIsEnabledOnNextBoot.GetPointer());

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::RequestPairingAsync(nn::sf::Out<nn::pctl::detail::AsyncData> pAsyncData, nn::sf::Out<nn::sf::NativeHandle> pEvent, const nn::sf::InArray<char>& code) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    // 以前連携時に貯まっていたイベントがあれば削除しておく
    g_pWatcher->GetWatcherEventStorage().DiscardAllEvents();

    service::common::AsyncContext* pContext;

    PairingInfoDataHolder* pData = new PairingInfoDataHolder(g_pWatcher->AcquirePairingInfoData());
    NN_UTIL_SCOPE_EXIT
    {
        if (pData != nullptr)
        {
            delete pData;
        }
    };
    if ((*pData)->arrayOwners != nullptr)
    {
        FreeMemoryBlock((*pData)->arrayOwners);
    }

    (*pData)->countOwners = 0;
    (*pData)->arrayOwners = static_cast<watcher::ServerDeviceOwner*>(
        AllocateMemoryBlock(sizeof(watcher::ServerDeviceOwner) * MaxServerDeviceOwnerCount)
        );
    NN_RESULT_THROW_UNLESS((*pData)->arrayOwners != nullptr, nn::pctl::ResultUnexpected());

    NN_RESULT_DO(
        g_pWatcher->GetNetworkManager().RequestPairing(
            &pContext,
            &(*pData)->deviceId,
            &(*pData)->countOwners,
            code.GetData(),
            code.GetLength(),
            (*pData)->arrayOwners,
            MaxServerDeviceOwnerCount
            )
        );

    PushAsyncData(pAsyncData.GetPointer(), pContext, pData);
    pData = nullptr; // delete されないようにする
    auto systemEvent = pContext->GetSystemEvent();
    NN_SDK_ASSERT_NOT_NULL(systemEvent);
    *pEvent = nn::sf::NativeHandle(
        nn::os::DetachReadableHandleOfSystemEvent(systemEvent),
        true // 呼び出し元で管理されるようにする
        );

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::FinishRequestPairing(nn::sf::Out<nn::pctl::detail::PairingInfoBase> pInfo, const nn::pctl::detail::AsyncData& asyncData) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    PairingInfoDataHolder* pData = nullptr;
    auto pContext = PopAsyncData(&pData, asyncData);
    // 戻り値が nullptr の場合はキャンセルされている可能性あり
    NN_RESULT_THROW_UNLESS(pContext != nullptr, nn::pctl::ResultCanceled());
    NN_UTIL_SCOPE_EXIT
    {
        if (pData != nullptr)
        {
            delete pData;
        }
    };

    // 関数内で待機
    NN_RESULT_DO(WaitAsyncContext(pContext));

    NN_STATIC_ASSERT(sizeof(pInfo->id) >= sizeof((*pData)->deviceId));

    pInfo->id = static_cast<uint64_t>((*pData)->deviceId);
    pInfo->state = PairingState::PairingState_Processing;

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::AuthorizePairingAsync(nn::sf::Out<nn::pctl::detail::AsyncData> pAsyncData, nn::sf::Out<nn::sf::NativeHandle> pEvent, const nn::pctl::detail::PairingInfoBase& pairingInfo) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    PairingInfoDataHolder data = g_pWatcher->AcquirePairingInfoData();
    NN_RESULT_THROW_UNLESS(data->deviceId == static_cast<watcher::ServerDeviceId>(pairingInfo.id), nn::pctl::ResultPairingFailed());
    NN_RESULT_THROW_UNLESS(pairingInfo.state == PairingState::PairingState_Processing, nn::pctl::ResultPairingFailed());

    service::common::AsyncContext* pContext;

    NN_RESULT_DO(
        g_pWatcher->GetNetworkManager().RequestAuthorizePairing(
            &pContext, static_cast<watcher::ServerDeviceId>(pairingInfo.id)
            )
        );

    PushAsyncData(pAsyncData.GetPointer(), pContext, nullptr);
    auto systemEvent = pContext->GetSystemEvent();
    NN_SDK_ASSERT_NOT_NULL(systemEvent);
    *pEvent = nn::sf::NativeHandle(
        nn::os::DetachReadableHandleOfSystemEvent(systemEvent),
        true // 呼び出し元で管理されるようにする
        );

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::FinishAuthorizePairing(nn::sf::Out<nn::pctl::detail::PairingInfoBase> pUpdatedInfo, const nn::pctl::detail::AsyncData& asyncData) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    auto pContext = PopAsyncData(nullptr, asyncData);
    // 戻り値が nullptr の場合はキャンセルされている可能性あり
    NN_RESULT_THROW_UNLESS(pContext != nullptr, nn::pctl::ResultCanceled());

    // 関数内で待機
    NN_RESULT_DO(WaitAsyncContext(pContext));

    // 連携が完了しているので保存済みのIDを返す
    pUpdatedInfo->id = g_pWatcher->GetNetworkManager().GetSavedDeviceId();
    pUpdatedInfo->state = PairingState::PairingState_Active;

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::RetrievePairingInfoAsync(nn::sf::Out<nn::pctl::detail::AsyncData> pAsyncData, nn::sf::Out<nn::sf::NativeHandle> pEvent) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    if (!g_pWatcher->GetNetworkManager().IsPairingActive())
    {
        pAsyncData->id = 0;
        pAsyncData->index = 0;
        // 無効なハンドルを返す(shimレイヤーでハンドルして 0 値を返すようにする)
        *pEvent = nn::sf::NativeHandle();
        NN_RESULT_SUCCESS;
    }

    PairingInfoDataHolder* pData = new PairingInfoDataHolder(g_pWatcher->AcquirePairingInfoData());
    NN_UTIL_SCOPE_EXIT
    {
        if (pData != nullptr)
        {
            delete pData;
        }
    };
    watcher::ServerDeviceId deviceId = g_pWatcher->GetNetworkManager().GetSavedDeviceId();

    if ((*pData)->arrayOwners != nullptr)
    {
        FreeMemoryBlock((*pData)->arrayOwners);
    }

    (*pData)->deviceId = deviceId;
    (*pData)->countOwners = 0;
    (*pData)->arrayOwners = static_cast<watcher::ServerDeviceOwner*>(
        AllocateMemoryBlock(sizeof(watcher::ServerDeviceOwner) * MaxServerDeviceOwnerCount)
        );
    NN_RESULT_THROW_UNLESS((*pData)->arrayOwners != nullptr, nn::pctl::ResultUnexpected());

    service::common::AsyncContext* pContext;

    NN_RESULT_DO(
        g_pWatcher->GetNetworkManager().RequestRetrieveOwners(
            &pContext,
            &(*pData)->countOwners,
            (*pData)->arrayOwners,
            MaxServerDeviceOwnerCount,
            deviceId
            )
        );

    PushAsyncData(pAsyncData.GetPointer(), pContext, pData);
    pData = nullptr; // delete されないようにする
    auto systemEvent = pContext->GetSystemEvent();
    NN_SDK_ASSERT_NOT_NULL(systemEvent);
    *pEvent = nn::sf::NativeHandle(
        nn::os::DetachReadableHandleOfSystemEvent(systemEvent),
        true // 呼び出し元で管理されるようにする
        );

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::FinishRetrievePairingInfo(nn::sf::Out<nn::pctl::detail::PairingInfoBase> pInfo, const nn::pctl::detail::AsyncData& asyncData) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    PairingInfoDataHolder* pData = nullptr;
    auto pContext = PopAsyncData(&pData, asyncData);
    // 戻り値が nullptr の場合はキャンセルされている可能性あり
    NN_RESULT_THROW_UNLESS(pContext != nullptr, nn::pctl::ResultCanceled());
    NN_UTIL_SCOPE_EXIT
    {
        if (pData != nullptr)
        {
            delete pData;
        }
    };

    // 関数内で待機
    NN_RESULT_TRY(WaitAsyncContext(pContext))
        // このエラーの場合は連携情報が削除されていることを表すのでローカルデータの削除処理も行う
        NN_RESULT_CATCH(nn::pctl::ResultServerResourceIsNotFound)
        {
            g_pWatcher->GetNetworkManager().HandlePairingDeletedStatus();
            NN_RESULT_THROW(nn::pctl::ResultPairingDeleted());
        }
        // ResultServerResourceIsNotFound と同様のハンドリングをする
        // (ただしResult自体はそのまま返す)
        NN_RESULT_CATCH(nn::pctl::ResultServerResourceNotAvailable)
        {
            g_pWatcher->GetNetworkManager().HandlePairingDeletedStatus();
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    // 0人は想定外なので削除された扱いにする
    if ((*pData)->countOwners == 0)
    {
        g_pWatcher->GetNetworkManager().HandlePairingDeletedStatus();
        NN_RESULT_THROW(nn::pctl::ResultPairingDeleted());
    }

    pInfo->id = static_cast<uint64_t>((*pData)->deviceId);
    pInfo->state = PairingState::PairingState_Active;

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::UnlinkPairingAsync(nn::sf::Out<nn::pctl::detail::AsyncData> pAsyncData, nn::sf::Out<nn::sf::NativeHandle> pEvent, bool force) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    auto& manager = g_pWatcher->GetNetworkManager();
    if (!manager.IsPairingActive())
    {
        pAsyncData->id = 0;
        pAsyncData->index = 0;
        // 無効なハンドルを返す(shimレイヤーでハンドルして何もしないようにする)
        *pEvent = nn::sf::NativeHandle();
        NN_RESULT_SUCCESS;
    }
    watcher::ServerDeviceId deviceId = manager.GetSavedDeviceId();

    common::AsyncContext* pContext;
    auto result = manager.RequestUnlinkDevice(&pContext, deviceId, force);

    // force == true の場合は失敗を返さない
    NN_RESULT_THROW_UNLESS(result.IsSuccess() || force, result);
    if (result.IsFailure())
    {
        NN_DETAIL_PCTL_WARN("pctl: RequestUnlinkDevice failed: 0x%08X\n", result.GetInnerValueForDebug());
        NN_RESULT_SUCCESS;
    }

    PushAsyncData(pAsyncData.GetPointer(), pContext, nullptr);
    auto systemEvent = pContext->GetSystemEvent();
    NN_SDK_ASSERT_NOT_NULL(systemEvent);
    *pEvent = nn::sf::NativeHandle(
        nn::os::DetachReadableHandleOfSystemEvent(systemEvent),
        true // 呼び出し元で管理されるようにする
        );

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::FinishUnlinkPairing(const nn::pctl::detail::AsyncData& asyncData, bool force) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    auto pContext = PopAsyncData(nullptr, asyncData);
    // 戻り値が nullptr の場合はキャンセルされている可能性あり
    NN_RESULT_THROW_UNLESS(pContext != nullptr, nn::pctl::ResultCanceled());

    auto result = WaitAsyncContext(pContext);
    NN_RESULT_THROW_UNLESS(result.IsSuccess() || force, result);
    if (result.IsFailure())
    {
        NN_DETAIL_PCTL_WARN("pctl: RequestUnlinkDevice returns failure: 0x%08X\n", result.GetInnerValueForDebug());
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetAccountMiiImageAsync(nn::sf::Out<nn::pctl::detail::AsyncData> pAsyncData, nn::sf::Out<nn::sf::NativeHandle> pEvent, nn::sf::Out<std::uint32_t> pActualSize, const nn::sf::OutBuffer& pImage, const nn::pctl::detail::PairingAccountInfoBase& accountInfo) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    PairingInfoDataHolder* pData = new PairingInfoDataHolder(g_pWatcher->AcquirePairingInfoData());
    NN_UTIL_SCOPE_EXIT
    {
        if (pData != nullptr)
        {
            delete pData;
        }
    };
    NN_RESULT_THROW_UNLESS(accountInfo.id == static_cast<uint64_t>((*pData)->deviceId), nn::pctl::ResultNeedsRefresh());
    NN_RESULT_THROW_UNLESS((*pData)->countOwners > 0, nn::pctl::ResultNeedsRefresh());

    // 現時点では常に 0 番目の要素を用いる
    auto& owner = (*pData)->arrayOwners[0];

    auto& manager = g_pWatcher->GetNetworkManager();
    common::AsyncContext* pContext = nullptr;

    // Mii画像処理の排他制御
    NN_UTIL_LOCK_GUARD(manager.GetMutexForMiiImage());

    NN_RESULT_DO(
        manager.TryToDownloadMiiImage(
            &pContext,
            owner.user
        )
    );
    // pContext == nullptr の場合は既に取得済み
    if (pContext == nullptr)
    {
        NN_RESULT_DO(
            manager.GetMiiImage(
                pActualSize.GetPointer(),
                owner.user,
                pImage.GetPointerUnsafe(),
                pImage.GetSize()
                )
            );
        pAsyncData->id = 0;
        pAsyncData->index = 0;
        *pEvent = nn::sf::NativeHandle();
        NN_RESULT_SUCCESS;
    }

    PushAsyncData(pAsyncData.GetPointer(), pContext, pData);
    pData = nullptr; // delete されないようにする
    auto systemEvent = pContext->GetSystemEvent();
    NN_SDK_ASSERT_NOT_NULL(systemEvent);
    *pEvent = nn::sf::NativeHandle(
        nn::os::DetachReadableHandleOfSystemEvent(systemEvent),
        true // 呼び出し元で管理されるようにする
        );

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::FinishGetAccountMiiImage(nn::sf::Out<std::uint32_t> pActualSize, const nn::sf::OutBuffer& pImage, const nn::pctl::detail::AsyncData& asyncData) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    PairingInfoDataHolder* pData = nullptr;
    auto pContext = PopAsyncData(&pData, asyncData);
    // 戻り値が nullptr の場合はキャンセルされている可能性あり
    NN_RESULT_THROW_UNLESS(pContext != nullptr, nn::pctl::ResultCanceled());
    NN_UTIL_SCOPE_EXIT
    {
        if (pData != nullptr)
        {
            delete pData;
        }
    };

    // 現時点では常に 0 番目の要素を用いる
    auto& owner = (*pData)->arrayOwners[0];

    auto& manager = g_pWatcher->GetNetworkManager();

    // 関数内で待機(result はここでは見ない)
    while (!pContext->IsFinished())
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
    }
    // Closeも中で行われる
    NN_RESULT_DO(
        manager.FinishDownloadMiiImage(pContext, owner.user)
    );

    NN_RESULT_DO(
        manager.GetMiiImage(
            pActualSize.GetPointer(),
            owner.user,
            pImage.GetPointerUnsafe(),
            pImage.GetSize()
            )
        );

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::GetAccountMiiImageContentTypeAsync(nn::sf::Out<nn::pctl::detail::AsyncData> pAsyncData, nn::sf::Out<nn::sf::NativeHandle> pEvent, nn::sf::Out<std::uint32_t> pActualLength, const nn::sf::OutArray<char>& pContentType, const nn::pctl::detail::PairingAccountInfoBase& accountInfo) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);

    PairingInfoDataHolder* pData = new PairingInfoDataHolder(g_pWatcher->AcquirePairingInfoData());
    NN_UTIL_SCOPE_EXIT
    {
        if (pData != nullptr)
        {
            delete pData;
        }
    };
    NN_RESULT_THROW_UNLESS(accountInfo.id == static_cast<uint64_t>((*pData)->deviceId), nn::pctl::ResultNeedsRefresh());
    NN_RESULT_THROW_UNLESS((*pData)->countOwners > 0, nn::pctl::ResultNeedsRefresh());

    // 現時点では常に 0 番目の要素を用いる
    auto& owner = (*pData)->arrayOwners[0];

    auto& manager = g_pWatcher->GetNetworkManager();
    common::AsyncContext* pContext = nullptr;

    // Mii画像処理の排他制御
    NN_UTIL_LOCK_GUARD(manager.GetMutexForMiiImage());

    NN_RESULT_DO(
        manager.TryToDownloadMiiImage(
            &pContext,
            owner.user
        )
    );
    // pContext == nullptr の場合は既に取得済み
    if (pContext == nullptr)
    {
        NN_RESULT_DO(
            manager.GetMiiImageContentType(
                pActualLength.GetPointer(),
                owner.user,
                pContentType.GetData(),
                pContentType.GetLength()
                )
            );
        pAsyncData->id = 0;
        pAsyncData->index = 0;
        *pEvent = nn::sf::NativeHandle();
        NN_RESULT_SUCCESS;
    }

    PushAsyncData(pAsyncData.GetPointer(), pContext, pData);
    pData = nullptr; // delete されないようにする
    auto systemEvent = pContext->GetSystemEvent();
    NN_SDK_ASSERT_NOT_NULL(systemEvent);
    *pEvent = nn::sf::NativeHandle(
        nn::os::DetachReadableHandleOfSystemEvent(systemEvent),
        true // 呼び出し元で管理されるようにする
        );

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::FinishGetAccountMiiImageContentType(nn::sf::Out<std::uint32_t> pActualLength, const nn::sf::OutArray<char>& pContentType, const nn::pctl::detail::AsyncData& asyncData) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    PairingInfoDataHolder* pData = nullptr;
    auto pContext = PopAsyncData(&pData, asyncData);
    // 戻り値が nullptr の場合はキャンセルされている可能性あり
    NN_RESULT_THROW_UNLESS(pContext != nullptr, nn::pctl::ResultCanceled());
    NN_UTIL_SCOPE_EXIT
    {
        if (pData != nullptr)
        {
            delete pData;
        }
    };

    // 現時点では常に 0 番目の要素を用いる
    auto& owner = (*pData)->arrayOwners[0];

    auto& manager = g_pWatcher->GetNetworkManager();

    // 関数内で待機(result はここでは見ない)
    while (!pContext->IsFinished())
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
    }
    // Closeも中で行われる
    NN_RESULT_DO(
        manager.FinishDownloadMiiImage(pContext, owner.user)
    );

    NN_RESULT_DO(
        manager.GetMiiImageContentType(
            pActualLength.GetPointer(),
            owner.user,
            pContentType.GetData(),
            pContentType.GetLength()
            )
        );

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::SynchronizeParentalControlSettingsAsync(nn::sf::Out<nn::pctl::detail::AsyncData> pAsyncData, nn::sf::Out<nn::sf::NativeHandle> pEvent) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    auto& manager = g_pWatcher->GetNetworkManager();
    if (!manager.IsPairingActive())
    {
        pAsyncData->id = 0;
        pAsyncData->index = 0;
        // 無効なハンドルを返す(shimレイヤーでハンドルして何もしないようにする)
        *pEvent = nn::sf::NativeHandle();
        NN_RESULT_SUCCESS;
    }
    service::common::AsyncContext* pContext;

    // フォアグラウンドモードで呼び出し
    NN_RESULT_DO(
        manager.RequestSynchronizeSettingsForeground(
            &pContext, manager.GetSavedDeviceId(), true
            )
        );

    PushAsyncData(pAsyncData.GetPointer(), pContext, nullptr);
    auto systemEvent = pContext->GetSystemEvent();
    NN_SDK_ASSERT_NOT_NULL(systemEvent);
    *pEvent = nn::sf::NativeHandle(
        nn::os::DetachReadableHandleOfSystemEvent(systemEvent),
        true // 呼び出し元で管理されるようにする
        );

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::RequestUpdateExemptionListAsync(nn::sf::Out<nn::pctl::detail::AsyncData> pAsyncData, nn::sf::Out<nn::sf::NativeHandle> pEvent, nn::ncm::ApplicationId applicationId, bool isExempt) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());
    auto& manager = g_pWatcher->GetNetworkManager();
    NN_RESULT_THROW_UNLESS(manager.IsPairingActive(), nn::pctl::ResultPairingNotActive());

    service::common::AsyncContext* pContext;
    NN_RESULT_DO(
        manager.RequestUpdateExemptionList(
            &pContext, manager.GetSavedDeviceId(), applicationId, isExempt
        )
    );

    PushAsyncData(pAsyncData.GetPointer(), pContext, nullptr);
    auto systemEvent = pContext->GetSystemEvent();
    NN_SDK_ASSERT_NOT_NULL(systemEvent);
    *pEvent = nn::sf::NativeHandle(
        nn::os::DetachReadableHandleOfSystemEvent(systemEvent),
        true // 呼び出し元で管理されるようにする
    );

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::RequestPostEvents(nn::sf::Out<int> outCount, nn::sf::OutArray<detail::service::watcher::EventData> pEventData) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());
    auto& manager = g_pWatcher->GetNetworkManager();
    NN_RESULT_THROW_UNLESS(manager.IsPairingActive(), nn::pctl::ResultPairingNotActive());

    g_pWatcher->GetWatcherEventManager().PreparePostEvent();

    size_t tmpCount = 0;
    NN_UTIL_SCOPE_EXIT
    {
        *outCount = static_cast<uint32_t>(tmpCount);
    };

    service::common::AsyncContext* pContext;
    NN_RESULT_DO(
        manager.RequestPostEvents(
            &pContext,
            manager.GetSavedDeviceId(),
            &tmpCount,
            pEventData.GetData(),
            pEventData.GetLength()
        )
    );

    AsyncData asyncData;

    PushAsyncData(&asyncData, pContext, nullptr);
    auto systemEvent = pContext->GetSystemEvent();
    NN_SDK_ASSERT_NOT_NULL(systemEvent);

    nn::os::WaitSystemEvent(systemEvent);

    pContext = PopAsyncData(nullptr, static_cast<const AsyncData>(asyncData));
    // 戻り値が nullptr の場合はキャンセルされている可能性あり
    NN_RESULT_THROW_UNLESS(pContext != nullptr, nn::pctl::ResultCanceled());

    // 関数内で待機
    NN_RESULT_TRY(WaitAsyncContext(pContext))
        // このエラーの場合は連携情報が削除されていることを表すのでローカルデータの削除処理も行う
        NN_RESULT_CATCH(nn::pctl::ResultServerResourceIsNotFound)
        {
            g_pWatcher->GetNetworkManager().HandlePairingDeletedStatus();
            NN_RESULT_THROW(nn::pctl::ResultPairingDeleted());
        }
        // ResultServerResourceIsNotFound と同様のハンドリングをする
        // (ただしResult自体はそのまま返す)
        NN_RESULT_CATCH(nn::pctl::ResultServerResourceNotAvailable)
        {
            g_pWatcher->GetNetworkManager().HandlePairingDeletedStatus();
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

// deprecated
nn::Result ParentalControlServiceImpl::FinishSynchronizeParentalControlSettings(const nn::pctl::detail::AsyncData& asyncData) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    auto pContext = PopAsyncData(nullptr, asyncData);
    // 戻り値が nullptr の場合はキャンセルされている可能性あり
    NN_RESULT_THROW_UNLESS(pContext != nullptr, nn::pctl::ResultCanceled());

    // 関数内で待機
    NN_RESULT_TRY(WaitAsyncContext(pContext))
        // このエラーの場合は連携情報が削除されていることを表すのでローカルデータの削除処理も行う
        NN_RESULT_CATCH(nn::pctl::ResultServerResourceIsNotFound)
        {
            g_pWatcher->GetNetworkManager().HandlePairingDeletedStatus();
            NN_RESULT_THROW(nn::pctl::ResultPairingDeleted());
        }
        // ResultServerResourceIsNotFound と同様のハンドリングをする
        // (ただしResult自体はそのまま返す)
        NN_RESULT_CATCH(nn::pctl::ResultServerResourceNotAvailable)
        {
            g_pWatcher->GetNetworkManager().HandlePairingDeletedStatus();
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

nn::Result ParentalControlServiceImpl::FinishSynchronizeParentalControlSettingsWithLastUpdated(nn::sf::Out<nn::time::PosixTime> outLastUpdated, nn::pctl::detail::AsyncData asyncData) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_CHECK_CAPABILITY(m_capability, ipc::Capability::Capability_Watcher);
    // Watcherが初期化されていない == この処理が不要なモードで動作している
    NN_RESULT_THROW_UNLESS(IsWatcherAvailable(), nn::pctl::ResultInvalidOperation());

    auto pContext = PopAsyncData(nullptr, asyncData);
    // 戻り値が nullptr の場合はキャンセルされている可能性あり
    NN_RESULT_THROW_UNLESS(pContext != nullptr, nn::pctl::ResultCanceled());

    // 関数内で待機
    NN_RESULT_TRY(WaitAsyncContext(pContext))
        // このエラーの場合は連携情報が削除されていることを表すのでローカルデータの削除処理も行う
        NN_RESULT_CATCH(nn::pctl::ResultServerResourceIsNotFound)
        {
            g_pWatcher->GetNetworkManager().HandlePairingDeletedStatus();
            NN_RESULT_THROW(nn::pctl::ResultPairingDeleted());
        }
        // ResultServerResourceIsNotFound と同様のハンドリングをする
        // (ただしResult自体はそのまま返す)
        NN_RESULT_CATCH(nn::pctl::ResultServerResourceNotAvailable)
        {
            g_pWatcher->GetNetworkManager().HandlePairingDeletedStatus();
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    if (outLastUpdated.GetPointer() != nullptr)
    {
        auto isValidUpdatedTime = g_pWatcher->GetNetworkManager().GetSettingsUpdatedTime(outLastUpdated.GetPointer());
        NN_SDK_ASSERT(isValidUpdatedTime, "Settings are retrieved but the last updated time is not valid");
        NN_UNUSED(isValidUpdatedTime);
    }

    NN_RESULT_SUCCESS;
}

// deprecated
nn::Result ParentalControlServiceImpl::ConfirmPlayableApplicationVideoOld(const nn::sf::InArray<std::int8_t>& ratingAge) NN_NOEXCEPT
{
    NN_RESULT_DO(ConfirmPlayableApplicationVideo(nn::ncm::ApplicationId::GetInvalidId(), ratingAge));

    NN_RESULT_SUCCESS;
}

void ParentalControlServiceImpl::PushAsyncData(AsyncData* outData, common::AsyncContext* pContext, PairingInfoDataHolder* pDataHolder) NN_NOEXCEPT
{
    int index = m_AsyncDataInProgress.nextTargetIndex;
    ++m_AsyncDataInProgress.nextTargetIndex;
    if (m_AsyncDataInProgress.nextTargetIndex == MaxAsyncContextConcurrencyPerSession)
    {
        m_AsyncDataInProgress.nextTargetIndex = 0;
    }
    auto& target = m_AsyncDataInProgress.asyncContextInProgressArray[index];
    if (target.pContext != nullptr)
    {
        // キャンセルする
        common::AsyncContext* p = target.pContext;
        target.pContext = nullptr;
        if (target.pDataHolder != nullptr)
        {
            delete target.pDataHolder;
        }
        CancelAndCloseAsyncContext(p);
    }

    int32_t id = 0;
    // 非 0 で既存のデータと重複しない id を発行する
    while (NN_STATIC_CONDITION(true))
    {
        nn::os::GenerateRandomBytes(&id, sizeof(id));
        if (id == 0)
        {
            continue;
        }
        bool found = false;
        for (auto& d : m_AsyncDataInProgress.asyncContextInProgressArray)
        {
            if (d.id == id)
            {
                found = true;
                break;
            }
        }
        if (!found)
        {
            break;
        }
    }
    // 登録する
    target.pContext = pContext;
    target.pDataHolder = pDataHolder;
    target.id = id;
    target.reserved = 0;

    outData->id = id;
    outData->index = static_cast<int32_t>(index);
}

common::AsyncContext* ParentalControlServiceImpl::PopAsyncData(PairingInfoDataHolder** ppDataHolder, const AsyncData& data) NN_NOEXCEPT
{
    if (ppDataHolder != nullptr)
    {
        *ppDataHolder = nullptr;
    }
    if (data.index < 0 || data.index >= MaxAsyncContextConcurrencyPerSession)
    {
        return nullptr;
    }
    auto& d = m_AsyncDataInProgress.asyncContextInProgressArray[data.index];
    if (d.id != 0 && d.id == data.id)
    {
        common::AsyncContext* p = d.pContext;
        if (ppDataHolder != nullptr)
        {
            *ppDataHolder = d.pDataHolder;
        }
        d.pContext = nullptr;
        d.id = 0;
        return p;
    }
    return nullptr;
}


bool ParentalControlServiceImpl::DisableAllFeaturesImpl(bool isRecoveryMode) NN_NOEXCEPT
{
    auto isDisabled = g_pMain->GetSettingsManager().IsAllFeaturesDisabled();
    auto result = g_pMain->GetSettingsManager().DisableAllFeatures(isRecoveryMode);
    // Watcherが初期化されていない場合や、呼び出し前に無効化されていた場合はこのif文以下の処理は不要
    // (result (無効化したかどうか)は元から isDisabled が true であっても
    // 無効化処理を行ったという意味で true になることがあるのでこの判定では用いない)
    if (!isDisabled && IsWatcherAvailable())
    {
        // (スレッドも停止する)
        g_pWatcher->GetWatcherEventManager().DisableEventProcess();

        // 元に戻してその状態を書き出す
        if (isRecoveryMode)
        {
            g_pWatcher->GetWatcherEventStorage().DiscardEventsSinceInitialize();
        }
        g_pWatcher->GetWatcherEventStorage().WriteDataToFile();

        // (スレッドも停止する)
        g_pWatcher->GetNetworkManager().DisableNetworkFeature();

        // 設定・状態変更という意味でイベントを発行する
        g_pWatcher->GetWatcherEventManager().GetSynchronizationEvent()->Signal();
    }
    return result;
}

}}}}
