﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/pctl_ApplicationStateManager.h>

#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/ipc/pctl_IpcConfig.h>
#include <nn/pctl/detail/service/pctl_ServiceMain.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/common/pctl_SystemInfo.h>

namespace nn { namespace pctl { namespace detail { namespace service {

void ApplicationStateManager::SetCurrentApplicationInfo(const PctlApplicationInfo& pctlApplicationInfo) NN_NOEXCEPT
{
    auto& states = g_pMain->GetStatesRef();
    NN_STATIC_ASSERT(
        (std::is_same<
            decltype(states.currentApplicationInfo),
            PctlApplicationInfo
        >::value)
    );
    std::memcpy(&states.currentApplicationInfo, &pctlApplicationInfo, sizeof(pctlApplicationInfo));
    states.applicationIdFromPdmEvent = nn::ncm::ApplicationId::GetInvalidId();
    states.isApplicationLaunchTimeValid = false;
    states.isApplicationSuspended = false;
    states.freeCommunicationUsed = false;
    states.stereoVisionUsed = false;
}

void ApplicationStateManager::ClearCurrentApplicationInfo(const PctlApplicationInfo& pctlApplicationInfo) NN_NOEXCEPT
{
    auto& states = g_pMain->GetStatesRef();
    // 「現在起動している」タイトルが自分自身であれば無効化する
    if (states.currentApplicationInfo.processId == pctlApplicationInfo.processId)
    {
        // プレイ中のアプリケーションをリセット
        states.currentApplicationInfo.applicationId = nn::ncm::ApplicationId::GetInvalidId();
        states.currentApplicationInfo.processId = 0;
        states.isApplicationLaunchTimeValid = false;
        states.isApplicationSuspended = false;
        states.freeCommunicationUsed = false;
        states.stereoVisionUsed = false;
    }
}

bool ApplicationStateManager::GetCurrentApplicationInfo(nn::ncm::ApplicationId* outRunningApplication) NN_NOEXCEPT
{
    auto& states = g_pMain->GetStatesRef();

    if (states.currentApplicationInfo.applicationId == nn::ncm::ApplicationId::GetInvalidId())
    {
        return false;
    }

    *outRunningApplication = states.currentApplicationInfo.applicationId;
    return true;
}

void ApplicationStateManager::SetRestrictionTemporaryUnlocked() NN_NOEXCEPT
{
    // ここまでの pdm イベントを収集する
    RecordPlayDataBasedEvents();

    auto& states = g_pMain->GetStatesRef();
    states.temporaryUnlocked = true;
}

void ApplicationStateManager::UnsetRestrictionTemporaryUnlocked() NN_NOEXCEPT
{
    auto& states = g_pMain->GetStatesRef();
    if (!states.temporaryUnlocked)
    {
        return;
    }

    // 一時解除中に発生したアプリケーション開始や終了はイベントを記録しないが、
    // 状態は管理できるように読み取りを行う
    RecordPlayDataBasedEvents();

    states.temporaryUnlocked = false;
}

void ApplicationStateManager::AddApplicationLaunchEvent(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    if (!IsWatcherAvailable())
    {
        return;
    }

    nn::time::PosixTime timestamp;
    if (!common::GetNetworkTime(&timestamp))
    {
        return;
    }

    auto& states = g_pMain->GetStatesRef();

    states.applicationLaunchTimeReal = timestamp;
    states.isApplicationLaunchTimeValid = true;

    // PDMイベントにおけるプレイ中のアプリケーションをセット
    states.applicationIdFromPdmEvent = applicationId;
    states.isApplicationSuspended = false;

    g_pWatcher->GetWatcherEventStorage().InsertApplicationLaunchEvent(applicationId, timestamp);
}

void ApplicationStateManager::AddApplicationTerminateEvent(nn::ncm::ApplicationId applicationId, nn::util::optional<nn::time::PosixTime> timestamp) NN_NOEXCEPT
{
    if (!IsWatcherAvailable())
    {
        return;
    }

    if (timestamp == nn::util::nullopt)
    {
        return;
    }

    auto& states = g_pMain->GetStatesRef();
    if (states.applicationIdFromPdmEvent != applicationId)
    {
        return;
    }

    g_pWatcher->GetWatcherEventStorage().InsertApplicationTerminateEvent(applicationId, *timestamp);

    // プレイ中のアプリケーションをリセット
    states.applicationIdFromPdmEvent = nn::ncm::ApplicationId::GetInvalidId();
    states.isApplicationSuspended = false;
}

void ApplicationStateManager::AddApplicationRejectEvent(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    if (!IsWatcherAvailable())
    {
        return;
    }

    g_pWatcher->GetWatcherEventStorage().AddApplicationRejectedEvent(applicationId);
}

void ApplicationStateManager::AddApplicationSuspendEvent(nn::ncm::ApplicationId applicationId, nn::time::PosixTime timestampPdm) NN_NOEXCEPT
{
    if (!IsWatcherAvailable())
    {
        return;
    }

    auto& states = g_pMain->GetStatesRef();
    if (states.applicationIdFromPdmEvent != applicationId)
    {
        return;
    }

    nn::time::PosixTime timeAt = timestampPdm;
    // このイベントにおいてpdmが持つ時刻は分単位で丸められており、秒単位で持っているアプリケーション開始イベントより
    // 過去になる可能性があるので、その場合は開始イベントの時刻に合わせる
    if (states.isApplicationLaunchTimeValid && timeAt < states.applicationLaunchTimeReal)
    {
        timeAt = states.applicationLaunchTimeReal;
    }

    if (!states.isApplicationSuspended)
    {
        states.isApplicationSuspended = true;
    }

    if (timestampPdm.value != 0)
    {
        g_pWatcher->GetWatcherEventStorage().InsertApplicationSuspendedEvent(applicationId, timeAt);
    }
}

void ApplicationStateManager::AddApplicationResumeEvent(nn::ncm::ApplicationId applicationId, nn::time::PosixTime timestampPdm) NN_NOEXCEPT
{
    if (!IsWatcherAvailable())
    {
        return;
    }

    auto& states = g_pMain->GetStatesRef();
    if (states.applicationIdFromPdmEvent != applicationId)
    {
        return;
    }

    nn::time::PosixTime timeAt = timestampPdm;
    // このイベントにおいてpdmが持つ時刻は分単位で丸められており、秒単位で持っているアプリケーション開始イベントより
    // 過去になる可能性があるので、その場合は開始イベントの時刻に合わせる
    if (states.isApplicationLaunchTimeValid && timeAt < states.applicationLaunchTimeReal)
    {
        timeAt = states.applicationLaunchTimeReal;
    }

    if (states.isApplicationSuspended)
    {
        states.isApplicationSuspended = false;
    }

    if (timestampPdm.value != 0)
    {
        g_pWatcher->GetWatcherEventStorage().InsertApplicationResumedEvent(applicationId, timeAt);
    }
}

void ApplicationStateManager::AddUserOpenEvent(const nn::account::Uid& uid, nn::time::PosixTime timestampPdm) NN_NOEXCEPT
{
    if (!IsWatcherAvailable())
    {
        return;
    }
    if (timestampPdm.value != 0)
    {
        g_pWatcher->GetWatcherEventStorage().InsertUserOpenedEvent(uid, timestampPdm);
    }
}

void ApplicationStateManager::AddUserCloseEvent(const nn::account::Uid& uid, nn::time::PosixTime timestampPdm) NN_NOEXCEPT
{
    if (!IsWatcherAvailable())
    {
        return;
    }
    if (timestampPdm.value != 0)
    {
        g_pWatcher->GetWatcherEventStorage().InsertUserClosedEvent(uid, timestampPdm);
    }
}

void ApplicationStateManager::RecordPlayDataBasedEventsImpl(bool readWithoutRecord, bool readWithoutPairing) NN_NOEXCEPT
{
    // サーバー連携関連機能が使用できない(メンテナンスモード含む)場合は何も行わない
    if (!IsWatcherAvailable())
    {
        return;
    }
    // 未連携時は何もしない
    if (!readWithoutPairing && !g_pWatcher->GetNetworkManager().IsPairingActive())
    {
        return;
    }

    NN_FUNCTION_LOCAL_STATIC( nn::os::SdkMutexType , s_MutexForRecordPlayData , =NN_OS_SDK_MUTEX_INITIALIZER() );
    NN_UTIL_LOCK_GUARD(s_MutexForRecordPlayData);

    auto lastAppEventOffset = g_pWatcher->GetNetworkManager().GetLastPlayDataManagerAppEventOffset();
    auto lastAccountEventOffset = g_pWatcher->GetNetworkManager().GetLastPlayDataManagerAccountEventOffset();

    RecordPlayDataBasedApplicationEvents(lastAppEventOffset, readWithoutRecord);
    RecordPlayDataBasedAccountEvents(lastAccountEventOffset, readWithoutRecord);

    g_pWatcher->GetNetworkManager().UpdateLastPlayDataManagerOffsets(
        lastAppEventOffset,
        lastAccountEventOffset
    );
    NN_DETAIL_PCTL_TRACE("[pctl] Check pdm events done. (last event offset = %lu, last user offset = %lu)\n",
        lastAppEventOffset,
        lastAccountEventOffset);
}

}}}}
