﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/util/util_FormatString.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include <nne/inaxxx/inaxxx_results.h>
#include <nne/inaxxx/inaxxx_types.h>
#include <nne/inaxxx/inaxxx_api.h>

#include <nn/pcm/detail/pcm_Log.h>
#include <nn/pcm/driver/pcm_Api.h>
#include <nn/pcm/driver/pcm_Suspend.h>

#include "detail/pcm_FirmwareDebugSettingsAccessor.h"

namespace {

    nn::pcm::driver::detail::FirmwareDebugSettingsAccessor g_FwdbgAccessor;

    nne::inaxxx::InaSession g_DeviceSessions[nn::pcm::MeasuringPoint_Number];
    bool g_IsDeviceSessionOpen[nn::pcm::MeasuringPoint_Number] = { false };

    nne::inaxxx::InaSession& GetInaSession(nn::pcm::MeasuringPoint point) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(0 <= point && point < nn::pcm::MeasuringPoint_Number);
        return g_DeviceSessions[static_cast<int>(point)];
    }
    bool IsDeviceSessionOpen(nn::pcm::MeasuringPoint point) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(0 <= point && point < nn::pcm::MeasuringPoint_Number);
        return g_IsDeviceSessionOpen[static_cast<int>(point)];
    }

    // 参照カウントを守る Mutex
    struct StaticMutex
    {
        nn::os::MutexType mutex;
        void lock() NN_NOEXCEPT
        {
            nn::os::LockMutex(&mutex);
        }
        void unlock() NN_NOEXCEPT
        {
            nn::os::UnlockMutex(&mutex);
        }
    } g_DeviceAccessMutex = { NN_OS_MUTEX_INITIALIZER(false) };
}

namespace nn {
namespace pcm {
namespace driver {

void Initialize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_DeviceAccessMutex);
    // nne::inaxxx::Initialize(); // 宣言はあるが定義がなかった

    // 全計測点をオープンしておく
    for (int i = 0; i < nn::pcm::MeasuringPoint_Number; ++i)
    {
        g_IsDeviceSessionOpen[i] = false;

        char settingName[12]; // "devnameNNNN\0"
        nn::util::SNPrintf(settingName, std::extent<decltype(settingName)>::value, "devname%d", i);

        char devName[64];
        if (g_FwdbgAccessor.ReadSettingString(devName, std::extent<decltype(devName)>::value, settingName) &&
            devName[0] != '\0' // 空文字列でない
            )
        {
            auto result = nne::inaxxx::OpenSession(&g_DeviceSessions[i], devName);
            if (result == nne::inaxxx::Result_Success)
            {
                // セッションオープン成功をもって利用可能とする
                // 失敗時は未実装のポイントか I2C 通信に失敗しているので、何もしない。
                g_IsDeviceSessionOpen[i] = true;
            }
        }
    }
}

void Finalize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_DeviceAccessMutex);

    // 全計測点を閉じる
    for (int i = 0; i < nn::pcm::MeasuringPoint_Number; ++i)
    {
        if (g_IsDeviceSessionOpen[i])
        {
            nne::inaxxx::CloseSession(&g_DeviceSessions[i]);
        }
    }

    // nne::inaxxx::Finalize(); // 宣言はあるが定義がなかった
}

bool IsServiceEnabled() NN_NOEXCEPT
{
    bool isServiceEnabled = true;
    g_FwdbgAccessor.ReadSetting(&isServiceEnabled, "enable");
    return isServiceEnabled;
}

bool IsSupported(MeasuringPoint point) NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_DeviceAccessMutex);

    return IsDeviceSessionOpen(point);
}

int ReadCurrentPower(MeasuringPoint point) NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_DeviceAccessMutex);

    // 非実装の計測ポイントでは常に 0 mW を返す
    if(!IsDeviceSessionOpen(point))
    {
        return 0;
    }

    nne::inaxxx::Power powerUw; // Power in micro watt
    nne::inaxxx::Result result = nne::inaxxx::get_bus_power(&GetInaSession(point), nne::inaxxx::CHANNEL_0, &powerUw);

    // Success でなかったら、ログを出し、0 mW を返す
    if(result != nne::inaxxx::Result_Success)
    {
        NN_DETAIL_PCM_ERROR("Warning! Could not get current power (%d). Please retry.\n", result);
        return 0;
    }
    //NN_DETAIL_PCM_TRACE_V1("powerUw=%d\n", powerUw);
    return (powerUw / 1000); // Conversion to milli watt
}

int ReadCurrentVoltage(MeasuringPoint point) NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_DeviceAccessMutex);

    // 非実装の計測ポイントでは常に 0 mV を返す
    if(!IsDeviceSessionOpen(point))
    {
        return 0;
    }

    nne::inaxxx::Voltage voltageUv; // Voltage in micro volt
    nne::inaxxx::Result result = nne::inaxxx::get_bus_voltage(&GetInaSession(point), nne::inaxxx::CHANNEL_0, &voltageUv);

    // Success でなかったら、ログを出し、0 mV を返す
    if(result != nne::inaxxx::Result_Success)
    {
        NN_DETAIL_PCM_ERROR("Warning! Could not get current voltage (%d). Please retry.\n", result);
        return 0;
    }
    //NN_DETAIL_PCM_TRACE_V1("voltageUv=%d\n", voltageUv);
    return (voltageUv / 1000); // Conversion to milli volt
}

void SuspendAllMeasuringPoints() NN_NOEXCEPT
{
    nne::inaxxx::Result result = nne::inaxxx::Suspend();
    // ログだけ残して先に進む
    if (result != nne::inaxxx::Result_Success)
    {
        NN_DETAIL_PCM_ERROR("Warning! Could not suspend inaxxx (%d)\n", result);
    }
}

void ResumeAllMeasuringPoints() NN_NOEXCEPT
{
    nne::inaxxx::Result result = nne::inaxxx::Resume();
    // ログだけ残して先に進む
    if (result != nne::inaxxx::Result_Success)
    {
        NN_DETAIL_PCM_ERROR("Warning! Could not suspend inaxxx (%d)\n", result);
    }
}

} // driver
} // pcm
} // nn
