﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/sf/impl/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/pcie/pcie.h>
#include <nn/pcie/driver/pcie.h>
#include "../detail/pcie_Macros.h"
#include "../detail/pcie_ServiceName.h"
#include "pcie_ManagerImpl.h"

namespace nn { namespace pcie { namespace server {

class ManagerImpl::SessionImpl
{
    friend ManagerImpl;
public:
    SessionImpl(ManagerImpl* pParent, nn::pcie::ClassDriverConfig *pConfig,
                nn::sf::NativeHandle&& clientProcessHandle,
                nn::sf::Out<nn::sf::NativeHandle> registrationEventHandle) NN_NOEXCEPT
        : m_Parent(pParent, true), m_ClassDriverHandle(0), m_Config(*pConfig),
        m_ClientProcessHandle(std::move(clientProcessHandle)), m_FunctionCount(0),
        m_RegistrationEvent(nn::os::EventClearMode_AutoClear, true),
        m_PmEvent(nn::os::EventClearMode_AutoClear, true),
        m_IrqEvent(nn::os::EventClearMode_AutoClear, true)
    {
        NN_UNUSED(m_Config);
        NN_PCIE_LOG_INFO("ManagerImpl::SessionImpl(), free mem 0x%x\n", nn::pcie::detail::MemoryGetTotalFreeSize());
        registrationEventHandle.Set(nn::sf::NativeHandle(m_RegistrationEvent.GetReadableHandle(), false));
    }
    ~SessionImpl()
    {
        nn::pcie::driver::UnregisterClassDriver(m_ClassDriverHandle);
        m_RegistrationEvent.Signal();
        NN_PCIE_LOG_INFO("ManagerImpl::~SessionImpl(), free mem 0x%x\n", nn::pcie::detail::MemoryGetTotalFreeSize());
    }
    nn::Result QueryFunctions(const nn::sf::OutBuffer& output,nn::sf::Out<int32_t> returnedCount) NN_NOEXCEPT;
    nn::Result AcquireFunction(nn::sf::Out<nn::sf::NativeHandle> pmEventHandle, FunctionHandle functionHandle) NN_NOEXCEPT;
    nn::Result ReleaseFunction(FunctionHandle functionHandle) NN_NOEXCEPT;
    nn::Result ResetFunction(FunctionHandle functionHandle, FunctionResetOptionMask FunctionResetOptionMask) NN_NOEXCEPT;
    nn::Result GetFunctionState(const nn::sf::OutBuffer& dataOut, FunctionHandle functionHandle) NN_NOEXCEPT;
    nn::Result GetBarProfile(nn::sf::Out<uint64_t> outBase, nn::sf::Out<uint64_t> outSize,
                             nn::sf::Out<RegionFlagMask> outRegionFlagMask, FunctionHandle functionHandle, uint8_t bar) NN_NOEXCEPT;
    nn::Result ReadConfig(nn::sf::Out<uint32_t> dataOut, FunctionHandle functionHandle, uint32_t offset, BusAccessWidth width) NN_NOEXCEPT;
    nn::Result WriteConfig(uint32_t dataIn, FunctionHandle functionHandle, uint32_t offset, BusAccessWidth width) NN_NOEXCEPT;
    nn::Result ReadBarRegion(const nn::sf::OutBuffer& dataOut, FunctionHandle functionHandle, uint8_t bar, uint32_t offset, BusAccessWidth accessWidth) NN_NOEXCEPT;
    nn::Result WriteBarRegion(const nn::sf::InBuffer& dataIn, FunctionHandle functionHandle, uint8_t bar, uint32_t offset, BusAccessWidth accessWidth) NN_NOEXCEPT;
    nn::Result FindCapability(nn::sf::Out<uint32_t> offsetOut, FunctionHandle functionHandle, CapabilityId capabilityId) NN_NOEXCEPT;
    nn::Result FindExtendedCapability(nn::sf::Out<uint32_t> offsetOut, FunctionHandle functionHandle, ExtendedCapabilityId extendedCapabilityId) NN_NOEXCEPT;
    nn::Result MapDma(nn::sf::Out<nn::pcie::BusAddress> busAddrOut, FunctionHandle functionHandle, DmaDirection direction, uint64_t procVa, uint32_t size) NN_NOEXCEPT;
    nn::Result UnmapDmaByProcVa(FunctionHandle functionHandle, uint64_t procVa) NN_NOEXCEPT;
    nn::Result UnmapDmaByBusAddress(FunctionHandle functionHandle, nn::pcie::BusAddress busAddr) NN_NOEXCEPT;
    nn::Result GetDmaBusAddress(nn::sf::Out<nn::pcie::BusAddress> busAddrOut, FunctionHandle functionHandle,  uint64_t procVa, uint32_t size) NN_NOEXCEPT;
    nn::Result GetDmaBusAddressRange(nn::sf::Out<nn::pcie::BusAddress> outBase, nn::sf::Out<nn::pcie::BusAddress> outSize, FunctionHandle functionHandle) NN_NOEXCEPT;
    nn::Result SetDmaEnable(FunctionHandle functionHandle, bool isBusMasterEnabled) NN_NOEXCEPT;
    nn::Result AcquireIrq(nn::sf::Out<nn::sf::NativeHandle> irqEventHandle, FunctionHandle functionHandle, IrqType irqType) NN_NOEXCEPT;
    nn::Result ReleaseIrq(FunctionHandle functionHandle) NN_NOEXCEPT;
    nn::Result SetIrqEnable(FunctionHandle functionHandle, int32_t zeroBasedIrqNumber, bool isIrqEnabled) NN_NOEXCEPT;
    static void PmStateCallback(uintptr_t classDriverContext, uintptr_t functionContext, PmState pmState) NN_NOEXCEPT;
    static void IrqCallback(uintptr_t functionContext, int32_t zeroBasedIrqNumber) NN_NOEXCEPT;
    nn::Result SetAspmEnable(FunctionHandle functionHandle, bool isEnabled) NN_NOEXCEPT;
    nn::Result SetResetUponResumeEnable(FunctionHandle functionHandle, bool isEnabled) NN_NOEXCEPT;

private:
    void SetHandle(nn::pcie::ClassDriverHandle classDriverHandle) {m_ClassDriverHandle = classDriverHandle;}
    void SignalRegistrationEvent() {m_RegistrationEvent.Signal();}
    void SignalPmEvent() {m_PmEvent.Signal();}
    void SignalIrqEvent(int32_t zeroBasedIrqNumber) {NN_UNUSED(zeroBasedIrqNumber); m_IrqEvent.Signal();}
    nn::sf::SharedPointer<ManagerImpl>   m_Parent;
    nn::pcie::ClassDriverHandle          m_ClassDriverHandle;
    nn::pcie::ClassDriverConfig          m_Config;
    nn::sf::NativeHandle                 m_ClientProcessHandle;
    int32_t                              m_FunctionCount;
    nn::os::SystemEvent                  m_RegistrationEvent;
    nn::os::SystemEvent                  m_PmEvent;
    nn::os::SystemEvent                  m_IrqEvent;
};

nn::Result ManagerImpl::SessionImpl::QueryFunctions(const nn::sf::OutBuffer& output,nn::sf::Out<int32_t> returnedCount) NN_NOEXCEPT
{
    nn::Result result;
    int32_t count = 0;
    result = nn::pcie::driver::QueryFunctions(m_ClassDriverHandle,
                                              reinterpret_cast<FunctionState*>(output.GetPointerUnsafe()),
                                              output.GetSize(), &count);
    returnedCount.Set(count);
    return result;
}

nn::Result ManagerImpl::SessionImpl::AcquireFunction(nn::sf::Out<nn::sf::NativeHandle> pmEventHandle, FunctionHandle functionHandle) NN_NOEXCEPT
{
    nn::Result result = ResultSuccess();

    // At this time we allow single class driver registration to acquire only one device function
    if(m_FunctionCount > 0)
    {
        return ResultMaximumExceeded();
    }

    if((result = nn::pcie::driver::AcquireFunction(m_ClassDriverHandle, functionHandle,
                                                   reinterpret_cast<uintptr_t>(this))).IsSuccess())
    {
        pmEventHandle.Set(nn::sf::NativeHandle(m_PmEvent.GetReadableHandle(), false));
    }
    return result;
}

nn::Result ManagerImpl::SessionImpl::ReleaseFunction(FunctionHandle functionHandle) NN_NOEXCEPT
{
    nn::Result result = ResultSuccess();
    if((result = nn::pcie::driver::ReleaseFunction(m_ClassDriverHandle, functionHandle)).IsSuccess())
    {
        m_FunctionCount--;
    }
    return result;
}

nn::Result ManagerImpl::SessionImpl::ResetFunction(FunctionHandle functionHandle, FunctionResetOptionMask FunctionResetOptionMask) NN_NOEXCEPT
{
    nn::Result result;
    result = nn::pcie::driver::ResetFunction(m_ClassDriverHandle, functionHandle, FunctionResetOptionMask);
    return result;
}

nn::Result ManagerImpl::SessionImpl::GetFunctionState(const nn::sf::OutBuffer& dataOut, FunctionHandle functionHandle) NN_NOEXCEPT
{
    nn::Result result;
    result = nn::pcie::driver::GetFunctionState(m_ClassDriverHandle, functionHandle,
                                                reinterpret_cast<FunctionState*>(dataOut.GetPointerUnsafe()));
    return result;
}

nn::Result ManagerImpl::SessionImpl::GetBarProfile(nn::sf::Out<uint64_t> outBase, nn::sf::Out<uint64_t> outSize,
                                                   nn::sf::Out<RegionFlagMask> outRegionFlagMask, FunctionHandle functionHandle, uint8_t bar) NN_NOEXCEPT
{
    nn::Result result;
    nn::pcie::driver::DriverBarProfile barProfile = {nullptr, 0, 0, 0};
    result = nn::pcie::driver::GetBarProfile(m_ClassDriverHandle, functionHandle, bar, &barProfile);
    *outBase = barProfile.base;
    *outSize = barProfile.size;
    *outRegionFlagMask = barProfile.flags;
    return result;
}

nn::Result ManagerImpl::SessionImpl::ReadConfig(nn::sf::Out<uint32_t> dataOut, FunctionHandle functionHandle, uint32_t offset, BusAccessWidth width) NN_NOEXCEPT
{
    nn::Result result;
    switch(width)
    {
    case BusAccessWidth_8Bit:
    {
        uint8_t data8 = 0;
        result = nn::pcie::driver::ReadConfig8(m_ClassDriverHandle, functionHandle, offset, &data8);
        dataOut.Set(static_cast<uint32_t>(data8));
        break;
    }
    case BusAccessWidth_16Bit:
    {
        uint16_t data16 = 0;
        result = nn::pcie::driver::ReadConfig16(m_ClassDriverHandle, functionHandle, offset, &data16);
        dataOut.Set(static_cast<uint32_t>(data16));
        break;
    }
    case BusAccessWidth_32Bit:
    {
        uint32_t data32 = 0;
        result = nn::pcie::driver::ReadConfig32(m_ClassDriverHandle, functionHandle, offset, &data32);
        dataOut.Set(static_cast<uint32_t>(data32));
        break;
    }
    default:
        result = ResultInvalidArgument();
        break;
    }
    return result;
}

nn::Result ManagerImpl::SessionImpl::WriteConfig(uint32_t dataIn, FunctionHandle functionHandle, uint32_t offset, BusAccessWidth width) NN_NOEXCEPT
{
    nn::Result result;
    switch(width)
    {
    case BusAccessWidth_8Bit:
        result = nn::pcie::driver::WriteConfig8(m_ClassDriverHandle, functionHandle, offset, static_cast<const uint8_t>(dataIn));
        break;
    case BusAccessWidth_16Bit:
        result = nn::pcie::driver::WriteConfig16(m_ClassDriverHandle, functionHandle, offset, static_cast<const uint16_t>(dataIn));
        break;
    case BusAccessWidth_32Bit:
        result = nn::pcie::driver::WriteConfig32(m_ClassDriverHandle, functionHandle, offset, static_cast<const uint32_t>(dataIn));
        break;
    default:
        result = ResultInvalidArgument();
        break;
    }
    return result;
}

nn::Result ManagerImpl::SessionImpl::ReadBarRegion(const nn::sf::OutBuffer& dataOut, FunctionHandle functionHandle, uint8_t bar,
                                                   uint32_t offset, BusAccessWidth accessWidth) NN_NOEXCEPT
{
    nn::Result result;
    result = nn::pcie::driver::ReadBarRegion(m_ClassDriverHandle, functionHandle, bar, offset, accessWidth,
                                             dataOut.GetSize(), reinterpret_cast<void*>(dataOut.GetPointerUnsafe()));
    return result;
}

nn::Result ManagerImpl::SessionImpl::WriteBarRegion(const nn::sf::InBuffer& dataIn, FunctionHandle functionHandle, uint8_t bar,
                                                    uint32_t offset, BusAccessWidth accessWidth) NN_NOEXCEPT
{
    nn::Result result;
    result = nn::pcie::driver::WriteBarRegion(m_ClassDriverHandle, functionHandle, bar, offset, accessWidth,
                                              dataIn.GetSize(), reinterpret_cast<const void*>(dataIn.GetPointerUnsafe()));
    return result;
}

nn::Result ManagerImpl::SessionImpl::FindCapability(nn::sf::Out<uint32_t> offsetOut, FunctionHandle functionHandle, CapabilityId capabilityId) NN_NOEXCEPT
{
    nn::Result result;
    size_t offset = 0;
    result = nn::pcie::driver::FindCapability(m_ClassDriverHandle, functionHandle, capabilityId,  &offset);
    *offsetOut = static_cast<uint32_t>(offset);
    return result;
}

nn::Result ManagerImpl::SessionImpl::FindExtendedCapability(nn::sf::Out<uint32_t> offsetOut, FunctionHandle functionHandle, ExtendedCapabilityId extendedCapabilityId) NN_NOEXCEPT
{
    nn::Result result;
    size_t offset = 0;
    result = nn::pcie::driver::FindExtendedCapability(m_ClassDriverHandle, functionHandle, extendedCapabilityId, &offset);
    *offsetOut = static_cast<uint32_t>(offset);
    return result;
}

nn::Result ManagerImpl::SessionImpl::MapDma(nn::sf::Out<nn::pcie::BusAddress> busAddrOut, FunctionHandle functionHandle, DmaDirection direction,
                                            uint64_t procVa, uint32_t size) NN_NOEXCEPT
{
    nn::Result result;
    result = nn::pcie::driver::MapDma(m_ClassDriverHandle, functionHandle, direction, m_ClientProcessHandle.GetOsHandle(),
                                      reinterpret_cast<void*>(procVa), static_cast<size_t>(size), &(*busAddrOut));
    return result;
}

nn::Result ManagerImpl::SessionImpl::UnmapDmaByProcVa(FunctionHandle functionHandle, uint64_t procVa) NN_NOEXCEPT
{
    nn::Result result;
    result = nn::pcie::driver::UnmapDma(m_ClassDriverHandle, functionHandle, m_ClientProcessHandle.GetOsHandle(), reinterpret_cast<void*>(procVa));
    return result;
}

nn::Result ManagerImpl::SessionImpl::UnmapDmaByBusAddress(FunctionHandle functionHandle, nn::pcie::BusAddress busAddr) NN_NOEXCEPT
{
    nn::Result result;
    result = nn::pcie::driver::UnmapDma(m_ClassDriverHandle, functionHandle, busAddr);
    return result;
}

nn::Result ManagerImpl::SessionImpl::GetDmaBusAddress(nn::sf::Out<nn::pcie::BusAddress> busAddrOut, FunctionHandle functionHandle,
                                                      uint64_t procVa, uint32_t size) NN_NOEXCEPT
{
    nn::Result result;
    result = nn::pcie::driver::GetDmaBusAddress(m_ClassDriverHandle, functionHandle, m_ClientProcessHandle.GetOsHandle(), reinterpret_cast<void*>(procVa),
                                             static_cast<size_t>(size), &(*busAddrOut));
    return result;
}

nn::Result ManagerImpl::SessionImpl::GetDmaBusAddressRange(nn::sf::Out<nn::pcie::BusAddress> outBase,
                                                           nn::sf::Out<nn::pcie::BusAddress> outSize,
                                                           FunctionHandle functionHandle) NN_NOEXCEPT
{
    nn::Result result;
    result = nn::pcie::driver::GetDmaBusAddressRange(m_ClassDriverHandle,
                                                     functionHandle,
                                                     &(*outBase),
                                                     &(*outSize));
    return result;
}

nn::Result ManagerImpl::SessionImpl::SetDmaEnable(FunctionHandle functionHandle, bool isBusMasterEnabled) NN_NOEXCEPT
{
    nn::Result result;
    result = nn::pcie::driver::SetDmaEnable(m_ClassDriverHandle, functionHandle, isBusMasterEnabled);
    return result;
}

nn::Result ManagerImpl::SessionImpl::AcquireIrq(nn::sf::Out<nn::sf::NativeHandle> irqEventHandle, FunctionHandle functionHandle, IrqType irqType) NN_NOEXCEPT
{
    nn::Result result;
    if((result = nn::pcie::driver::AcquireIrq(m_ClassDriverHandle, functionHandle, ManagerImpl::SessionImpl::IrqCallback,
                                              reinterpret_cast<uintptr_t>(this), irqType, 1)).IsSuccess())
    {
        irqEventHandle.Set(nn::sf::NativeHandle(m_IrqEvent.GetReadableHandle(), false));
    }
    return result;
}

nn::Result ManagerImpl::SessionImpl::ReleaseIrq(FunctionHandle functionHandle) NN_NOEXCEPT
{
    nn::Result result;
    result = nn::pcie::driver::ReleaseIrq(m_ClassDriverHandle, functionHandle);
    return result;
}

nn::Result ManagerImpl::SessionImpl::SetIrqEnable(FunctionHandle functionHandle, int32_t zeroBasedIrqNumber, bool isIrqEnabled) NN_NOEXCEPT
{
    nn::Result result;
    result = nn::pcie::driver::SetIrqEnable(m_ClassDriverHandle, functionHandle,zeroBasedIrqNumber, isIrqEnabled);
    return result;
}

void ManagerImpl::SessionImpl::PmStateCallback(uintptr_t classDriverContext, uintptr_t functionContext, PmState pmState) NN_NOEXCEPT
{
    NN_UNUSED(classDriverContext);
    NN_UNUSED(pmState);
    ManagerImpl::SessionImpl *pSession = reinterpret_cast<ManagerImpl::SessionImpl*>(functionContext);
    NN_PCIE_LOG_VERBOSE("ManagerImpl::SessionImpl::PmStateCallback(pmState=%d)\n", pmState);
    pSession->SignalPmEvent();
}

void ManagerImpl::SessionImpl::IrqCallback(uintptr_t functionContext, int32_t zeroBasedIrqNumber) NN_NOEXCEPT
{
    ManagerImpl::SessionImpl* pSession = reinterpret_cast<ManagerImpl::SessionImpl*>(functionContext);
    pSession->SignalIrqEvent(zeroBasedIrqNumber);
}

nn::Result ManagerImpl::SessionImpl::SetAspmEnable(FunctionHandle functionHandle, bool isEnabled) NN_NOEXCEPT
{
    nn::Result result;
    result = nn::pcie::driver::SetAspmEnable(m_ClassDriverHandle, functionHandle, isEnabled);
    return result;
}

nn::Result ManagerImpl::SessionImpl::SetResetUponResumeEnable(FunctionHandle functionHandle, bool isEnabled) NN_NOEXCEPT
{
    nn::Result result;
    result = nn::pcie::driver::SetResetUponResumeEnable(m_ClassDriverHandle, functionHandle, isEnabled);
    return result;
}

ManagerImpl::ManagerImpl() NN_NOEXCEPT
{
    NN_PCIE_LOG_VERBOSE("ManagerImpl::ManagerImpl()\n");

    // Create the dedicated heap used for creation of ManagerImpl::SessionImpl objects, created for every registered class driver
    m_HeapHandle = nn::lmem::CreateExpHeap(&m_HeapBuffer, sizeof(m_HeapBuffer), nn::lmem::CreationOption_ThreadSafe);
    m_Allocator.Attach(m_HeapHandle);
}

ManagerImpl::~ManagerImpl() NN_NOEXCEPT
{
    NN_PCIE_LOG_VERBOSE("ManagerImpl::~ManagerImpl()\n");
    nn::lmem::DestroyExpHeap(m_HeapHandle);
}

nn::Result ManagerImpl::RegisterClassDriver(nn::sf::Out<nn::sf::SharedPointer<nn::pcie::detail::ISession>> outSession,
                                            nn::sf::Out<nn::sf::NativeHandle> registrationEventHandle,
                                            ClassDriverConfig config,
                                            nn::sf::NativeHandle clientProcessHandle) NN_NOEXCEPT
{
    typedef nn::sf::ObjectFactory<MyAllocator::Policy>  Factory;
    nn::Result result = ResultSuccess();
    nn::pcie::ClassDriverHandle classDriverHandle = 0;
    nn::pcie::ClassDriverConfig workingConfig = config;
    auto p = Factory::CreateSharedEmplaced<detail::ISession, ManagerImpl::SessionImpl>(&m_Allocator, this, &config,
                                                                                       std::move(clientProcessHandle),
                                                                                       registrationEventHandle);
    clientProcessHandle.Detach();
    workingConfig.classDriverContext = reinterpret_cast<uint64_t>(Factory::GetEmplacedImplPointer<ManagerImpl::SessionImpl>(p));
    if((result=nn::pcie::driver::RegisterClassDriver(&classDriverHandle, &workingConfig, RegistrationCallback,
                                                     ManagerImpl::SessionImpl::PmStateCallback)).IsSuccess())
    {
        p.GetImpl().SetHandle(classDriverHandle);
        *outSession = std::move(p);
    }
    else
    {
        p.Reset();
    }
    return result;
}

nn::Result ManagerImpl::QueryFunctions(const nn::sf::OutBuffer& output,nn::sf::Out<int32_t> returnedCount) NN_NOEXCEPT
{
    nn::Result result;
    int32_t count = 0;
    result = nn::pcie::driver::QueryFunctions(reinterpret_cast<FunctionState*>(output.GetPointerUnsafe()),
                                              output.GetSize(), &count);
    returnedCount.Set(count);
    return result;
}

void ManagerImpl::RegistrationCallback(uintptr_t classDriverContext) NN_NOEXCEPT
{
    ManagerImpl::SessionImpl *pSession = reinterpret_cast<ManagerImpl::SessionImpl*>(classDriverContext);
    pSession->SignalRegistrationEvent();
}

} // namespace server
} // namespace pci
} // namespace nn
