﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Mutex.h>
#include <nn/sf/sf_Types.h>                  // for nn::sf::SharedPointer
#include <nn/sf/sf_NativeHandle.h>
#include <nn/dd.h>

#include <nn/pcie/pcie.h>
#include <nn/pcie/driver/pcie_Types.h>

#include "detail/pcie_ServiceName.h"
#include "pcie_CreateManager.h"

#define NN_PCIE_ACCESS_BY_HIPC

namespace nn { namespace pcie {

namespace {

// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
// DFC と HIPC とで共通でよい。
nn::sf::SharedPointer<nn::pcie::detail::IManager> g_Manager;

// Initialize の参照カウント
int g_InitializeCount = 0;

// 参照カウントを守る Mutex
nn::os::Mutex g_InitializeCountMutex(false);

}

Result Initialize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_InitializeCountMutex);

    if (g_InitializeCount == 0)
    {

        NN_SDK_ASSERT( !g_Manager);

        // マクロによって、HIPC/DFC を切り替える
#ifdef NN_PCIE_ACCESS_BY_HIPC
        g_Manager = nn::pcie::CreateManagerByHipc();
#else
        g_Manager = nn::pcie::CreateManagerByDfc();
#endif
    }

    g_InitializeCount++;

    return ResultSuccess();
}

Result Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount > 0);

    g_InitializeCount--;

    if (g_InitializeCount == 0)
    {

        NN_SDK_ASSERT(g_Manager);
        // 共有ポインタへの nullptr の代入で解放できる。
        g_Manager = nullptr;
    }

    return ResultSuccess();
}

Result QueryFunctions(FunctionState*    pOutFunctionState,
                      int32_t*          pOutFunctionStateCount,
                      size_t            functionStateBufferSize) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    NN_ABORT_UNLESS_NOT_NULL(pOutFunctionState);
    NN_ABORT_UNLESS_NOT_NULL(pOutFunctionStateCount);
    NN_SDK_ASSERT(g_Manager);
    nn::sf::OutBuffer queryOutput(reinterpret_cast<char *>(pOutFunctionState), functionStateBufferSize);
    result = g_Manager->QueryFunctions(queryOutput, pOutFunctionStateCount);
    return result;
}

Result RegisterClassDriver(ClassDriverHandle*       pOutHandle,
                           nn::os::SystemEventType* pOutEvent,
                           const ClassDriverConfig* pInConfig,
                           nn::os::EventClearMode   eventClearMode) NN_NOEXCEPT
{
    Result result;
    nn::sf::NativeHandle sfEventHandle;
    NN_ABORT_UNLESS_NOT_NULL(pOutHandle);
    NN_ABORT_UNLESS_NOT_NULL(pOutEvent);
    NN_SDK_ASSERT(g_Manager);
    nn::sf::SharedPointer<nn::pcie::detail::ISession> session;
    if((result = g_Manager->RegisterClassDriver(&session, &sfEventHandle, *pInConfig,
                                                nn::sf::NativeHandle(nn::dd::GetCurrentProcessHandle(), false))).IsSuccess())
    {
        *pOutHandle = reinterpret_cast<ClassDriverHandle>(session.Detach());
        nn::os::AttachReadableHandleToSystemEvent(pOutEvent, sfEventHandle.GetOsHandle(), sfEventHandle.IsManaged(), eventClearMode);
        sfEventHandle.Detach();
    }
    return result;
}

Result UnregisterClassDriver(nn::os::SystemEventType* pInEvent,
                             ClassDriverHandle        classDriverHandle) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    NN_UNUSED(pInEvent);
    NN_ABORT_UNLESS_NOT_NULL(pInEvent);
    NN_SDK_ASSERT(g_Manager);
    nn::sf::ReleaseSharedObject(reinterpret_cast<nn::pcie::detail::ISession *>(classDriverHandle));
    nn::os::DestroySystemEvent(pInEvent);
    return result;
}

Result QueryFunctions(FunctionState*    pOutFunctionState,
                      int32_t*          pOutFunctionStateCount,
                      ClassDriverHandle classDriverHandle,
                      size_t            functionStateBufferSize) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pOutFunctionState);
    NN_ABORT_UNLESS_NOT_NULL(pOutFunctionStateCount);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    nn::sf::OutBuffer queryOutput(reinterpret_cast<char *>(pOutFunctionState), functionStateBufferSize);
    result = pS->QueryFunctions(queryOutput, pOutFunctionStateCount);
    return result;
}

Result AcquireFunction(nn::os::SystemEventType* pOutEvent,
                       ClassDriverHandle        classDriverHandle,
                       FunctionHandle           functionHandle,
                       nn::os::EventClearMode   eventClearMode) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::sf::NativeHandle sfEventHandle;
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pOutEvent);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    if((result = pS->AcquireFunction(&sfEventHandle, functionHandle)).IsSuccess())
    {
        nn::os::AttachReadableHandleToSystemEvent(pOutEvent, sfEventHandle.GetOsHandle(), sfEventHandle.IsManaged(), eventClearMode);
        sfEventHandle.Detach();
    }
    return result;
}

Result ReleaseFunction(nn::os::SystemEventType* pInEvent,
                       ClassDriverHandle        classDriverHandle,
                       FunctionHandle           functionHandle) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    NN_ABORT_UNLESS_NOT_NULL(pInEvent);
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pInEvent);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    if((result = pS->ReleaseFunction(functionHandle)).IsSuccess())
    {
        nn::os::DestroySystemEvent(pInEvent);
    }
    return result;
}

Result ResetFunction(ClassDriverHandle classDriverHandle,
                     FunctionHandle functionHandle,
                     FunctionResetOptionMask resetOptionMask) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    result = pS->ResetFunction(functionHandle, resetOptionMask);
    return result;
}

Result GetFunctionState(FunctionState*    pOutFunctionState,
                        ClassDriverHandle classDriverHandle,
                        FunctionHandle    functionHandle) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pOutFunctionState);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    nn::sf::OutBuffer stateOutput(reinterpret_cast<char *>(pOutFunctionState), sizeof(FunctionState));
    result = pS->GetFunctionState(stateOutput, functionHandle);
    return result;
}

Result GetBarProfile(BarProfile*       pOutBarProfile,
                     ClassDriverHandle classDriverHandle,
                     FunctionHandle    functionHandle,
                     uint8_t           bar) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    uint64_t       base = 0ULL;
    uint64_t       size = 0ULL;
    RegionFlagMask flags = 0;
    NN_ABORT_UNLESS_NOT_NULL(pOutBarProfile);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    if((result = pS->GetBarProfile(&base, &size, &flags, functionHandle, bar)).IsSuccess())
    {
        pOutBarProfile->size   = static_cast<size_t>(size);
        pOutBarProfile->flags  = flags;
    }
    return result;
}

Result GetMappedBar(volatile uint8_t** pOutMappedBar,
                    ClassDriverHandle  classDriverHandle,
                    FunctionHandle     functionHandle,
                    uint8_t            bar) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    uint64_t       base = 0ULL;
    uint64_t       size = 0ULL;
    RegionFlagMask flags = 0;
    NN_ABORT_UNLESS_NOT_NULL(pOutMappedBar);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    *pOutMappedBar=NULL;
    if((result = pS->GetBarProfile(&base, &size, &flags, functionHandle, bar)).IsSuccess())
    {
        if((base != 0ULL) && (size != 0))
        {
            *pOutMappedBar = reinterpret_cast<volatile uint8_t*>
                (nn::dd::QueryIoMappingAddress(base, size));
        }
    }
    return result;
}


Result ReadConfig8(uint8_t*          pOutData8,
                   ClassDriverHandle classDriverHandle,
                   FunctionHandle    functionHandle,
                   size_t            offset) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    uint32_t data32 = 0;
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pOutData8);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    result = pS->ReadConfig(&data32, functionHandle, static_cast<uint32_t>(offset), BusAccessWidth_8Bit);
    *pOutData8 = static_cast<uint8_t>(data32);
    return result;
}

Result ReadConfig16(uint16_t*         pOutData16,
                    ClassDriverHandle classDriverHandle,
                    FunctionHandle    functionHandle,
                    size_t            offset) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    uint32_t data32 = 0;
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pOutData16);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    result = pS->ReadConfig(&data32, functionHandle, static_cast<uint32_t>(offset), BusAccessWidth_16Bit);
    *pOutData16 = static_cast<uint16_t>(data32);
    return result;
}

Result ReadConfig32(uint32_t*         pOutData32,
                    ClassDriverHandle classDriverHandle,
                    FunctionHandle    functionHandle,
                    size_t            offset) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    uint32_t data32 = 0;
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pOutData32);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    result = pS->ReadConfig(&data32, functionHandle, static_cast<uint32_t>(offset), BusAccessWidth_32Bit);
    *pOutData32 = static_cast<uint32_t>(data32);
    return result;
}

Result WriteConfig8(uint8_t           data8,
                    ClassDriverHandle classDriverHandle,
                    FunctionHandle    functionHandle,
                    size_t            offset)  NN_NOEXCEPT
{
    Result result = ResultSuccess();
    uint32_t data32 = static_cast<uint32_t>(data8);
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    result = pS->WriteConfig(data32, functionHandle, static_cast<uint32_t>(offset), BusAccessWidth_8Bit);
    return result;
}

Result WriteConfig16(uint16_t          data16,
                     ClassDriverHandle classDriverHandle,
                     FunctionHandle    functionHandle,
                     size_t            offset) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    uint32_t data32 = static_cast<uint32_t>(data16);
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    result = pS->WriteConfig(data32, functionHandle, static_cast<uint32_t>(offset), BusAccessWidth_16Bit);
    return result;
}

Result WriteConfig32(uint32_t          data32,
                     ClassDriverHandle classDriverHandle,
                     FunctionHandle    functionHandle,
                     size_t            offset) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    result = pS->WriteConfig(data32, functionHandle, static_cast<uint32_t>(offset), BusAccessWidth_32Bit);
    return result;
}

Result ReadBarRegion8(uint8_t*          pOutData8,
                      ClassDriverHandle classDriverHandle,
                      FunctionHandle    functionHandle,
                      uint8_t           bar,
                      size_t            offset) NN_NOEXCEPT
{
    return ReadBarRegion(reinterpret_cast<void*>(pOutData8), classDriverHandle, functionHandle, bar,
                         static_cast<uint32_t>(offset), BusAccessWidth_8Bit, sizeof(uint8_t));
}

Result ReadBarRegion16(uint16_t*         pOutData16,
                       ClassDriverHandle classDriverHandle,
                       FunctionHandle    functionHandle,
                       uint8_t           bar,
                       size_t            offset) NN_NOEXCEPT
{
    return ReadBarRegion(reinterpret_cast<void*>(pOutData16), classDriverHandle, functionHandle, bar,
                         static_cast<uint32_t>(offset), BusAccessWidth_16Bit, sizeof(uint16_t));
}

Result ReadBarRegion32(uint32_t*         pOutData32,
                       ClassDriverHandle classDriverHandle,
                       FunctionHandle    functionHandle,
                       uint8_t           bar,
                       size_t            offset) NN_NOEXCEPT
{
    return ReadBarRegion(reinterpret_cast<void*>(pOutData32), classDriverHandle, functionHandle, bar,
                         static_cast<uint32_t>(offset), BusAccessWidth_32Bit, sizeof(uint32_t));
}

Result ReadBarRegion64(uint64_t*         pOutData64,
                       ClassDriverHandle classDriverHandle,
                       FunctionHandle    functionHandle,
                       uint8_t           bar,
                       size_t            offset) NN_NOEXCEPT
{
    return ReadBarRegion( reinterpret_cast<void*>(pOutData64), classDriverHandle, functionHandle, bar,
                          static_cast<uint32_t>(offset), BusAccessWidth_64Bit,sizeof(uint64_t));
}

Result ReadBarRegion(void *            pOutData,
                     ClassDriverHandle classDriverHandle,
                     FunctionHandle    functionHandle,
                     uint8_t           bar,
                     size_t            offset,
                     BusAccessWidth    accessWidth,
                     size_t            readSize) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pOutData);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    nn::sf::OutBuffer dataout(reinterpret_cast<char *>(pOutData), readSize);
    result = pS->ReadBarRegion(dataout, functionHandle, bar, static_cast<uint32_t>(offset), accessWidth);
    return result;
}

Result WriteBarRegion8(uint8_t           data8,
                       ClassDriverHandle classDriverHandle,
                       FunctionHandle    functionHandle,
                       uint8_t           bar,
                       size_t            offset) NN_NOEXCEPT
{
    return WriteBarRegion(reinterpret_cast<const void*>(&data8), classDriverHandle, functionHandle, bar,
                          static_cast<uint32_t>(offset), BusAccessWidth_8Bit, sizeof(uint8_t));
}

Result WriteBarRegion16(uint16_t          data16,
                        ClassDriverHandle classDriverHandle,
                        FunctionHandle    functionHandle,
                        uint8_t           bar,
                        size_t            offset) NN_NOEXCEPT
{
    return WriteBarRegion(reinterpret_cast<const void*>(&data16), classDriverHandle, functionHandle, bar,
                          static_cast<uint32_t>(offset), BusAccessWidth_16Bit, sizeof(uint16_t));
}

Result WriteBarRegion32(uint32_t          data32,
                        ClassDriverHandle classDriverHandle,
                        FunctionHandle    functionHandle,
                        uint8_t           bar,
                        size_t            offset) NN_NOEXCEPT
{
    return WriteBarRegion(reinterpret_cast<const void*>(&data32), classDriverHandle, functionHandle, bar,
                          static_cast<uint32_t>(offset), BusAccessWidth_32Bit, sizeof(uint32_t));
}

Result WriteBarRegion64(uint64_t          data64,
                        ClassDriverHandle classDriverHandle,
                        FunctionHandle    functionHandle,
                        uint8_t           bar,
                        size_t            offset) NN_NOEXCEPT
{
    return WriteBarRegion(reinterpret_cast<const void*>(&data64), classDriverHandle, functionHandle, bar,
                          static_cast<uint32_t>(offset), BusAccessWidth_64Bit, sizeof(uint64_t));
}

Result WriteBarRegion(const void*       pInData,
                      ClassDriverHandle classDriverHandle,
                      FunctionHandle    functionHandle,
                      uint8_t           bar,
                      size_t            offset,
                      BusAccessWidth    accessWidth,
                      size_t            writeSize) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pInData);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    nn::sf::InBuffer dataIn(reinterpret_cast<const char *>(pInData), writeSize);
    result = pS->WriteBarRegion(dataIn, functionHandle, bar, static_cast<uint32_t>(offset), accessWidth);
    return result;
}

Result FindCapability(size_t*           pOutOffset,
                      ClassDriverHandle classDriverHandle,
                      FunctionHandle    functionHandle,
                      CapabilityId      capabilityId) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    uint32_t offset = 0;
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pOutOffset);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    result = pS->FindCapability(&offset, functionHandle, capabilityId);
    *pOutOffset = static_cast<size_t>(offset);
    return result;
}

Result FindExtendedCapability(size_t*              pOutOffset,
                              ClassDriverHandle    classDriverHandle,
                              FunctionHandle       functionHandle,
                              ExtendedCapabilityId extendedCapabilityId) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    uint32_t offset = 0;
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pOutOffset);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    result = pS->FindExtendedCapability(&offset, functionHandle, extendedCapabilityId);
    *pOutOffset = offset;
    return result;
}

Result MapDma(BusAddress*       pOutBusAddress,
              ClassDriverHandle classDriverHandle,
              FunctionHandle    functionHandle,
              DmaDirection      direction,
              void*             base,
              size_t            size) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::pcie::BusAddress busAddr = 0;
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(base);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    result = pS->MapDma(&busAddr, functionHandle, direction, reinterpret_cast<uint64_t>(base), static_cast<uint32_t>(size));
    if(pOutBusAddress!=NULL) {*pOutBusAddress = busAddr;}
    return result;
}

Result UnmapDma(ClassDriverHandle classDriverHandle,
                FunctionHandle    functionHandle,
                void*             base) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(base);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    result = pS->UnmapDmaByProcVa(functionHandle, reinterpret_cast<uint64_t>(base));
    return result;
}

Result UnmapDma(ClassDriverHandle classDriverHandle,
                FunctionHandle    functionHandle,
                BusAddress        busAddress)  NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    result = pS->UnmapDmaByBusAddress(functionHandle, busAddress);
    return result;
}

Result GetDmaBusAddress(BusAddress*       pOutBusAddress,
                        ClassDriverHandle classDriverHandle,
                        FunctionHandle    functionHandle,
                        void*             address,
                        size_t            size)  NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::pcie::BusAddress busAddress = 0;
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pOutBusAddress);
    NN_ABORT_UNLESS_NOT_NULL(address);
    NN_ABORT_UNLESS_NOT_NULL(pOutBusAddress);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    result = pS->GetDmaBusAddress(&busAddress, functionHandle, reinterpret_cast<uint64_t>(address), static_cast<uint32_t>(size));
    *pOutBusAddress = busAddress;
    return result;
}

Result GetDmaBusAddressRange(BusAddress        *pOutBase,
                             BusAddress        *pOutSize,
                             ClassDriverHandle  classDriverHandle,
                             FunctionHandle     functionHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutBase);
    NN_ABORT_UNLESS_NOT_NULL(pOutSize);
    NN_SDK_ASSERT(g_Manager);
    Result result = ResultSuccess();
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    nn::pcie::BusAddress base = 0;
    nn::pcie::BusAddress size = 0;
    result = pS->GetDmaBusAddressRange(&base, &size, functionHandle);
    *pOutBase = base;
    *pOutSize = size;
    return result;
}

Result SetDmaEnable(ClassDriverHandle classDriverHandle,
                    FunctionHandle    functionHandle,
                    bool              isBusMasterEnabled) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    result = pS->SetDmaEnable(functionHandle, isBusMasterEnabled);
    return result;
}

Result AcquireIrq( nn::os::SystemEventType* pOutEvent,
                   ClassDriverHandle        classDriverHandle,
                   FunctionHandle           functionHandle,
                   nn::os::EventClearMode   eventClearMode,
                   IrqType                  irqType) NN_NOEXCEPT
{
    Result result;
    nn::sf::NativeHandle sfEventHandle;
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pOutEvent);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    if((result = pS->AcquireIrq(&sfEventHandle, functionHandle, irqType)).IsSuccess())
    {
        nn::os::AttachReadableHandleToSystemEvent(pOutEvent, sfEventHandle.GetOsHandle(), sfEventHandle.IsManaged(), eventClearMode);
        sfEventHandle.Detach();
    }
    return result;
}

Result ReleaseIrq(nn::os::SystemEventType* pInEvent,
                  ClassDriverHandle        classDriverHandle,
                  FunctionHandle           functionHandle) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    NN_ABORT_UNLESS_NOT_NULL(pInEvent);
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    if((result = pS->ReleaseIrq(functionHandle)).IsSuccess())
    {
        nn::os::DestroySystemEvent(pInEvent);
    }
    return result;
}

Result SetIrqEnable(ClassDriverHandle classDriverHandle,
                    FunctionHandle    functionHandle,
                    int32_t           zeroBasedIrqNumber,
                    bool              isIrqEnabled) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    result = pS->SetIrqEnable(functionHandle, zeroBasedIrqNumber, isIrqEnabled);
    return result;
}

Result SetAspmEnable(ClassDriverHandle classDriverHandle,
                     FunctionHandle    functionHandle,
                     bool              isEnabled) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    result = pS->SetAspmEnable(functionHandle, isEnabled);
    return result;
}

Result SetResetUponResumeEnable(ClassDriverHandle classDriverHandle,
                                FunctionHandle    functionHandle,
                                bool              isEnabled) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::pcie::detail::ISession* pS = reinterpret_cast<nn::pcie::detail::ISession*>(classDriverHandle);
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_SDK_ASSERT(g_Manager);
    result = pS->SetResetUponResumeEnable(functionHandle, isEnabled);
    return result;
}

} // namespace pcie
} // namespace nn
