﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "detail/pcie_PrivateIncludes.h"

namespace nn { namespace pcie { namespace driver {

namespace {
// Instantiation of PCI driver
detail::Driver pciDriver;
}

static Result GetBusDeviceFunction(detail::Bus **ppBus, DeviceNumber *pDevNum, FunctionNumber *pFuncNum,
                                   ClassDriverHandle classDriverHandle, FunctionHandle functionHandle,
                                   detail::Client::SecurityPolicy policy)
{
    Result result;
    detail::EndpointFunction* pEpFunc = NULL;
    if((result = detail::Client::GetEndpointFunction(&pEpFunc, NULL, &pciDriver,
                                                     classDriverHandle, functionHandle, policy)).IsSuccess())
    {
        *ppBus = pEpFunc->GetBus();
        *pDevNum = pEpFunc->GetDevNum();
        *pFuncNum = pEpFunc->GetFuncNum();
    }
    return result;
}

static Result CopyBarRegion(void *pDestination, const void *pSource, BusAccessWidth accessWidth, size_t size)
{
    Result result = ResultSuccess();
    switch (accessWidth)
    {
    case BusAccessWidth_8Bit:
    {
        const volatile uint8_t *pS = reinterpret_cast<const volatile uint8_t *>(pSource);
        uint8_t *pD = reinterpret_cast<uint8_t *>(pDestination);
        for (size_t offset = 0; offset < size; offset++)
        {
            *pD++ = *pS++;
        }
        break;
    }
    case BusAccessWidth_16Bit:
    {
        if ((reinterpret_cast<uintptr_t>(pDestination) & 0x1) || (reinterpret_cast<uintptr_t>(pSource) & 0x1))
        {
            result = ResultInvalidAlignment();
            break;
        }
        const volatile uint16_t *pS = reinterpret_cast<const volatile uint16_t *>(pSource);
        uint16_t *pD = reinterpret_cast<uint16_t *>(pDestination);
        for (size_t offset = 0; offset < size; offset += sizeof(uint16_t))
        {
            *pD++ = *pS++;
        }
        break;
    }
    case BusAccessWidth_32Bit:
    {
        if ((reinterpret_cast<uintptr_t>(pDestination) & 0x3) || (reinterpret_cast<uintptr_t>(pSource) & 0x3))
        {
            result = ResultInvalidAlignment();
            break;
        }
        const volatile uint32_t *pS = reinterpret_cast<const volatile uint32_t *>(pSource);
        uint32_t *pD = reinterpret_cast<uint32_t *>(pDestination);
        for (size_t offset = 0; offset < size; offset += sizeof(uint32_t))
        {
            *pD++ = *pS++;
        }
        break;
    }
    case BusAccessWidth_64Bit:
    {
        if ((reinterpret_cast<uintptr_t>(pDestination) & 0x7) || (reinterpret_cast<uintptr_t>(pSource) & 0x7))
        {
            result = ResultInvalidAlignment();
            break;
        }
        const volatile uint64_t *pS = reinterpret_cast<const volatile uint64_t *>(pSource);
        uint64_t *pD = reinterpret_cast<uint64_t *>(pDestination);
        for (size_t offset = 0; offset < size; offset += sizeof(uint64_t))
        {
            *pD++ = *pS++;
        }
        break;
    }
    default:
        result = ResultInvalidAccessWidth();
        break;
    }
    return result;
}


Result Initialize()
{
    return pciDriver.Initialize();
}

Result Finalize()
{
    return pciDriver.Finalize();
}

Result GetLoggedState(LoggedState* pOutput, bool isCleared)
{
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    detail::RootComplex* pRootComplex = nullptr;
    *pOutput = InvalidLoggedState;
    if((result = pciDriver.GetRootComplex(&pRootComplex)).IsSuccess())
    {
        if((result = pRootComplex->GetLoggedState(&pOutput->rootComplex, isCleared)).IsSuccess())
        {
            int32_t unused = 0;
            result = QueryFunctions(pOutput->functions, sizeof(pOutput->functions), &unused);
        }
    }
    return result;
}

void SetLoggedStateCallback(LoggedStateCallback callback, uintptr_t context)
{
    pciDriver.SetLoggedStateCallback(callback, context);
}

Result QueryFunctions(FunctionState* pOutput, size_t queryBufferSize, int32_t* pNumReturned)
{
    nn::Result result = ResultSuccess();
    FunctionHandle handles[nn::pcie::MaxEnumeratedDeviceFunctions];
    int32_t maxReturned = queryBufferSize / sizeof(FunctionState);
    maxReturned = (maxReturned < nn::pcie::MaxEnumeratedDeviceFunctions) ? maxReturned :
        nn::pcie::MaxEnumeratedDeviceFunctions;
    int32_t handleCount = detail::EndpointFunction::FindEndpointFunctions(handles, &pciDriver, maxReturned,
                                                                          &detail::EndpointFunction::WildcardSearchCriteria,
                                                                          true);
    // For all endpoint functions
    int32_t outputIndex = 0;
    for (int32_t handleIndex = 0; (handleIndex < handleCount) && (outputIndex < maxReturned); handleIndex++)
    {
        detail::EndpointFunction *pEpFunc;
        if ((pEpFunc = detail::EndpointFunction::GetEndpointFunction(&pciDriver, handles[handleIndex])) != NULL)
        {
            pEpFunc->GetState(pOutput + outputIndex);
            outputIndex++;
        }
    }
    if(pNumReturned!=NULL) {*pNumReturned = outputIndex;}
    return result;
}

Result RegisterClassDriver(ClassDriverHandle *pReturnedHandle, const ClassDriverConfig *pCfg,
                           RegistrationCallback registrationCallback, PmStateCallback pmStateCallback)
{
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    detail::Client *pClient = new detail::Client(&pciDriver, pCfg, registrationCallback, pmStateCallback);

    if (pClient)
    {
        result = pClient->Initialize(pReturnedHandle);
    }
    else
    {
        *pReturnedHandle = InvalidClassDriverHandle;
        result = ResultMemAllocFailure();
    }

    if(pClient!=NULL && !result.IsSuccess())
    {
        delete pClient;
        pClient = NULL;
    }

    return result;
}

Result UnregisterClassDriver(ClassDriverHandle classDriverHandle)
{
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    detail::Client *pClient = detail::Client::GetClient(&pciDriver, classDriverHandle);
    if (pClient != NULL)
    {
        pClient->Finalize();
        delete pClient;
    }
    else
    {
        result = ResultInvalidClassDriverHandle();
    }
    return result;
}

Result QueryFunctions(ClassDriverHandle classDriverHandle, FunctionState* pOutput, size_t queryBufferSize,
                      int32_t* pNumReturned)
{
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    detail::Client *pClient =  detail::Client::GetClient(&pciDriver, classDriverHandle);
    if (pClient != NULL)
    {
        int32_t funcCount = pClient->QueryFunctions(pOutput, queryBufferSize);
        if(pNumReturned!=NULL) {*pNumReturned = funcCount;}
    }
    else
    {
        result = ResultInvalidClassDriverHandle();
    }
    return result;
}

Result AcquireFunction(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle, uintptr_t functionContext)
{
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    detail::EndpointFunction* pEpFunc = detail::EndpointFunction::GetEndpointFunction(&pciDriver, functionHandle);

    if(pEpFunc != NULL)
    {
        if(!pEpFunc->IsClientAssigned())
        {
            detail::Client *pClient = NULL;
            if ((result = detail::Client::GetEndpointFunction(NULL, &pClient, &pciDriver, classDriverHandle,
                                                              functionHandle, detail::Client::SecurityPolicy_OpenIfNotAcquired)).IsSuccess())
            {
                result = pClient->AcquireFunction(functionHandle, functionContext);
            }
        }
        else
        {
            result = ResultAlreadyAcquired();
        }
    }
    else
    {
        result = ResultInvalidFunctionHandle();
    }

    return result;
}

Result ReleaseFunction(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle)
{
    detail::Client *pClient= NULL;
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if((pClient = detail::Client::GetClient(&pciDriver, classDriverHandle)) != NULL)
    {
        result = pClient->ReleaseFunction(functionHandle);
    }
    else
    {
        result = ResultInvalidClassDriverHandle();
    }
    return result;
}

Result ResetFunction(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle, FunctionResetOptionMask resetOptionMask)
{
    detail::EndpointFunction *pEpFunc = NULL;
    detail::Client *pClient= NULL;
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = detail::Client::GetEndpointFunction(&pEpFunc, &pClient, &pciDriver, classDriverHandle,
                                                      functionHandle, detail::Client::SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        BusNumber busNumber = pEpFunc->GetBus()->GetBusNum();
        DeviceNumber devNumber = pEpFunc->GetDevNum();

        // Skipping reset upon resume option is no longer relevant if manual reset is requested
        pClient->SetResetUponResumeEnable(functionHandle, true);

        // Try function level reset
        if(FunctionResetOption_DoFunctionLevelReset & resetOptionMask)
        {
            pEpFunc->DoFunctionLevelReset();
        }

        result = pciDriver.ResetDevice(busNumber, devNumber, resetOptionMask);
    }
    return result;
}

Result GetFunctionState(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle, FunctionState* pOutput)
{
    detail::EndpointFunction *pEpFunc = NULL;
    detail::Client *pClient= NULL;
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);

    // Try to get from endpoint
    if ((result = detail::Client::GetEndpointFunction(&pEpFunc, &pClient, &pciDriver, classDriverHandle,
                                                      functionHandle, detail::Client::SecurityPolicy_OpenIfNotAcquired)).IsSuccess())
    {
        pEpFunc->GetState(pOutput);
    }

    // If function is in D3 state, get last state from client
    if(result.IsFailure())
    {
        if((pClient = detail::Client::GetClient(&pciDriver, classDriverHandle)) != NULL)
        {
            result = pClient->GetFunctionState(pOutput, functionHandle);
        }
    }

    return result;
}

Result GetBarProfile(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle,
                     uint8_t bar, DriverBarProfile *pReturnedProfile)
{
    detail::EndpointFunction *pEpFunc = NULL;
    detail::Client *pClient= NULL;
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = detail::Client::GetEndpointFunction(&pEpFunc, &pClient, &pciDriver, classDriverHandle,
                                                      functionHandle, detail::Client::SecurityPolicy_OpenIfNotAcquired)).IsSuccess())
    {
        result = pEpFunc->GetBarProfile(bar, pReturnedProfile);
    }
    return result;
}

Result ReadConfig8(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle,
                   size_t offset, uint8_t *pOut)
{
    Result result;
    detail::Bus *pBus;
    DeviceNumber devNum;
    FunctionNumber funcNum;
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = GetBusDeviceFunction(&pBus, &devNum, &funcNum, classDriverHandle,
                                       functionHandle, detail::Client::SecurityPolicy_OpenIfNotAcquired)).IsSuccess())
    {
        result = pBus->ReadConfigByte(devNum, funcNum, offset, pOut);
    }
    return result;
}

Result ReadConfig16(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle,
                    size_t offset, uint16_t *pOut)
{
    Result result;
    detail::Bus *pBus;
    DeviceNumber devNum;
    FunctionNumber funcNum;
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = GetBusDeviceFunction(&pBus, &devNum, &funcNum,classDriverHandle,
                                       functionHandle, detail::Client::SecurityPolicy_OpenIfNotAcquired)).IsSuccess())
    {
        result = pBus->ReadConfigWord(devNum, funcNum, offset, pOut);
    }
    return result;
}

Result ReadConfig32(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle,
                    size_t offset, uint32_t *pOut)
{
    Result result;
    detail::Bus *pBus;
    DeviceNumber devNum;
    FunctionNumber funcNum;
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = GetBusDeviceFunction(&pBus, &devNum, &funcNum, classDriverHandle,
                                       functionHandle, detail::Client::SecurityPolicy_OpenIfNotAcquired)).IsSuccess())
    {
        result = pBus->ReadConfigDWord(devNum, funcNum, offset, pOut);
    }
    return result;
}

Result WriteConfig8(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle,
                    size_t offset, uint8_t data)
{
    Result result;
    detail::Bus *pBus;
    DeviceNumber devNum;
    FunctionNumber funcNum;
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = GetBusDeviceFunction(&pBus, &devNum, &funcNum, classDriverHandle,
                                       functionHandle, detail::Client::SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        result = pBus->WriteConfigByte(devNum, funcNum, offset, data);
    }
    return result;
}

Result WriteConfig16(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle,
                     size_t offset, uint16_t data)
{
    Result result;
    detail::Bus *pBus;
    DeviceNumber devNum;
    FunctionNumber funcNum;
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = GetBusDeviceFunction(&pBus, &devNum, &funcNum, classDriverHandle,
                                       functionHandle, detail::Client::SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        result = pBus->WriteConfigWord(devNum, funcNum, offset, data);
    }
    return result;
}

Result WriteConfig32(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle,
                     size_t offset, uint32_t data)
{
    Result result;
    detail::Bus *pBus;
    DeviceNumber devNum;
    FunctionNumber funcNum;
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = GetBusDeviceFunction(&pBus, &devNum, &funcNum, classDriverHandle,
                                       functionHandle, detail::Client::SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        result = pBus->WriteConfigDWord(devNum, funcNum, offset, data);
    }
    return result;
}

Result ReadBarRegion8(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle,
                      uint8_t bar, size_t offset, uint8_t *pRead8)
{
    return nn::pcie::driver::ReadBarRegion(classDriverHandle, functionHandle, bar, offset, BusAccessWidth_8Bit,
                                           sizeof(uint8_t), reinterpret_cast<void*>(pRead8));
}

Result ReadBarRegion16(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle,
                       uint8_t bar, size_t offset, uint16_t *pRead16)
{
    return nn::pcie::driver::ReadBarRegion(classDriverHandle, functionHandle, bar, offset, BusAccessWidth_16Bit,
                                           sizeof(uint16_t), reinterpret_cast<void*>(pRead16));
}

Result ReadBarRegion32(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle,
                       uint8_t bar, size_t offset, uint32_t *pRead32)
{
    return nn::pcie::driver::ReadBarRegion(classDriverHandle, functionHandle, bar, offset, BusAccessWidth_32Bit,
                                           sizeof(uint32_t), reinterpret_cast<void*>(pRead32));
}

Result ReadBarRegion64(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle,
                       uint8_t bar, size_t offset, uint64_t *pRead64)
{
    return nn::pcie::driver::ReadBarRegion(classDriverHandle, functionHandle, bar, offset, BusAccessWidth_64Bit,
                                           sizeof(uint64_t), reinterpret_cast<void*>(pRead64));
}

Result ReadBarRegion(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle, uint8_t bar, size_t offset,
                     BusAccessWidth accessWidth, size_t readSize, void *pReadData)
{
    detail::EndpointFunction *pEpFunc;
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = detail::Client::GetEndpointFunction(&pEpFunc, NULL, &pciDriver, classDriverHandle,
                                                      functionHandle, detail::Client::SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        detail::ResourceAddr resourceAddress;
        if ((result = pEpFunc->GetBarRegion(&resourceAddress, bar, static_cast<detail::ResourceSize>(offset),
                                            static_cast<detail::ResourceSize>(readSize))).IsSuccess())
        {
            result = CopyBarRegion(pReadData, reinterpret_cast<const void *>(resourceAddress),
                                   accessWidth, readSize);
        }
    }
    return result;
}

Result WriteBarRegion8(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle,
                       uint8_t bar, size_t offset, uint8_t data8)
{
    return nn::pcie::driver::WriteBarRegion(classDriverHandle, functionHandle, bar, offset, BusAccessWidth_8Bit,
                                            sizeof(uint8_t), reinterpret_cast<const void*>(&data8));
}

Result WriteBarRegion16(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle,
                        uint8_t bar, size_t offset, uint16_t data16)
{
    return nn::pcie::driver::WriteBarRegion(classDriverHandle, functionHandle, bar, offset, BusAccessWidth_16Bit,
                                            sizeof(uint16_t), reinterpret_cast<const void*>(&data16));
}

Result WriteBarRegion32(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle,
                        uint8_t bar, size_t offset, uint32_t data32)
{
    return nn::pcie::driver::WriteBarRegion(classDriverHandle, functionHandle, bar, offset, BusAccessWidth_32Bit,
                                            sizeof(uint32_t), reinterpret_cast<const void*>(&data32));
}

Result WriteBarRegion64(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle,
                        uint8_t bar, size_t offset, uint64_t data64)
{
    return nn::pcie::driver::WriteBarRegion(classDriverHandle, functionHandle, bar, offset, BusAccessWidth_64Bit,
                                            sizeof(uint64_t), reinterpret_cast<const void*>(&data64));
}

Result WriteBarRegion(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle, uint8_t bar, size_t offset,
                      BusAccessWidth accessWidth, size_t writeSize, const void *pWriteData)
{
    detail::EndpointFunction *pEpFunc;
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = detail::Client::GetEndpointFunction(&pEpFunc, NULL, &pciDriver, classDriverHandle,
                                      functionHandle, detail::Client::SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        detail::ResourceAddr resourceAddress;
        if ((result = pEpFunc->GetBarRegion(&resourceAddress, bar, static_cast<detail::ResourceSize>(offset),
                                            static_cast<detail::ResourceSize>(writeSize))).IsSuccess())
        {
            result = CopyBarRegion(reinterpret_cast<void *>(resourceAddress), pWriteData,
                                   accessWidth, writeSize);
        }
    }
    return result;
}

Result FindCapability(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle, CapabilityId capabilityId, size_t *pReturnedOffset)
{
    detail::EndpointFunction *pEpFunc;
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = detail::Client::GetEndpointFunction(&pEpFunc, NULL, &pciDriver, classDriverHandle,
                                                      functionHandle, detail::Client::SecurityPolicy_OpenIfNotAcquired)).IsSuccess())
    {
        int32_t offset;
        if ((offset = pEpFunc->FindCapability(capabilityId)) > 0)
        {
            *pReturnedOffset = static_cast<size_t>(offset);
        }
        else
        {
            result = ResultNotAvailable();
        }
    }
    return result;
}

Result FindExtendedCapability(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle, ExtendedCapabilityId extendedCapabilityId,
                              size_t *pReturnedOffset)
{
    detail::EndpointFunction *pEpFunc;
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = detail::Client::GetEndpointFunction(&pEpFunc, NULL, &pciDriver, classDriverHandle,
                                                      functionHandle, detail::Client::SecurityPolicy_OpenIfNotAcquired)).IsSuccess())
    {
        int32_t offset;
        if ((offset = pEpFunc->FindExtendedCapability(extendedCapabilityId)) > 0)
        {
            *pReturnedOffset = static_cast<size_t>(offset);
        }
        else
        {
            result = ResultNotAvailable();
        }
    }
    return result;
}

Result MapDma(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle, DmaDirection direction,
              nn::dd::ProcessHandle procHandle, void *base, size_t size, BusAddress *pReturnedBusAddressBase)
{
    detail::EndpointFunction *pEpFunc;
    Result result = ResultSuccess();

    if(reinterpret_cast<uintptr_t>(base) & (nn::pcie::MinimumDmaAddressAlignment - 1))
    {
        return ResultInvalidAlignment();
    }

    if(size & (nn::pcie::MinimumDmaAddressAlignment - 1))
    {
        return ResultInvalidSizePadding();
    }

    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = detail::Client::GetEndpointFunction(&pEpFunc, NULL, &pciDriver, classDriverHandle,
                                                      functionHandle, detail::Client::SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        nn::dd::MemoryPermission devicePermission;
        switch (direction)
        {
        case DmaDirection_BiDirectional:
            devicePermission = nn::dd::MemoryPermission_ReadWrite;
            break;
        case DmaDirection_FromDevice:
            devicePermission = nn::dd::MemoryPermission_WriteOnly;
            break;
        case DmaDirection_ToDevice:
            devicePermission = nn::dd::MemoryPermission_ReadOnly;
            break;
        default:
            return ResultInvalidDmaDirection();
            break;
        }
        result = pEpFunc->CreateDmaMapping(procHandle, reinterpret_cast<uint64_t>(base), size,
                                           nn::dd::MemoryPermission_ReadWrite, devicePermission, pReturnedBusAddressBase);
    }
    return result;
}

Result UnmapDma(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle, nn::dd::ProcessHandle processHandle, void *base)
{
    detail::EndpointFunction *pEpFunc;
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = detail::Client::GetEndpointFunction(&pEpFunc, NULL, &pciDriver, classDriverHandle,
                                                      functionHandle, detail::Client::SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        result = pEpFunc->DestroyDmaMappingByProcVa(processHandle, reinterpret_cast<uint64_t>(base));
    }
    return result;
}

Result UnmapDma(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle, BusAddress busAddr)
{
    detail::EndpointFunction *pEpFunc;
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = detail::Client::GetEndpointFunction(&pEpFunc, NULL, &pciDriver, classDriverHandle,
                                      functionHandle, detail::Client::SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        result = pEpFunc->DestroyDmaMappingByBusAddress(busAddr);
    }
    return result;
}

Result GetDmaBusAddress(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle, nn::dd::ProcessHandle processHandle,
                        void *base, size_t size, BusAddress *pReturnedBusAddressBase)
{
    NN_UNUSED(size);
    detail::EndpointFunction *pEpFunc;
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = detail::Client::GetEndpointFunction(&pEpFunc, NULL, &pciDriver, classDriverHandle,
                                                      functionHandle, detail::Client::SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        result = pEpFunc->GetDmaBusAddress(processHandle, reinterpret_cast<uint64_t>(base), pReturnedBusAddressBase);
    }
    return result;
}

Result GetDmaBusAddressRange(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle, BusAddress *pOutBase, BusAddress *pOutSize)
{
    Result result = ResultSuccess();
    detail::EndpointFunction *pEpFunc;
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = detail::Client::GetEndpointFunction(&pEpFunc, NULL, &pciDriver, classDriverHandle,
                                                      functionHandle, detail::Client::SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        result = pEpFunc->GetDmaBusAddressRange(pOutBase, pOutSize);
    }
    return result;
}

Result SetDmaEnable(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle, bool isBusMasterEnabled)
{
    detail::EndpointFunction *pEpFunc;
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = detail::Client::GetEndpointFunction(&pEpFunc, NULL, &pciDriver, classDriverHandle,
                                                      functionHandle, detail::Client::SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        pEpFunc->SetDMAEnable(isBusMasterEnabled);
    }
    return result;
}

Result GetMaxIrqCount(int32_t* pOutMaxIrqCount, ClassDriverHandle classDriverHandle, FunctionHandle functionHandle)
{
    detail::EndpointFunction *pEpFunc;
    Result result = ResultSuccess();
    *pOutMaxIrqCount = 0;
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = detail::Client::GetEndpointFunction(&pEpFunc, NULL, &pciDriver, classDriverHandle,
                                                      functionHandle, detail::Client::SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        *pOutMaxIrqCount = pEpFunc->GetMaxNumMsiVectors();
    }
    return result;
}

Result AcquireIrq(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle, IrqCallback callback,
                  uintptr_t irqContext, IrqType irqType, int32_t numberOfVectors)
{
    detail::Client* pClient;
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = detail::Client::GetEndpointFunction(NULL, &pClient, &pciDriver, classDriverHandle,
                                                      functionHandle, detail::Client::SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        result = pClient->AcquireIrq(functionHandle, callback, irqContext, irqType, numberOfVectors);
    }
    return result;
}

Result ReleaseIrq(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle)
{
    detail::Client *pClient= NULL;
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if((pClient = detail::Client::GetClient(&pciDriver, classDriverHandle)) != NULL)
    {
        result = pClient->ReleaseIrq(functionHandle);
    }
    else
    {
        result = ResultInvalidClassDriverHandle();
    }
    return result;
}

Result SetIrqEnable(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle, int32_t zeroBasedIrqNumber, bool irqEnabled)
{
    detail::EndpointFunction *pEpFunc;
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = detail::Client::GetEndpointFunction(&pEpFunc, NULL, &pciDriver, classDriverHandle,
                                                     functionHandle, detail::Client::SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        result = pEpFunc->SetIrqEnable(zeroBasedIrqNumber, irqEnabled);
    }
    return result;
}

Result SetAspmEnable(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle, bool isEnabled)
{
    detail::EndpointFunction *pEpFunc;
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = detail::Client::GetEndpointFunction(&pEpFunc, NULL, &pciDriver, classDriverHandle,
                                                          functionHandle, detail::Client::SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        detail::RootComplex *pRC;
        pEpFunc->GetBus()->GetRootComplex(&pRC);
        result = pRC->SetAspmEnable(pEpFunc, isEnabled);
    }
    return result;
}

Result SetResetUponResumeEnable(ClassDriverHandle classDriverHandle, FunctionHandle functionHandle, bool isEnabled)
{
    detail::Client* pClient;
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(pciDriver.m_AccessMutex);
    if ((result = detail::Client::GetEndpointFunction(NULL, &pClient, &pciDriver, classDriverHandle,
                                                      functionHandle, detail::Client::SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        pClient->SetResetUponResumeEnable(functionHandle, isEnabled);
    }
    return result;
}



} // end of namespace driver
} // end of namespace pcie
} // end of namespace nn
