﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "../pcie_PrivateIncludes.h"
#include "pcie_Tegra21xRootComplex.h"

namespace nn { namespace pcie { namespace driver { namespace detail {

Tegra21xRootComplex::Tegra21xRootComplex(Driver *pDriver, Config *pCfg)
    : RootComplex(pDriver, pCfg)
    , m_Ldo1Voltage(0)
    , m_RpClkClampThreshold(0)
    , m_PadsRefClkCfg0(0)
    , m_PcieActiveLaneMap(0)
    , m_PcieInactiveLaneMap(0)
    , m_UsbLaneMap(0)
    , m_PadctrlMuxIddqMask(0)
{
    memset(m_SocName, 0, sizeof(m_SocName));
    m_ControllerType = ControllerType_Tegra21x;
    for (int32_t index = 0; index < AFI_INT_TOTAL_VECTORS; index++)
    {
        m_port0IntIrqMgrs[index].SetPoolId(intPort0IrqMgrBasePoolId + index);
        m_port1IntIrqMgrs[index].SetPoolId(intPort1IrqMgrBasePoolId + index);
    }
}

Tegra21xRootComplex::~Tegra21xRootComplex()
{

}

void* Tegra21xRootComplex::operator new(size_t size) NN_NOEXCEPT
{
    return nn::pcie::detail::MemoryAllocAligned(size, MinDataAlignmentSize, "Tegra21xRootComplex");
}

void Tegra21xRootComplex::operator delete(void *p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    nn::pcie::detail::MemoryFree(p, "Tegra21xRootComplex");
}

Result Tegra21xRootComplex::InitPhy()
{
    Result result = ResultSuccess();

    // See page 1191 of TK2 TRM document.  Pages 1361-2 of version 1.1.

    // Start with default values
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_1_0_REG  = 0x01900017;
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_REG  = 0x00000000;
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_3_0_REG  = 0x00037850;
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_4_0_REG  = 0x0050A100;
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_5_0_REG  = 0x00000021;
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_6_0_REG  = 0x02840000;
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_7_0_REG  = 0x00000333;
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_REG  = 0x00070333;
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_9_0_REG  = 0x00000000;
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_REG = 0x08000000;
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_11_0_REG = 0x00000000;
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_12_0_REG = 0x00000000;

    if(nn::util::Strncmp(m_SocName, "T214", sizeof(m_SocName)) == 0)
    {
        // Program PLL defaults for PLL0
        NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_REG =
            NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_CFG_WS_MASK                 |
            NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_CFG_RESET_MASK              |
            (0x0000 << NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_CFG_WDATA_SHIFT) |
            (0x2    << NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_CFG_ADDR_SHIFT) ;
        NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_REG =
            NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_CFG_WS_MASK                 |
            NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_CFG_RESET_MASK              |
            (0x7051 << NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_CFG_WDATA_SHIFT) |
            (0x3    << NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_CFG_ADDR_SHIFT) ;
        NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_REG =
            NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_CFG_WS_MASK                 |
            NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_CFG_RESET_MASK              |
            (0x0130 << NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_CFG_WDATA_SHIFT) |
            (0x25   << NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_CFG_ADDR_SHIFT) ;
        NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_REG =
            NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_CFG_WS_MASK                 |
            NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_CFG_RESET_MASK              |
            (0x0017 << NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_CFG_WDATA_SHIFT) |
            (0x1E   << NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_10_0_CFG_ADDR_SHIFT) ;

        // Enable overrides to enable SW control over PLL
        // Set PWR/CAL/RCAL OVRD
        NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_1_0_REG |= XUSB_PADCTL_UPHY_PLL_P0_CTL_1_0_PLL0_PWR_OVRD;
        NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_REG |= XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_PLL0_CAL_OVRD;
        NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_REG |= XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_PLL0_RCAL_OVRD;
    }
    else
    {
        // Apply calibration control values
        NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_REG =
            (NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_REG & ~XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_PLL0_CAL_CTRL_MASK) |
            (0x136 << XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_PLL0_CAL_CTRL_SHIFT);

        NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_5_0_REG =
            (NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_5_0_REG & ~XUSB_PADCTL_UPHY_PLL_P0_CTL_5_0_PLL0_DC0_CTRL_MASK) |
            (0x2a << XUSB_PADCTL_UPHY_PLL_P0_CTL_5_0_PLL0_DC0_CTRL_SHIFT);

        // Enable overrides to enable SW control over PLL
        // Set PWR/CAL/RCAL OVRD
        NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_1_0_REG |= XUSB_PADCTL_UPHY_PLL_P0_CTL_1_0_PLL0_PWR_OVRD;
        NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_REG |= XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_PLL0_CAL_OVRD;
        NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_REG |= XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_PLL0_RCAL_OVRD;

        // Power up PLL
        NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_4_0_REG =
            (NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_4_0_REG &
                ~(XUSB_PADCTL_UPHY_PLL_P0_CTL_4_0_PLL0_TXCLKREF_SEL_MASK | XUSB_PADCTL_UPHY_PLL_P0_CTL_4_0_PLL0_REFCLK_SEL_MASK)) |
            (0x2 << XUSB_PADCTL_UPHY_PLL_P0_CTL_4_0_PLL0_TXCLKREF_SEL_SHIFT) | XUSB_PADCTL_UPHY_PLL_P0_CTL_4_0_PLL0_TXCLKREF_EN;

        NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_1_0_REG =
            (NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_1_0_REG &
                ~(XUSB_PADCTL_UPHY_PLL_P0_CTL_1_0_PLL0_FREQ_NDIV_MASK | XUSB_PADCTL_UPHY_PLL_P0_CTL_1_0_PLL0_FREQ_MDIV_MASK)) |
            (0x19 << XUSB_PADCTL_UPHY_PLL_P0_CTL_1_0_PLL0_FREQ_NDIV_SHIFT);
    }


    // Clear PLL0 iddq and sleep
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_1_0_REG &= ~XUSB_PADCTL_IOPHY_PLL_S0_CTL1_PLL_IDDQ;
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_1_0_REG &= ~XUSB_PADCTL_IOPHY_PLL_S0_CTL1_PLL_SLEEP;

    // Wait at least 100ns
    nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(100));

    // Calibration
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_REG |= XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_PLL0_CAL_EN;
    WaitPolledRegister32(&NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_REG, XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_PLL0_CAL_DONE, false,
                         PllPollIntervalInUs, MaxPllWaitTimeInUs,
                         "UPHY CAL -> 1");
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_REG &= ~XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_PLL0_CAL_EN;
    WaitPolledRegister32(&NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_REG, XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_PLL0_CAL_DONE, true,
                         PllPollIntervalInUs, MaxPllWaitTimeInUs,
                         "UPHY CAL -> 0");

    // Enable PLL (20us lock time)
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_1_0_REG |= XUSB_PADCTL_UPHY_PLL_P0_CTL_1_0_PLL0_ENABLE;
    WaitPolledRegister32(&NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_1_0_REG, XUSB_PADCTL_UPHY_PLL_P0_CTL_1_0_PLL0_LOCKDET_STATUS, false,
                         PllPollIntervalInUs, MaxPllWaitTimeInUs,
                         "UPHY");

    // Resistor Calibration
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_REG |= XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_PLL0_RCAL_EN;
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_REG |= XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_PLL0_RCAL_CLK_EN;
    WaitPolledRegister32(&NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_REG, XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_PLL0_RCAL_DONE, false,
                         PllPollIntervalInUs, MaxPllWaitTimeInUs,
                         "UPHY RCAL -> 1");
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_REG &= ~XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_PLL0_RCAL_EN;
    WaitPolledRegister32(&NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_REG, XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_PLL0_RCAL_DONE, true,
                         PllPollIntervalInUs, MaxPllWaitTimeInUs,
                         "UPHY RCAL -> 0");
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_REG &= ~XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_PLL0_RCAL_CLK_EN;


    // Record RCAL Value, useful for topics such as NSBG-7574
    m_LoggedState.pllResistorCalibrationValue = (NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_REG &
        XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_PLL0_RCAL_VAL_MASK) >> XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_PLL0_RCAL_VAL_SHIFT;

    // Enable Hardware Power Sequencer
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetClockEnabled(nn::pcv::Module_XUsbIoPll, true));
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_1_0_REG &= ~XUSB_PADCTL_UPHY_PLL_P0_CTL_1_0_PLL0_PWR_OVRD;
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_REG &= ~XUSB_PADCTL_UPHY_PLL_P0_CTL_2_0_PLL0_CAL_OVRD;
    NN_PCI_XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_REG &= ~XUSB_PADCTL_UPHY_PLL_P0_CTL_8_0_PLL0_RCAL_OVRD;
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1));
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetClockEnabled(nn::pcv::Module_XUsbIoPllHwSeq, true));

    return result;
}

Result Tegra21xRootComplex::InitXusb()
{

    uint32_t muxRegVal = NN_PCI_XUSB_PADCTL_USB3_PAD_MUX_REG;
    uint32_t pcieLaneMap = m_PcieActiveLaneMap | m_PcieInactiveLaneMap;

    // For all XUSB lanes
    for(int laneIndex = 0, pcieLaneCount = 0; laneIndex < 8; laneIndex++)
    {
        // Bit position of the ownership field for this lane
        int shift = XUSB_PADCTL_USB3_PAD_MUX_ASSIGN_LANE0_SHIFT +
            laneIndex * (XUSB_PADCTL_USB3_PAD_MUX_ASSIGN_LANE1_SHIFT - XUSB_PADCTL_USB3_PAD_MUX_ASSIGN_LANE0_SHIFT);

        // If this lane will be assigned to either PCIe or USB
        if((pcieLaneMap & (1 << laneIndex)) || (m_UsbLaneMap & (1 << laneIndex)))
        {
            muxRegVal &= ~(3 << shift);
        }

        // If this lane is associated with PCIe
        if(pcieLaneMap & (1 << laneIndex))
        {
            // Configure PCIe lane ownership
            if(pcieLaneCount == 0)
            {
                muxRegVal |= (0 << shift); // select x1
            }
            else
            {
                muxRegVal |= (3 << shift); // select x4
            }

            // Form the IDDQ mask
            m_PadctrlMuxIddqMask |= (XUSB_PADCTL_USB3_PAD_MUX_PCIE_IDDQ_DISABLE_LANE0_SHIFT << (1 * laneIndex));

            pcieLaneCount++;
        }
        // Else if tihs is for USB
        else if(m_UsbLaneMap & (1 << laneIndex))
        {
            muxRegVal |= (1 << shift);
        }
    }

    // Update lane ownership
    NN_PCI_XUSB_PADCTL_USB3_PAD_MUX_REG = muxRegVal;

    // Disable IDDQ
    NN_PCI_XUSB_PADCTL_USB3_PAD_MUX_REG |= m_PadctrlMuxIddqMask;

    if(nn::util::Strncmp(m_SocName, "T214", sizeof(m_SocName)) == 0)
    {
        // Apply configuration to PCIe lanes
        uint32_t commandValue = NN_PCIE_XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_8_0_CFG_WS_MASK                  |
                                NN_PCIE_XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_8_0_CFG_RESET_MASK               |
                                (0x0080  << NN_PCIE_XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_8_0_CFG_WDATA_SHIFT) |
                                (0x97    << NN_PCIE_XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_8_0_CFG_ADDR_SHIFT);
        if(pcieLaneMap & 0x01)
        {
            NN_PCIE_XUSB_PADCTL_UPHY_MISC_PAD_P0_CTL_8_0_REG = commandValue;
        }
        if(pcieLaneMap & 0x02)
        {
            NN_PCIE_XUSB_PADCTL_UPHY_MISC_PAD_P1_CTL_8_0_REG = commandValue;
        }
        if(pcieLaneMap & 0x04)
        {
            NN_PCIE_XUSB_PADCTL_UPHY_MISC_PAD_P2_CTL_8_0_REG = commandValue;
        }
        if(pcieLaneMap & 0x08)
        {
            NN_PCIE_XUSB_PADCTL_UPHY_MISC_PAD_P3_CTL_8_0_REG = commandValue;
        }
        if(pcieLaneMap & 0x10)
        {
            NN_PCIE_XUSB_PADCTL_UPHY_MISC_PAD_P4_CTL_8_0_REG = commandValue;
        }

        // For all XUSB lanes
        for(int laneIndex = 0; laneIndex < 8; laneIndex++)
        {
            uint32_t selectedLane;
            // That which are assigned to USB
            if((selectedLane = (m_UsbLaneMap & (1 << laneIndex))) != 0)
            {
                volatile uint32_t* pMiscPadCtlReg = nullptr;
                // Resolve the port specific register
                if(selectedLane & 0x02)
                {
                    pMiscPadCtlReg = &NN_PCIE_XUSB_PADCTL_UPHY_MISC_PAD_P1_CTL_8_0_REG;
                }
                else if(selectedLane & 0x10)
                {
                    pMiscPadCtlReg = &NN_PCIE_XUSB_PADCTL_UPHY_MISC_PAD_P4_CTL_8_0_REG;
                }
                else if(selectedLane & 0x20)
                {
                    pMiscPadCtlReg = &NN_PCIE_XUSB_PADCTL_UPHY_MISC_PAD_P5_CTL_8_0_REG;
                }
                if(pMiscPadCtlReg != nullptr)
                {
                    static const LaneConfiguration usb3LaneData[] =
                    { // This configuration taken from Raptor source - it's not in the latest TRM.
                        {0x1,  0x0002}, {0x4,  0x0032}, {0x7,  0x0022}, {0x35, 0x2587}, {0x49, 0x0FC7}, {0x52, 0x0001}, {0x53, 0x3C0F},
                        {0x56, 0xC00F}, {0x5D, 0xFF07}, {0x5E, 0x141A}, {0x97, 0x0080}
                    };
                    commandValue = NN_PCIE_XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_8_0_CFG_WS_MASK | NN_PCIE_XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_8_0_CFG_RESET_MASK;
                    // Program the PAD
                    for(int laneDataIndex = 0; laneDataIndex < static_cast<int>(sizeof(usb3LaneData) / sizeof(usb3LaneData[0])); laneDataIndex++)
                    {
                        uint32_t padCtlRegVal = commandValue |
                            (usb3LaneData[laneDataIndex].data    << NN_PCIE_XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_8_0_CFG_WDATA_SHIFT) |
                            (usb3LaneData[laneDataIndex].address << NN_PCIE_XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_8_0_CFG_ADDR_SHIFT);
                        *pMiscPadCtlReg = padCtlRegVal;
                    }
                }
            }
        }
    }

    // Clear AUX_MUX_LP0 related bits in XUSB_PADCTL_ELPG_PROGRAM_1_0
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1));
    NN_PCI_XUSB_PADCTL_ELPG_PROGRAM_1_0_REG &= ~NN_PCI_XUSB_PADCTL_ELPG_PROGRAM_1_0_AUX_MUX_LP0_CLAMP_EN;
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(100));
    NN_PCI_XUSB_PADCTL_ELPG_PROGRAM_1_0_REG &= ~NN_PCI_XUSB_PADCTL_ELPG_PROGRAM_1_0_AUX_MUX_LP0_CLAMP_EN_EARLY;
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(100));
    NN_PCI_XUSB_PADCTL_ELPG_PROGRAM_1_0_REG &= ~NN_PCI_XUSB_PADCTL_ELPG_PROGRAM_1_0_AUX_MUX_LP0_VCORE_DOWN;
    // NN_PCI_XUSB_PADCTL_ELPG_PROGRAM_1_0_REG = 0x00000ff8;  // from reference addr dump
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(200));

    // Set lane IDDQ and SLEEP override, SW WAR 1507600
    if(pcieLaneMap & 0x01)
    {
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P0_CTL_2_0_REG |= (XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_OVRDS | XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_IDDQS);
    }
    if(pcieLaneMap & 0x02)
    {
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P1_CTL_2_0_REG |= (XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_OVRDS | XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_IDDQS);
    }
    if(pcieLaneMap & 0x04)
    {
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P2_CTL_2_0_REG |= (XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_OVRDS | XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_IDDQS);
    }
    if(pcieLaneMap & 0x08)
    {
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P3_CTL_2_0_REG |= (XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_OVRDS | XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_IDDQS);
    }
    if(pcieLaneMap & 0x10)
    {
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P4_CTL_2_0_REG |= (XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_OVRDS | XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_IDDQS);
    }

    // Clear lane IDDQ and SLEEP overrides, SW WAR 1507600
    if(pcieLaneMap & 0x01)
    {
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P0_CTL_2_0_REG &= ~XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_OVRDS;
    }
    if(pcieLaneMap & 0x02)
    {
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P1_CTL_2_0_REG &= ~XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_OVRDS;
    }
    if(pcieLaneMap & 0x04)
    {
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P2_CTL_2_0_REG &= ~XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_OVRDS;
    }
    if(pcieLaneMap & 0x08)
    {
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P3_CTL_2_0_REG &= ~XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_OVRDS;
    }
    if(pcieLaneMap & 0x10)
    {
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P4_CTL_2_0_REG &= ~XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_OVRDS;
    }

    // Power down unused "port 0" lanes 1,2,3. We use only 4, since our PCB supports only X1 PCIe bus.
    if(m_PcieInactiveLaneMap & 0x01)
    {
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P0_CTL_2_0_REG |=
            (XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_RX_SLEEP|XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_TX_SLEEP);
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P0_CTL_2_0_REG |=
            (XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_TX_IDDQ | XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_RX_IDDQ |
             XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_TX_IDDQ_OVRD| XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_RX_IDDQ_OVRD);
    }
    if(m_PcieInactiveLaneMap & 0x02)
    {
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P1_CTL_2_0_REG |=
            (XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_RX_SLEEP|XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_TX_SLEEP);
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P1_CTL_2_0_REG |=
            (XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_TX_IDDQ | XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_RX_IDDQ |
             XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_TX_IDDQ_OVRD| XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_RX_IDDQ_OVRD);
    }
    if(m_PcieInactiveLaneMap & 0x04)
    {
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P2_CTL_2_0_REG |=
            (XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_RX_SLEEP|XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_TX_SLEEP);
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P2_CTL_2_0_REG |=
            (XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_TX_IDDQ | XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_RX_IDDQ |
             XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_TX_IDDQ_OVRD| XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_RX_IDDQ_OVRD);
    }
    if(m_PcieInactiveLaneMap & 0x08)
    {
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P3_CTL_2_0_REG |=
            (XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_RX_SLEEP|XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_TX_SLEEP);
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P3_CTL_2_0_REG |=
            (XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_TX_IDDQ | XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_RX_IDDQ |
             XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_TX_IDDQ_OVRD| XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_RX_IDDQ_OVRD);
    }
    if(m_PcieInactiveLaneMap & 0x10)
    {
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P4_CTL_2_0_REG |=
            (XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_RX_SLEEP|XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_TX_SLEEP);
        NN_PCI_XUSB_PADCTL_UPHY_MISC_PAD_P4_CTL_2_0_REG |=
            (XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_TX_IDDQ | XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_RX_IDDQ |
             XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_TX_IDDQ_OVRD| XUSB_PADCTL_UPHY_MISC_PAD_PX_CTL_2_0_RX_IDDQ_OVRD);
    }

    return ResultSuccess();
} // NOLINT(impl/function_size)

Result Tegra21xRootComplex::InitPinMux()
{
    // Configure pin mux
    NN_PCI_PINMUX_AUX_PEX_L0_CLKREQ_N_0_REG =
        (NN_PCI_PINMUX_AUX_PEX_L0_CLKREQ_N_0_REG & ~PINMUX_AUX_PEX_LX_CLKREQ_N_0_E_OD) | PINMUX_AUX_PEX_LX_CLKREQ_N_0_TRISTATE;
    NN_PCI_PINMUX_AUX_PEX_L1_CLKREQ_N_0_REG =
        (NN_PCI_PINMUX_AUX_PEX_L1_CLKREQ_N_0_REG & ~PINMUX_AUX_PEX_LX_CLKREQ_N_0_E_OD) | PINMUX_AUX_PEX_LX_CLKREQ_N_0_TRISTATE;

    // Configure reset pad
    NN_PCI_PINMUX_AUX_PEX_L0_RST_N_0_REG = NN_PCI_PINMUX_AUX_PEX_L0_RST_N_0_REG & ~PINMUX_AUX_PEX_LX_RST_N_0_TRISTATE;
    NN_PCI_PINMUX_AUX_PEX_L1_RST_N_0_REG = NN_PCI_PINMUX_AUX_PEX_L1_RST_N_0_REG & ~PINMUX_AUX_PEX_LX_RST_N_0_TRISTATE;

    return ResultSuccess();
}

void Tegra21xRootComplex::SynchronizedWrite(volatile uint32_t *address, uint32_t value)
{
    *address = value;
    (void)(*address);
    dd::EnsureMemoryAccess();
}

Result Tegra21xRootComplex::DoPortOperation(DeviceNumber portNum, PortOperation op)
{
    Result result = ResultSuccess();
    uint32_t regVal32;
    volatile uint32_t *pPexCtrl = (portNum == 0) ? &NN_PCI_AFI_PEX0_CTRL_REG : &NN_PCI_AFI_PEX1_CTRL_REG;
    //volatile uint32_t *pPexSts = (portNum == 0) ? &NN_PCI_AFI_PEX0_STATUS_REG : &NN_PCI_AFI_PEX1_STATUS_REG;
    volatile uint32_t *pClkReqPinMux = (portNum == 0) ? &NN_PCI_PINMUX_AUX_PEX_L0_CLKREQ_N_0_REG : &NN_PCI_PINMUX_AUX_PEX_L1_CLKREQ_N_0_REG;

    switch (op)
    {
    case PORT_OP_DISABLE_CLOCK:
        // put endpoint into reset
        *pPexCtrl &= ~AFI_PEX_CTRL_RST_L;

        // Setup clkreq pin to be disabled completely
        *pClkReqPinMux = PINMUX_AUX_PEX_LX_CLKREQ_N_0_PARK | PINMUX_AUX_PEX_LX_CLKREQ_N_0_TRISTATE;

        // disable reference clock
        // - disable refclk pad
        // - disable clkreq pad
        // - disable clock override
        *pPexCtrl &= ~(AFI_PEX_CTRL_REFCLK_EN | AFI_PEX_CTRL_CLKREQ_EN | AFI_PEX_CTRL_REFCLK_OVR_EN);
        break;
    case PORT_OP_ENABLE_CLOCK_OVERRIDE:
        // Setup clkreq pin mux for manual or basic PCIe clock management
        *pClkReqPinMux = PINMUX_AUX_PEX_LX_CLKREQ_N_0_E_INPUT;

        // enable manual reference clock override
        // - enable refclk pad
        // - disable clkreq pad (clock forced on for now)
        // - enable clock override (clock forced on for now)
        *pPexCtrl = (*pPexCtrl & ~AFI_PEX_CTRL_CLKREQ_EN) | AFI_PEX_CTRL_REFCLK_EN | AFI_PEX_CTRL_REFCLK_OVR_EN;
        break;
    case PORT_OP_ENABLE_CLOCK_AUTOMATIC:
        // Setup clkreq pin mux for bidirectional automatic clock control
        *pClkReqPinMux = PINMUX_AUX_PEX_LX_CLKREQ_N_0_E_INPUT;

        // enable automatic reference clock control
        // - enable refclk pad
        // - enable clkreq pad (clock request pin is honored)
        // - disable clock override (clock control is automatic, per clock request)
        *pPexCtrl = (*pPexCtrl & ~AFI_PEX_CTRL_REFCLK_OVR_EN) | AFI_PEX_CTRL_CLKREQ_EN | AFI_PEX_CTRL_REFCLK_EN;
        break;
    case PORT_OP_RESET:
        *pPexCtrl &= ~AFI_PEX_CTRL_RST_L;
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(EndpointResetDurationInUs));
        *pPexCtrl |= AFI_PEX_CTRL_RST_L;
        break;
    case PORT_OP_HOLD_RESET:
        *pPexCtrl &= ~AFI_PEX_CTRL_RST_L;
        break;
    case PORT_OP_GET_STATE:
        // Check vendor specific link status
        ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_VEND_XP_OFFSET, sizeof(uint32_t), &regVal32);
        if (regVal32 & RP_VEND_XP_DL_UP)
        {
            m_LoggedState.ports[portNum].flagSet.Set<nn::pcie::RootPortStateFlag::DataLinkActiveVendor>(true);
        }
        else
        {
            m_LoggedState.ports[portNum].flagSet.Set<nn::pcie::RootPortStateFlag::DataLinkActiveVendor>(false);
            result = ResultPortLinkDown();
        }

        // Check standard PCIe link status
        ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_LINK_CONTROL_STATUS_OFFSET, sizeof(uint32_t), &regVal32);
        if (regVal32 & RP_LINK_CONTROL_STATUS_DL_LINK_ACTIVE)
        {
            m_LoggedState.ports[portNum].flagSet.Set<nn::pcie::RootPortStateFlag::DataLinkActiveStandard>(true);
        }
        else
        {
            m_LoggedState.ports[portNum].flagSet.Set<nn::pcie::RootPortStateFlag::DataLinkActiveStandard>(false);
            result = ResultPortLinkDown();
        }
        break;

    default:
        result = ResultBadInternalArgument();
        break;
    }
    return result;
}

Result Tegra21xRootComplex::SetPresentMapOverride(DeviceNumber portNum, bool isPresent)
{
    Result result = ResultSuccess();
    uint32_t regVal32 = 0;

    ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_PRIV_MISC_OFFSET, sizeof(uint32_t), &regVal32);
    regVal32 &= ~RP_PRIV_MISC_PRSNT_MAP_EP_MASK;
    regVal32 |= ((isPresent) ? 0xE : 0xF);
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_PRIV_MISC_OFFSET, sizeof(uint32_t), regVal32);

    return result;
}

Result Tegra21xRootComplex::DoPmeTurnoff()
{
    Result result = ResultSuccess();

    // This will broadcast PME_Turn_Off TLP Message
    NN_PCI_AFI_PCIE_PME_REG |= AFI_PCIE_PME_TURN_OFF;

    // This will wait for PME_TO_Ack from endpoints
    result = WaitPolledRegister32(&NN_PCI_AFI_PCIE_PME_REG, AFI_PCIE_PME_ACK, false,
                                  1000, 3000000, "PME_TO_Ack");

    // Required for PLL power down
    NN_PCI_AFI_PLLE_CONTROL_REG |= AFI_PLLE_CONTROL_BYPASS_PADS2PLLE_CONTROL;

    return result;
}
// Input is the scale value from the capability register.  Return the TPOWERON scale value, in microseconds.
uint32_t Tegra21xRootComplex::L1SSTPowerOnScaledValue(uint32_t capValue)
{
    uint32_t scale = (capValue >> ExtendedCapabilityValue_L1SSCapTPowerOnScaleShift) & ExtendedCapabilityValue_L1SSTPowerOnScaleMask;
    switch (scale)
    {
    case 0x00:
        scale = 2;
        break;
    case 0x01:
        scale = 10;
        break;
    case 0x02:
        scale = 100;
        break;
    default:
        scale = 0;
    }

    return ((capValue >> ExtendedCapabilityValue_L1SSCapTPowerOnValueShift) & ExtendedCapabilityValue_L1SSTPowerOnValueMask) * scale;
}

void Tegra21xRootComplex::DumpPCIeSpace(uint32_t *cfgBase, uint32_t *virtBase, uint32_t *physBase, uint32_t size)
{
    uint32_t *addr, *end;
    NN_UNUSED(cfgBase);
    NN_UNUSED(virtBase);
    NN_UNUSED(physBase);
    NN_UNUSED(size);
    end = (uint32_t*)((uint64_t)cfgBase + size);
    for (addr = cfgBase; addr < end; addr++)
    {
        if (0x0 != *addr)
        {
            NN_PCIE_LOG_INFO("0x%08x: 0x%08x\n", (uint32_t)((uintptr_t)addr - (uintptr_t)virtBase + (uintptr_t)physBase), *addr);
        }
    }
}

Result Tegra21xRootComplex::ConfigureCommonClock(BridgeFunction *pBf, EndpointFunction *pEf)
{
    uint16_t reg16 = 0;
    Result   result = ResultSuccess();

    NN_PCIE_RETURN_UPON_ERROR(pBf->ReadPCIeCapabilityWord(PcieOffset_LinkStatus, &reg16));
    if (reg16 & PcieValue_LinkStatusSlotClockConfiguration)
    {
        NN_PCIE_RETURN_UPON_ERROR(pEf->ReadPCIeCapabilityWord(PcieOffset_LinkStatus, &reg16));
        if (reg16 & PcieValue_LinkStatusSlotClockConfiguration)
        {
                NN_PCIE_RETURN_UPON_ERROR(pEf->ReadPCIeCapabilityWord(PcieOffset_LinkControl, &reg16));
                reg16 |= PcieValue_LinkControlCommonClockConfiguration;
                NN_PCIE_RETURN_UPON_ERROR(pEf->WritePCIeCapabilityWord(PcieOffset_LinkControl, reg16));

                NN_PCIE_RETURN_UPON_ERROR(pBf->ReadPCIeCapabilityWord(PcieOffset_LinkControl, &reg16));
                reg16 |= PcieValue_LinkControlCommonClockConfiguration;
                NN_PCIE_RETURN_UPON_ERROR(pBf->WritePCIeCapabilityWord(PcieOffset_LinkControl, reg16));

                result = pBf->RetrainLink();
        }
    }

    return result;
}

// Enable ASPM
Result Tegra21xRootComplex::SetAspmEnable(EndpointFunction *pEf, bool isEnabled)
{
    Result          result = ResultSuccess();
    BridgeFunction *pBf    = pEf->GetBus()->GetRootBridgeFunction();
    DeviceNumber portNum   =  pBf->GetDevNum();
    bool isL1ssCapable = false, isAspmCapable = false, isL1ssLtrCapable = false;
    uint32_t  bLinkValue = 0, epLinkValue = 0;
    uint32_t  bL1ssCap = 0, epL1ssCap = 0;
    int32_t   bL1ssBase = 0, epL1ssBase = 0;
    uint32_t  bDevCap2 = 0, epDevCap2 = 0, epDevCap = 0;
    uint8_t   epAcceptableLatency = 0, linkExitLatency = 0;

    // get link capabilities
    NN_PCIE_RETURN_UPON_ERROR(pBf->ReadPCIeCapabilityDWord(PcieOffset_LinkCapability, &bLinkValue));
    NN_PCIE_RETURN_UPON_ERROR(pEf->ReadPCIeCapabilityDWord(PcieOffset_LinkCapability, &epLinkValue));

    // determine if even situation is even ASPM capable
    isAspmCapable = (PcieValue_LinkCapabilityAspmSupport & bLinkValue) && (PcieValue_LinkCapabilityAspmSupport & epLinkValue);
    if(!(isAspmCapable))
    {
        NN_PCIE_LOG_INFO("port=%d or endpoint are not ASPM capable.\n", portNum);
        return ResultHardwareUnsupported();
    }

    // for L0s and L1, check if device can tolerate link's exit latencies
    pEf->ReadPCIeCapabilityDWord(PcieOffset_DeviceCapability, &epDevCap);
    // L0s
    epAcceptableLatency = (epDevCap & PcieValue_DeviceCapabilityL0sAcceptableLatency) >> PcieValue_DeviceCapabilityL0sAcceptableLatency_Shift;
    linkExitLatency = (bLinkValue & PcieValue_LinkCapabilityL0sExitLatency) >> PcieValue_LinkCapabilityL0sExitLatency_Shift;
    if ((epAcceptableLatency != 0x7) && (epAcceptableLatency < linkExitLatency))
    {
        NN_PCIE_LOG_INFO("Endpoint cannot tolerate link L0s exit latency.\n");
        return ResultHardwareUnsupported();
    }
    epAcceptableLatency = (epDevCap & PcieValue_DeviceCapabilityL1AcceptableLatency) >> PcieValue_DeviceCapabilityL1AcceptableLatency_Shift;
    linkExitLatency = (bLinkValue & PcieValue_LinkCapabilityL1ExitLatency) >> PcieValue_LinkCapabilityL1ExitLatency_Shift;
    if ((epAcceptableLatency != 0x7) && (epAcceptableLatency < linkExitLatency))
    {
        NN_PCIE_LOG_INFO("Endpoint cannot tolerate link L1 exit latency.\n");
        return ResultHardwareUnsupported();
    }


    // resolve L1SS base offsets
    bL1ssBase = pBf->FindExtendedCapability(ExtendedCapabilityId_L1Substates);
    epL1ssBase = pEf->FindExtendedCapability(ExtendedCapabilityId_L1Substates);
    if((bL1ssBase != 0) && (epL1ssBase != 0))
    {
        pBf->ReadPCIeExtendedCapabilityDWord(bL1ssBase + ExtendedCapabilityOffset_L1SSCap, &bL1ssCap);
        pEf->ReadPCIeExtendedCapabilityDWord(epL1ssBase + ExtendedCapabilityOffset_L1SSCap, &epL1ssCap);
        isL1ssCapable = (bL1ssCap & ExtendedCapabilityValue_L1SSCap) && (epL1ssCap & ExtendedCapabilityValue_L1SSCap) && (IsAspmL11Allowed || IsAspmL12Allowed);
    }

    // resolve LTR capability
    pBf->ReadPCIeCapabilityDWord(PcieOffset_DeviceCapability2, &bDevCap2);
    pEf->ReadPCIeCapabilityDWord(PcieOffset_DeviceCapability2, &epDevCap2);
    isL1ssLtrCapable = (bDevCap2 & PcieValue_DeviceCapability2LatencyToleranceReporting) &&
        (epDevCap2 & PcieValue_DeviceCapability2LatencyToleranceReporting) && isL1ssCapable;
    NN_PCIE_LOG_INFO("Tegra21xRootComplex::SetAspmEnable(isEnabled=%d) - port=%d, isAspmCapable=%d, isL1ssCapable=%d, isL1ssLtrCapable=%d.\n",
                     isEnabled, portNum, isAspmCapable, isL1ssCapable, isL1ssLtrCapable);

    // Is ASPM to be enabled?
    if (isEnabled)
    {
        // Do L1SS enable processing
        if(isL1ssCapable)
        {
            uint32_t l1ssCapOverrideMask = ( (!IsAspmL11Allowed) ? ExtendedCapabilityValue_L1SSCapL11ASPM : 0) | ( (!IsAspmL12Allowed) ? ExtendedCapabilityValue_L1SSCapL12ASPM : 0);
            uint32_t l1ssCap = (bL1ssCap & epL1ssCap) & ~l1ssCapOverrideMask;
            if(l1ssCap & (ExtendedCapabilityValue_L1SSCapL11ASPM|ExtendedCapabilityValue_L1SSCapL12ASPM))
            {
                NN_PCIE_LOG_INFO("Port %d: Enabling%s%s\n",
                                 portNum,
                                 (l1ssCap & ExtendedCapabilityValue_L1SSCapL11ASPM) ? " L1.1" : "",
                                 (l1ssCap & ExtendedCapabilityValue_L1SSCapL12ASPM) ? " L1.2" : "");

                // 2. Set common clock
                NN_PCIE_RETURN_UPON_ERROR(ConfigureCommonClock(pBf, pEf));

                // 3. Disable L1SS while we modify configuration regs.  Upstream first, then downstream.
                NN_PCIE_RETURN_UPON_ERROR(pEf->WritePCIeExtendedCapabilityDWord(epL1ssBase + ExtendedCapabilityOffset_L1SSCtrl1,
                                                                                0x0,
                                                                                ExtendedCapabilityValue_L1SSCapL11ASPM |
                                                                                ExtendedCapabilityValue_L1SSCapL12ASPM |
                                                                                ExtendedCapabilityValue_L1SSCapL11PM   |
                                                                                ExtendedCapabilityValue_L1SSCapL12PM));
                NN_PCIE_RETURN_UPON_ERROR(pBf->WritePCIeExtendedCapabilityDWord(bL1ssBase + ExtendedCapabilityOffset_L1SSCtrl1,
                                                                                0x0,
                                                                                ExtendedCapabilityValue_L1SSCapL11ASPM |
                                                                                ExtendedCapabilityValue_L1SSCapL12ASPM |
                                                                                ExtendedCapabilityValue_L1SSCapL11PM   |
                                                                                ExtendedCapabilityValue_L1SSCapL12PM));

                // 4. Enable dynamic clock request mechanism
                NN_PCIE_RETURN_UPON_ERROR(DoPortOperation(portNum, PORT_OP_ENABLE_CLOCK_AUTOMATIC));

                // 5. Set T_POWER_ON
                uint32_t rpScaleVal  = L1SSTPowerOnScaledValue(bL1ssCap);
                uint32_t devScaleVal = L1SSTPowerOnScaledValue(epL1ssCap);
                NN_PCIE_LOG_VERBOSE("TPOWER_VAL: rp=%d dev=%d\n", rpScaleVal, devScaleVal);
                uint32_t *pCap = (rpScaleVal > devScaleVal) ? &bL1ssCap : &epL1ssCap;           // Select max tpwron (rp, dev)and set ctrl regs
                uint32_t ctrlVal = (((*pCap >> ExtendedCapabilityValue_L1SSCapTPowerOnScaleShift) &
                                     ExtendedCapabilityValue_L1SSTPowerOnScaleMask) << ExtendedCapabilityValue_L1SSCtrl2TPowerOnScaleShift) |
                                   (((*pCap >> ExtendedCapabilityValue_L1SSCapTPowerOnValueShift) &
                                     ExtendedCapabilityValue_L1SSTPowerOnValueMask) << ExtendedCapabilityValue_L1SSCtrl2TPowerOnValueShift);
                NN_PCIE_RETURN_UPON_ERROR(pBf->WritePCIeExtendedCapabilityDWord(bL1ssBase + ExtendedCapabilityOffset_L1SSCtrl2,
                                                                                ctrlVal,
                                                                                ExtendedCapabilityValue_L1SSCtrl2TPowerOnVSMask));
                NN_PCIE_RETURN_UPON_ERROR(pEf->WritePCIeExtendedCapabilityDWord(epL1ssBase + ExtendedCapabilityOffset_L1SSCtrl2,
                                                                                ctrlVal,
                                                                                ExtendedCapabilityValue_L1SSCtrl2TPowerOnVSMask));

                // 6. Set common mode restore time
                // Tegra21x's value is 0xff (see TRM).  Reference codes sets 0xff.
                // Ok, Raptor says 30 us
                NN_PCIE_RETURN_UPON_ERROR(pBf->WritePCIeExtendedCapabilityDWord(bL1ssBase + ExtendedCapabilityOffset_L1SSCtrl1,
                                                                                0x1e << ExtendedCapabilityValue_L1SSPmSsCmRsTimeShift,
                                                                                ExtendedCapabilityValue_L1SSPmSsCmRsTimeMask << ExtendedCapabilityValue_L1SSPmSsCmRsTimeShift));

                // 7. Set L1.1/L1.2 threshold
                // Set to 106 us for all... approximately 4 (value) x 32ns (scale).  Reference for 106 us?
                // Ok, Raptor now says 55 us.
                uint32_t rpVal;
                rpVal = (0x37 << ExtendedCapabilityValue_L1SSCtrl1L12ThresValueShift) | (0x2 << ExtendedCapabilityValue_L1SSCtrl1L12ThresScaleShift);  // 55 x 1us
                NN_PCIE_RETURN_UPON_ERROR(pBf->WritePCIeExtendedCapabilityDWord(bL1ssBase + ExtendedCapabilityOffset_L1SSCtrl1,
                                                                                rpVal,
                                                                                (ExtendedCapabilityValue_L1SSCtrl1L12ThresValueMask << ExtendedCapabilityValue_L1SSCtrl1L12ThresValueShift) |
                                                                                (ExtendedCapabilityValue_L1SSCtrl1L12ThresScaleMask << ExtendedCapabilityValue_L1SSCtrl1L12ThresScaleShift)));
                NN_PCIE_RETURN_UPON_ERROR(pEf->WritePCIeExtendedCapabilityDWord(epL1ssBase + ExtendedCapabilityOffset_L1SSCtrl1,
                                                                                rpVal,
                                                                                (ExtendedCapabilityValue_L1SSCtrl1L12ThresValueMask << ExtendedCapabilityValue_L1SSCtrl1L12ThresValueShift) |
                                                                                (ExtendedCapabilityValue_L1SSCtrl1L12ThresScaleMask << ExtendedCapabilityValue_L1SSCtrl1L12ThresScaleShift)));

                // 8. Enable L1 substates
                // Enable downstream first, then upstream.
                // L1SS Mode(s) supported by both.
                NN_PCIE_RETURN_UPON_ERROR(pBf->WritePCIeExtendedCapabilityDWord(bL1ssBase + ExtendedCapabilityOffset_L1SSCtrl1,
                                                                                l1ssCap,
                                                                                ExtendedCapabilityValue_L1SSCapL11ASPM |
                                                                                ExtendedCapabilityValue_L1SSCapL12ASPM |
                                                                                ExtendedCapabilityValue_L1SSCapL11PM   |
                                                                                ExtendedCapabilityValue_L1SSCapL12PM));
                NN_PCIE_RETURN_UPON_ERROR(pEf->WritePCIeExtendedCapabilityDWord(epL1ssBase + ExtendedCapabilityOffset_L1SSCtrl1,
                                                                                l1ssCap,
                                                                                ExtendedCapabilityValue_L1SSCapL11ASPM |
                                                                                ExtendedCapabilityValue_L1SSCapL12ASPM |
                                                                                ExtendedCapabilityValue_L1SSCapL11PM   |
                                                                                ExtendedCapabilityValue_L1SSCapL12PM));
            }
        }

        // Enable ASPM L0s/L1 on link
        // We know from static values of rp and wireless link caps that L1 and L0s are supported. Enable both
        // Enable upstream first, then downstream.
        uint16_t linkControl;
        uint16_t myAspmValue = (bLinkValue & epLinkValue & PcieValue_LinkCapabilityAspmSupport) >> PcieValue_LinkCapabilityAspmSupport_Shift;

        // This is needed for L1SS only
        if(isL1ssCapable)
        {
            myAspmValue |= PcieValue_LinkControlEnableClockReq;
        }

        // Apply ASPM setting to bridge
        NN_PCIE_RETURN_UPON_ERROR(pBf->ReadPCIeCapabilityWord(PcieOffset_LinkControl, &linkControl));
        linkControl |= myAspmValue;
        NN_PCIE_RETURN_UPON_ERROR(pBf->WritePCIeCapabilityWord(PcieOffset_LinkControl, linkControl));

        // Apply ASPM setting to endpoint
        NN_PCIE_RETURN_UPON_ERROR(pEf->ReadPCIeCapabilityWord(PcieOffset_LinkControl, &linkControl));
        linkControl |= myAspmValue;
        NN_PCIE_RETURN_UPON_ERROR(pEf->WritePCIeCapabilityWord(PcieOffset_LinkControl, linkControl));

        // 10. Enable L1.2 LTR
        if(isL1ssCapable && isL1ssLtrCapable)
        {
            NN_PCIE_RETURN_UPON_ERROR(
                pBf->WritePCIeCapabilityDWord(PcieOffset_DeviceControl2,
                                              PcieValue_DeviceControl2LtrEnable,
                                              PcieValue_DeviceControl2LtrEnable)
            );
            NN_PCIE_RETURN_UPON_ERROR(
                pEf->WritePCIeCapabilityDWord(PcieOffset_DeviceControl2,
                                              PcieValue_DeviceControl2LtrEnable,
                                              PcieValue_DeviceControl2LtrEnable)
            );
        }

        if(nn::pcie::detail::s_LogLevel == nn::pcie::detail::LogLevel_Verbose)
        {
            MonitorPortCounters(pBf->GetDevNum(), true);
        }
    }
    else  // SetAspmEnable(false) case
    {
        if(nn::pcie::detail::s_LogLevel == nn::pcie::detail::LogLevel_Verbose)
        {
            MonitorPortCounters(pBf->GetDevNum(), false);
        }

        // Disable dynamic clock request mechanism
        DoPortOperation(portNum, PORT_OP_ENABLE_CLOCK_OVERRIDE);

        // Disable LTR
        if(isL1ssLtrCapable)
        {
            NN_PCIE_RETURN_UPON_ERROR(
                pBf->WritePCIeCapabilityDWord(PcieOffset_DeviceControl2,
                                              0,
                                              PcieValue_DeviceControl2LtrEnable)
            );
            NN_PCIE_RETURN_UPON_ERROR(
                pEf->WritePCIeCapabilityDWord(PcieOffset_DeviceControl2,
                                              0,
                                              PcieValue_DeviceControl2LtrEnable)
            );
        }

        // Disable L0s and L1
        uint16_t regVal16;
        // Disable L0s and L1 on endpoint
        NN_PCIE_RETURN_UPON_ERROR(pEf->ReadPCIeCapabilityWord(PcieOffset_LinkControl, &regVal16));
        regVal16 &= ~(PcieValue_LinkControlAspmL1Enable | PcieValue_LinkControlAspmL0sEnable | PcieValue_LinkControlEnableClockReq);
        NN_PCIE_RETURN_UPON_ERROR(pEf->WritePCIeCapabilityWord(PcieOffset_LinkControl, regVal16));

        // Disable L0s and L1 on bridge
        NN_PCIE_RETURN_UPON_ERROR(pBf->ReadPCIeCapabilityWord(PcieOffset_LinkControl, &regVal16));
        regVal16 &= ~(PcieValue_LinkControlAspmL1Enable | PcieValue_LinkControlAspmL0sEnable | PcieValue_LinkControlEnableClockReq);
        NN_PCIE_RETURN_UPON_ERROR(pBf->WritePCIeCapabilityWord(PcieOffset_LinkControl, regVal16));

        // Disable L1ss
        if(isL1ssCapable && (epL1ssBase != 0) && (bL1ssBase != 0))
        {
            // disable on endpoint
            NN_PCIE_RETURN_UPON_ERROR(
                pEf->WritePCIeExtendedCapabilityDWord(epL1ssBase + ExtendedCapabilityOffset_L1SSCtrl1,
                                                      0,
                                                      ExtendedCapabilityValue_L1SSCapL11ASPM |
                                                      ExtendedCapabilityValue_L1SSCapL12ASPM |
                                                      ExtendedCapabilityValue_L1SSCapL11PM   |
                                                      ExtendedCapabilityValue_L1SSCapL12PM)
            );

            // disable on bridge
            NN_PCIE_RETURN_UPON_ERROR(
                pBf->WritePCIeExtendedCapabilityDWord(bL1ssBase + ExtendedCapabilityOffset_L1SSCtrl1,
                                                      0,
                                                      ExtendedCapabilityValue_L1SSCapL11ASPM |
                                                      ExtendedCapabilityValue_L1SSCapL12ASPM |
                                                      ExtendedCapabilityValue_L1SSCapL11PM   |
                                                      ExtendedCapabilityValue_L1SSCapL12PM)
            );
        }
    }

    return result;
} // NOLINT(impl/function_size)

Result Tegra21xRootComplex::GetDmaBusAddressRange(BusAddress *pOutBase, BusAddress *pOutSize)
{
    *pOutBase = IoVaHeapBase;
    *pOutSize = IoVaHeapSize;
    return ResultSuccess();
}

Result Tegra21xRootComplex::GetLoggedState(RootComplexLoggedState* pOut, bool isCleared)
{
    *pOut = InvalidRootComplexLoggedState;

    // update per-port state with latest available information
    for (DeviceNumber portNum = 0; portNum < m_Config.numPorts; portNum++)
    {
        m_LoggedState.ports[portNum].speed = BusSpeed_Invalid;
        if (DoPortOperation(portNum, PORT_OP_GET_STATE).IsSuccess())
        {
            uint32_t regVal32 = 0;
            ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_LINK_CONTROL_STATUS_OFFSET, sizeof(uint32_t), &regVal32);
            switch ((regVal32 & RP_LINK_CONTROL_STATUS_SPEED_MASK) >> RP_LINK_CONTROL_STATUS_SPEED_SHIFT)
            {
            case PcieValue_LinkStatusCurrentLinkSpeed_2_5GB:
                m_LoggedState.ports[portNum].speed = BusSpeed_EGen1;
                break;
            case PcieValue_LinkStatusCurrentLinkSpeed_5_0GB:
                m_LoggedState.ports[portNum].speed = BusSpeed_EGen2;
                break;
            default:
                break;
            }
        }
    }

    // pass back logged state to caller
    *pOut = m_LoggedState;

    if(isCleared)
    {
        // clear cumulative counts
        memset(&m_LoggedState.globalCounts, 0, sizeof(m_LoggedState.globalCounts));

        // Clear per-port statistics
        for (DeviceNumber portNum = 0; portNum < m_Config.numPorts; portNum++)
        {
            m_LoggedState.ports[portNum].irqCount = 0;
            m_LoggedState.ports[portNum].afiMsgPmeCount = 0;
            m_LoggedState.ports[portNum].afiMsgIntxCount = 0;
            m_LoggedState.ports[portNum].afiMsgPcieFatalErrorCount = 0;
            m_LoggedState.ports[portNum].afiMsgPcieNonFatalErrorCount = 0;
            m_LoggedState.ports[portNum].afiMsgPcieCorrectableErrorCount = 0;
            m_LoggedState.ports[portNum].afiMsgPcieRootPortIntCount = 0;
            m_LoggedState.ports[portNum].afiMsgHotplugCount = 0;
        }

    }

    return ResultSuccess();
}

Result Tegra21xRootComplex::PortControl(DeviceNumber portNum, PortCommand portCommand)
{
    Result result = ResultSuccess();

    if(portCommand == PortCommand_HoldReset)
    {
        DoPortOperation(portNum, PORT_OP_ENABLE_CLOCK_OVERRIDE);
        DoPortOperation(portNum, PORT_OP_HOLD_RESET);
        NN_PCIE_LOG_INFO("Tegra21xRootComplex::PortControl() port %d reset activated.\n", portNum);
    }
    else if(portCommand == PortCommand_CycleReset)
    {
        bool linkUp = false;
        const char *pSpeed = "?";
        DoPortOperation(portNum, PORT_OP_ENABLE_CLOCK_OVERRIDE);
        DoPortOperation(portNum, PORT_OP_RESET);
        os::Tick startTick = nn::os::GetSystemTick();
        os::Tick timeoutTick = nn::os::ConvertToTick(nn::TimeSpan::FromMilliSeconds(MaxLinkWaitTimeInMs)) + startTick;

        // Release build will not reference these, mark as unused to avoid warning
        NN_UNUSED(pSpeed);
        NN_UNUSED(startTick);

        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(EndpointResetLtssmDetectTimeInUs));
        do
        {
            if (DoPortOperation(portNum, PORT_OP_GET_STATE).IsSuccess())
            {
                uint32_t regVal32 = 0;
                linkUp = true;
                ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_LINK_CONTROL_STATUS_OFFSET, sizeof(uint32_t), &regVal32);
                switch ((regVal32 & RP_LINK_CONTROL_STATUS_SPEED_MASK) >> RP_LINK_CONTROL_STATUS_SPEED_SHIFT)
                {
                case PcieValue_LinkStatusCurrentLinkSpeed_2_5GB:
                    pSpeed = "GEN1";
                    m_LoggedState.ports[portNum].speed = BusSpeed_EGen1;
                    break;
                case PcieValue_LinkStatusCurrentLinkSpeed_5_0GB:
                    pSpeed = "GEN2";
                    m_LoggedState.ports[portNum].speed = BusSpeed_EGen2;
                    break;
                default:
                    break;
                }
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(LinkPollIntervalInMs));
        }while (!linkUp && (nn::os::GetSystemTick() < timeoutTick));

        if(linkUp)
        {
            uint32_t portResetTimeInUs = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - startTick).GetMicroSeconds();
            NN_PCIE_LOG_INFO("Tegra21xRootComplex::PortControl() port %d, UP in %dus, speed %s.\n",
                             portNum, portResetTimeInUs, pSpeed);
            m_LoggedState.ports[portNum].flagSet.Set<nn::pcie::RootPortStateFlag::IsEnabled>(true);
            m_LoggedState.ports[portNum].portResetTimeInUs = portResetTimeInUs;
        }
        else
        {
            NN_PCIE_LOG_INFO("Tegra21xRootComplex::PortControl() port %d, DOWN.\n", portNum);
            DoPortOperation(portNum, PORT_OP_DISABLE_CLOCK);
            m_LoggedState.ports[portNum].flagSet.Set<nn::pcie::RootPortStateFlag::IsEnabled>(false);
            m_LoggedState.ports[portNum].speed = BusSpeed_Invalid;
            result = ResultOperationTimeout();
        }
    }
    else
    {
        result = ResultBadInternalArgument();
    }

    return result;
}

Result Tegra21xRootComplex::InitAFIBars()
{
    Result result = ResultSuccess();
    do
    {
        // Disable unused AFI BARs
        NN_PCI_AFI_AXI_BAR2_SZ_REG = 0;
        NN_PCI_AFI_AXI_BAR3_SZ_REG = 0;
        NN_PCI_AFI_AXI_BAR4_SZ_REG = 0;
        NN_PCI_AFI_AXI_BAR5_SZ_REG = 0;
        NN_PCI_AFI_AXI_BAR6_SZ_REG = 0;
        NN_PCI_AFI_AXI_BAR7_SZ_REG = 0;
        NN_PCI_AFI_AXI_BAR8_SZ_REG = 0;
        NN_PCI_AFI_AXI_BAR2_START_REG = 0;
        NN_PCI_AFI_AXI_BAR3_START_REG = 0;
        NN_PCI_AFI_AXI_BAR4_START_REG = 0;
        NN_PCI_AFI_AXI_BAR5_START_REG = 0;
        NN_PCI_AFI_AXI_BAR6_START_REG = 0;
        NN_PCI_AFI_AXI_BAR7_START_REG = 0;
        NN_PCI_AFI_AXI_BAR8_START_REG = 0;
        NN_PCI_AFI_FPCI_BAR2_REG = 0;
        NN_PCI_AFI_FPCI_BAR3_REG = 0;
        NN_PCI_AFI_FPCI_BAR4_REG = 0;
        NN_PCI_AFI_FPCI_BAR5_REG = 0;
        NN_PCI_AFI_FPCI_BAR6_REG = 0;
        NN_PCI_AFI_FPCI_BAR7_REG = 0;
        NN_PCI_AFI_FPCI_BAR8_REG = 0;

        // Disable MSI for now
        NN_PCI_AFI_MSI_BAR_SZ_REG      = 0;
        NN_PCI_AFI_MSI_FPCI_BAR_ST_REG = 0;
        NN_PCI_AFI_MSI_AXI_BAR_ST_REG  = 0;

        // AFI BAR0: m_ConfigSpaceStart
        NN_PCI_AFI_FPCI_BAR0_REG       = AFI_FPCI_BAR_TYPE1_EXT_CFG;
        NN_PCI_AFI_AXI_BAR0_SZ_REG     = (Params_ConfigSpaceSize >> AFI_AXI_BAR_SZ_SHIFT);
        NN_PCI_AFI_AXI_BAR0_START_REG  = m_ConfigSpaceStartPhys;

        // AFI BAR1: PCI_IO_SPACE
        NN_PCI_AFI_FPCI_BAR1_REG       = AFI_FPCI_BAR_MMMIO_CFG;
        NN_PCI_AFI_AXI_BAR1_SZ_REG     = (Params_IoSpaceSize >> AFI_AXI_BAR_SZ_SHIFT);
        NN_PCI_AFI_AXI_BAR1_START_REG  = m_IoSpaceStartPhys;

        // AFI BAR2: PCI_PREF_MEM
        NN_PCI_AFI_FPCI_BAR2_REG       = (((m_PrefetchMemStartPhys >> 12) & 0xfffff) << 4) | 0x1;
        NN_PCI_AFI_AXI_BAR2_SZ_REG     = (Params_PrefetchMemSize >> AFI_AXI_BAR_SZ_SHIFT);
        NN_PCI_AFI_AXI_BAR2_START_REG  = m_PrefetchMemStartPhys;

        // AFI BAR3: PCI_NONPREF_MEM
        NN_PCI_AFI_FPCI_BAR3_REG       = (((m_NoPrefetchMemStartPhys >> 12) & 0xfffff) << 4) | 0x1;
        NN_PCI_AFI_AXI_BAR3_SZ_REG     = (Params_NoPrefetchMemSize >> AFI_AXI_BAR_SZ_SHIFT);
        NN_PCI_AFI_AXI_BAR3_START_REG  = m_NoPrefetchMemStartPhys;

    } while (false);
    return result;
}

Result Tegra21xRootComplex::InitAFIRegisters()
{
    Result result = ResultSuccess();
    uint32_t afiFuseRegVal = 0;
    uint32_t laneMap = 0;

    // Speed limiting
    afiFuseRegVal = NN_PCI_AFI_FUSE_REG;
    if (m_Config.speedLimit < BusSpeed_EGen2)
    {
        afiFuseRegVal |= AFI_FUSE_PCIE_T0_GEN2_DIS;
    }
    else
    {
        afiFuseRegVal &= ~AFI_FUSE_PCIE_T0_GEN2_DIS;
    }
    NN_PCI_AFI_FUSE_REG = afiFuseRegVal;

    // Enable devices
    for (DeviceNumber portNum = 0; portNum < m_Config.numPorts; portNum++)
    {
        if (m_Config.ports[portNum].isEnabled)
        {
            NN_PCI_AFI_PCIE_CONFIG_REG &= ~(1 << ((portNum) + 1));
        }
        else
        {
            NN_PCI_AFI_PCIE_CONFIG_REG |= (1 << ((portNum) + 1));
        }
    }

    // XBAR config
    for (int32_t portIndex = 0; portIndex < m_Config.numPorts; portIndex++)
    {
        laneMap |= ((m_Config.ports[portIndex].laneCount) << (portIndex * 8));
    }
    switch (laneMap)
    {
    case 0x0000104:
        NN_PCI_AFI_PCIE_CONFIG_REG = (NN_PCI_AFI_PCIE_CONFIG_REG & ~AFI_PCIE_CONFIG_SM2TMS0_XBAR_CONFIG_MASK) |
            AFI_PCIE_CONFIG_SM2TMS0_XBAR_CONFIG_X4_X1;
        break;
    case 0x0000102:
        NN_PCI_AFI_PCIE_CONFIG_REG = (NN_PCI_AFI_PCIE_CONFIG_REG & ~AFI_PCIE_CONFIG_SM2TMS0_XBAR_CONFIG_MASK) |
            AFI_PCIE_CONFIG_SM2TMS0_XBAR_CONFIG_X2_X1;
        break;
    default:
        result = ResultBadControllerConfig();
        break;
    }

    // Implement AFI Clock Gate Hang WAR, per SIGLO-62839, NSBG-7267
    NN_PCI_AFI_CONFIGURATION_REG |= AFI_CONFIGURATION_CLKEN_OVERRIDE;

    // Enable FPCI interface
    NN_PCI_AFI_CONFIGURATION_REG |= AFI_CONFIGURATION_EN_FPCI;

    // Set register for PLL power down
    NN_PCI_AFI_PLLE_CONTROL_REG =
        (NN_PCI_AFI_PLLE_CONTROL_REG & ~(AFI_PLLE_CONTROL_BYPASS_PADS2PLLE_CONTROL   |
                                         AFI_PLLE_CONTROL_BYPASS_PCIE2PLLE_CONTROL)) |
        AFI_PLLE_CONTROL_PADS2PLLE_CONTROL_EN |
        AFI_PLLE_CONTROL_PCIE2PLLE_CONTROL_EN;

    // Power up bias
    NN_PCI_AFI_PEXBIAS_CTRL_REG &= ~AFI_PEXBIAS_CTRL_PWRD;

    // Override refclk initially as some EP may not assert clkreq on boot and for platform
    // not supporting clock power mananagement
    NN_PCI_AFI_PEX0_CTRL_REG |= AFI_PEX_CTRL_REFCLK_OVR_EN;
    NN_PCI_AFI_PEX1_CTRL_REG |= AFI_PEX_CTRL_REFCLK_OVR_EN;

    // Enable refclk and clkreq pad
    NN_PCI_AFI_PEX0_CTRL_REG =
        (NN_PCI_AFI_PEX0_CTRL_REG & ~AFI_PEX_CTRL_CLKREQ_EN) | AFI_PEX_CTRL_REFCLK_EN;
    NN_PCI_AFI_PEX1_CTRL_REG =
        (NN_PCI_AFI_PEX1_CTRL_REG & ~AFI_PEX_CTRL_CLKREQ_EN) | AFI_PEX_CTRL_REFCLK_EN;

    // Assert EP resets
    NN_PCI_AFI_PEX0_CTRL_REG &= ~AFI_PEX_CTRL_RST_L;
    NN_PCI_AFI_PEX1_CTRL_REG &= ~AFI_PEX_CTRL_RST_L;

    return result;
}

Result Tegra21xRootComplex::InitPCIeRegisters(DeviceNumber portNum)
{
    Result result = ResultSuccess();

    // Configure refclk pad
    NN_PCI_PADS_REFCLK_CFG0 = m_PadsRefClkCfg0;

    // Configure UPHY electrical settings
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_ECTL_2_R1_0_OFFSET, sizeof(uint32_t), 0x0000000F);
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_ECTL_2_R2_0_OFFSET, sizeof(uint32_t), 0x0000008F);
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_ECTL_5_R1_0_OFFSET, sizeof(uint32_t), 0x55010000);
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_ECTL_6_R1_0_OFFSET, sizeof(uint32_t), 0x00000001);
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_ECTL_5_R2_0_OFFSET, sizeof(uint32_t), 0x55010000);
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_ECTL_6_R2_0_OFFSET, sizeof(uint32_t), 0x00000001);
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_ECTL_4_R1_0_OFFSET, sizeof(uint32_t), 0x00670000);
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_ECTL_4_R2_0_OFFSET, sizeof(uint32_t), 0x00C70000);

    // Enable PCA
    // Historical: It was said it could be disabled after enumeration to save power
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_VEND_CTL2_OFFSET, 0, VEND_CTL2_PCA_ENABLE);

    // Enable clock clamping and change threshold for TMS clock clamping
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_PRIV_MISC_OFFSET, sizeof(uint32_t),
                     RP_PRIV_MISC_TMS_CLK_CLAMP_ENABLE | RP_PRIV_MISC_CTLR_CLK_CLAMP_ENABLE |
                     (m_RpClkClampThreshold << RP_PRIV_MISC_TMS_CLK_CLAMP_THRESHOLD_SHIFT) |
                     (m_RpClkClampThreshold << RP_PRIV_MISC_CTLR_CLK_CLAMP_THRESHOLD_SHIFT));

    // TRM says "Configure CYA to enable ASPM". What we are really doing is
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_VEND_XP1_OFFSET, sizeof(uint32_t), 0x40200150);

    // Set BIST
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_VEND_XP_BIST_OFFSET, sizeof(uint32_t),
                     RP_VEND_XP_BIST_GOTO_L1_L2_AFTER_DLLP_DONE);

    // Enable opportunistic update FC and ACK
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_VEND_XP_OFFSET,
                             0, RP_VEND_XP_OPPORTUNISTIC_UPDATEFC | RP_VEND_XP_OPPORTUNISTIC_ACK);

    // Update flow control threshold
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_VEND_XP_OFFSET,
                             RP_VEND_XP_UPDATE_FC_THRESHOLD_MASK,
                             (0x60 << RP_VEND_XP_UPDATE_FC_THRESHOLD_SHIFT));

    // Software tuning for the de-skew retry time
    // Historical reference:
    //   Take care of link speed change error in corner cases
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_VEND_CTL0_OFFSET,
                             RP_VEND_CTL0_DSK_RST_PULSE_WIDTH_MASK,
                             0x9 << RP_VEND_CTL0_DSK_RST_PULSE_WIDTH_SHIFT);

    // Power saving for sleep/idle
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_VEND_XP_PAD_PWRDN_OFFSET,
                             RP_VEND_XP_PAD_PWRDN_DISABLED_MASK,
                             1 << RP_VEND_XP_PAD_PWRDN_DISABLED_SHIFT);
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_VEND_XP_PAD_PWRDN_OFFSET,
                             RP_VEND_XP_PAD_PWRDN_DYNAMIC_MASK,
                             1 << RP_VEND_XP_PAD_PWRDN_DYNAMIC_SHIFT);
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_VEND_XP_PAD_PWRDN_OFFSET,
                             RP_VEND_XP_PAD_PWRDN_L1_MASK,
                             1 << RP_VEND_XP_PAD_PWRDN_L1_SHIFT);

    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_VEND_XP_PAD_PWRDN_OFFSET,
                             RP_VEND_XP_PAD_PWRDN_SLEEP_MODE_DYNAMIC_MASK,
                             3 << RP_VEND_XP_PAD_PWRDN_SLEEP_MODE_DYNAMIC_SHIFT);
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_VEND_XP_PAD_PWRDN_OFFSET,
                             RP_VEND_XP_PAD_PWRDN_SLEEP_MODE_L1_MASK,
                             3 << RP_VEND_XP_PAD_PWRDN_SLEEP_MODE_L1_SHIFT);
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_VEND_XP_PAD_PWRDN_OFFSET,
                             RP_VEND_XP_PAD_PWRDN_SLEEP_MODE_L1_CLKREQ_MASK,
                             3 << RP_VEND_XP_PAD_PWRDN_SLEEP_MODE_L1_CLKREQ_SHIFT);

    // Settings for the 19.2 MHz CLK25M clock frequency
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_TIMEOUT0_OFFSET, sizeof(uint32_t),
                     (0xa << RP_TIMEOUT0_PAD_PWRUP_SHIFT) |
                     (0x180 << RP_TIMEOUT0_PAD_PWRUP_CM_SHIFT) |
                     (0xa << RP_TIMEOUT0_PAD_SPDCHNG_GEN2_SHIFT));
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_TIMEOUT1_OFFSET, sizeof(uint32_t),
                     (0x10 << RP_TIMEOUT1_RCVRY_SPD_SUCCESS_EIDLE_SHIFT) |
                     (0x74 << RP_TIMEOUT1_RCVRY_SPD_UNSUCCESS_EIDLE_SHIFT) |
                     (0x2e8 << RP_TIMEOUT1_PAD_SPDCHNG_GEN1_SHIFT));

    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_XP_REF_OFFSET,
                             RP_XP_REF_CPL_TO_OVERRIDE_MASK,
                             1 << RP_XP_REF_CPL_TO_OVERRIDE_SHIFT);

    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_XP_REF_OFFSET,
                             RP_XP_REF_CPL_TO_CUSTOM_VALUE_MASK,
                             0x1770 << RP_XP_REF_CPL_TO_CUSTOM_VALUE_SHIFT);
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_XP_REF_OFFSET,
                             RP_XP_REF_MICROSECOND_LIMIT_MASK,
                             0x14 << RP_XP_REF_MICROSECOND_LIMIT_SHIFT);
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_XP_REF_OFFSET,
                             RP_XP_REF_MICROSECOND_MASK,
                             1 << RP_XP_REF_MICROSECOND_SHIFT);

    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_L1_PM_SUBSTATES_2_CYA_OFFSET,
                            RP_L1_PM_SUBSTATES_2_CYA_T_L1_2_DLY_MASK,
                            0x4d << RP_L1_PM_SUBSTATES_2_CYA_T_L1_2_DLY_SHIFT);
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_L1_PM_SUBSTATES_2_CYA_OFFSET,
                            RP_L1_PM_SUBSTATES_2_CYA_MICROSECOND_MASK,
                            0x13 << RP_L1_PM_SUBSTATES_2_CYA_MICROSECOND_SHIFT);
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_L1_PM_SUBSTATES_2_CYA_OFFSET,
                            RP_L1_PM_SUBSTATES_2_CYA_MICROSECOND_COMP_MASK,
                            0x2 << RP_L1_PM_SUBSTATES_2_CYA_MICROSECOND_COMP_SHIFT);

    // Set CLKREQ assertion delay for L1SS abort case
    // Historical reference:
    //   Bug#1461732 WAR, set clkreq asserted delay greater than
    //   power off time (2us) to avoid RP wakeup in L1.2_ENTRY
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_L1_PM_SUBSTATES_1_CYA_OFFSET,
                             RP_L1_PM_SUBSTATES_1_CYA_CLKREQ_ASSERTED_DLY_MASK,
                             0x4f << RP_L1_PM_SUBSTATES_1_CYA_CLKREQ_ASSERTED_DLY_SHIFT); // TRM also calls for 0x27
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_L1_PM_SUBSTATES_1_CYA_OFFSET,
                             RP_L1_PM_SUBSTATES_1_CYA_PWR_OFF_DLY_MASK,
                             0x26 << RP_L1_PM_SUBSTATES_1_CYA_PWR_OFF_DLY_SHIFT);

    // Unhide AER capability structure in the capabilities list
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_VEND_CTL1_OFFSET,
                             RP_VEND_CTL1_ERPT_MASK,
                             1 << RP_VEND_CTL1_ERPT_SHIFT);

    // Program presense map
    SetPresentMapOverride(portNum, true);

    // Program timers for L1 substate support
    // set cm_rtime = 30us and t_pwr_on = 70us as per HW team
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_L1_PM_SUBSTATES_CYA_OFFSET,
                             RP_L1_PM_SUBSTATES_CYA_CM_RTIME_MASK,
                             0x1E << RP_L1_PM_SUBSTATES_CYA_CM_RTIME_SHIFT);
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_L1_PM_SUBSTATES_CYA_OFFSET,
                             RP_L1_PM_SUBSTATES_CYA_T_PWRN_SCL_MASK |
                             RP_L1_PM_SUBSTATES_CYA_T_PWRN_VAL_MASK,
                             (1 << RP_L1_PM_SUBSTATES_CYA_T_PWRN_SCL_SHIFT) |
                             (7 << RP_L1_PM_SUBSTATES_CYA_T_PWRN_VAL_SHIFT));

    // TRM states that BIOS that does not support int line should set to 0xff.
    WriteConfigSpaceMasked32(0, portNum, 0, T_PCIE2_RP_INTR_BCR_OFFSET,
                             RP_INTR_BCR_INTR_LINE_MASK,
                             0xff << RP_INTR_BCR_INTR_LINE_SHIFT);

    return result;
}


Result Tegra21xRootComplex::InitPCIRegisters(DeviceNumber portNum)
{
    Result result = ResultSuccess();

    // Initialize command register
    WriteConfigSpace(0, portNum, 0, StandardConfigOffset_Command, sizeof(uint16_t), 0);

    // Clear primary, secondary and subordinate bus numbers
    WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_PrimaryBusNumber, sizeof(uint32_t), 0);

    // Clear primary, secondary and subordinate bus numbers
    WriteConfigSpace(0, portNum, 0, StandardConfigOffset_CacheLineSize, sizeof(uint32_t), 0);

    // Set IO base and limits to default zero, also clear secondary-side device status
    WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_IoBase, sizeof(uint32_t), 0xf9000000);
    WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_IoBaseUpper16, sizeof(uint16_t), 0);
    WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_IoLimitUpper16, sizeof(uint16_t), 0);

    // Set MEM base and limit to defaults
    WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_MemoryBase, sizeof(uint16_t), 0x00000010);
    WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_MemoryLimit, sizeof(uint16_t), 0);
    WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_PrefetchMemoryBase, sizeof(uint16_t), 0x00000010);
    WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_PrefetchMemoryLimit, sizeof(uint16_t), 0);
    WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_PrefetchBaseUpper32, sizeof(uint32_t), 0);
    WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_PrefetchLimitUpper32, sizeof(uint32_t), 0);

    // Set interrupt line and pin
    WriteConfigSpace(0, portNum, 0, Type0ConfigOffset_InterruptLine, sizeof(uint8_t), 0);
    WriteConfigSpace(0, portNum, 0, Type0ConfigOffset_InterruptPin, sizeof(uint8_t), 0);

    // Set bridge control to default
    WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_BridgeControl, sizeof(uint16_t), 0);

    // Clear stale status
    WriteConfigSpace(0, portNum, 0, StandardConfigOffset_Status, sizeof(uint16_t),
                     StandardConfigValue_StatusDetectedParity | StandardConfigValue_StatusSigSystemError |
                     StandardConfigValue_StatusRecMasterAbort | StandardConfigValue_StatusRecTargetAbort |
                     StandardConfigValue_StatusSigTargetAbort | StandardConfigValue_StatusParity);

    return result;
} // NOLINT(impl/function_size)

Result Tegra21xRootComplex::InitMSI()
{
    Result result = ResultSuccess();

    if ((m_pMsiBuffer = nn::pcie::detail::MemoryAllocAligned(Params_MsiBufferSize,
                                                             Params_MsiBufferSize,
                                                             "Tegra21xRootComplex Msi buffer")) != NULL)
    {
        memset(m_pMsiBuffer, 0, Params_MsiBufferSize);
        nn::dd::FlushDataCache(m_pMsiBuffer, Params_MsiBufferSize);
#if !defined NN_PCI_SMMU_DISABLED
        NN_PCIE_ABORT_UPON_ERROR(
            nn::dd::MapDeviceAddressSpaceNotAligned(
                &m_DeviceAddressSpace,
                nn::dd::GetCurrentProcessHandle(),
                reinterpret_cast<uintptr_t>(m_pMsiBuffer),
                Params_MsiBufferSize,
                IoVaInternalBase,
                nn::dd::MemoryPermission_ReadWrite));
#endif
        NN_PCI_AFI_MSI_FPCI_BAR_ST_REG = (IoVaInternalBase >> 8);
        NN_PCI_AFI_MSI_AXI_BAR_ST_REG  = IoVaInternalBase;
        NN_PCI_AFI_MSI_BAR_SZ_REG      = 1;

        // These get enabled on-demand later
        NN_PCI_AFI_MSI_EN_VEC0_REG     = 0;
        NN_PCI_AFI_MSI_EN_VEC1_REG     = 0;
        NN_PCI_AFI_MSI_EN_VEC2_REG     = 0;
        NN_PCI_AFI_MSI_EN_VEC3_REG     = 0;
        NN_PCI_AFI_MSI_EN_VEC4_REG     = 0;
        NN_PCI_AFI_MSI_EN_VEC5_REG     = 0;
        NN_PCI_AFI_MSI_EN_VEC6_REG     = 0;
        NN_PCI_AFI_MSI_EN_VEC7_REG     = 0;
    }
    else
    {
        result = ResultMemAllocFailure();
    }

    return result;
}

void Tegra21xRootComplex::ResolveMemMap()
{
    Result result = ResultSuccess();
    // Get virtual mapping of bases
    NN_PCIE_ABORT_IF_ZERO(m_PcieA1Base = nn::dd::QueryIoMappingAddress(NvPhysIo_PcieA1Addr, NvPhysIo_PcieA1Size));
    NN_PCIE_ABORT_IF_ZERO(m_PcieA2Base = nn::dd::QueryIoMappingAddress(NvPhysIo_PcieA2Addr, NvPhysIo_PcieA2Size));
    NN_PCIE_ABORT_IF_ZERO(m_PcieA3Base = nn::dd::QueryIoMappingAddress(NvPhysIo_PcieA3Addr, NvPhysIo_PcieA3Size));
    NN_PCIE_ABORT_IF_ZERO(m_XusbPadctlBase = nn::dd::QueryIoMappingAddress(NvPhysIo_XusbPadctlAddr, NvPhysIo_XusbPadctlSize));
    NN_PCIE_ABORT_IF_ZERO(m_PinMuxAuxBase = nn::dd::QueryIoMappingAddress(NvPhysIo_PinMuxAuxAddr, NvPhysIo_PinMuxAuxSize));

    // Derived base addresses of specific blocks
    m_PcieCtrl0CfgBase = m_PcieA1Base + 0x00000000;
    m_PcieCtrl1CfgBase = m_PcieA1Base + 0x00001000;
    m_PciePca0Base     = m_PcieA1Base + 0x00002000;
    m_PciePca1Base     = m_PcieA1Base + 0x00002100;
    m_PciePadsBase     = m_PcieA1Base + 0x00003000;
    m_PcieAfiBase      = m_PcieA1Base + 0x00003800;

    // Definition of fundamental config/IO/MEM regions
    m_ConfigSpaceStart       = reinterpret_cast<uint8_t *>(m_PcieA2Base);
    m_ConfigSpaceStartPhys   = NvPhysIo_PcieA2Addr;
    m_ConfigSpaceEnd         = m_ConfigSpaceStart + Params_ConfigSpaceSize;

    m_IoSpaceStart           = reinterpret_cast<uint8_t *>(m_ConfigSpaceEnd);
    m_IoSpaceStartPhys       = NvPhysIo_PcieA2Addr + Params_ConfigSpaceSize;

    m_PrefetchMemStart       = reinterpret_cast<uint8_t *>(m_PcieA3Base);
    m_PrefetchMemStartPhys   = NvPhysIo_PcieA3Addr;

    m_NoPrefetchMemStart     = reinterpret_cast<uint8_t *>(m_PcieA3Base + Params_PrefetchMemSize);
    m_NoPrefetchMemStartPhys = NvPhysIo_PcieA3Addr + Params_PrefetchMemSize;

    NN_PCIE_LOG_VERBOSE("Tegra21xRootComplex::ResolveMemMap():\n");
    NN_PCIE_LOG_VERBOSE("  m_ConfigSpaceStart       = 0x%p\n", m_ConfigSpaceStart);
    NN_PCIE_LOG_VERBOSE("  m_ConfigSpaceStartPhys   = 0x%p\n", reinterpret_cast<void*>(m_ConfigSpaceStartPhys));
    NN_PCIE_LOG_VERBOSE("  m_IoSpaceStart           = 0x%p\n", m_IoSpaceStart);
    NN_PCIE_LOG_VERBOSE("  m_IoSpaceStartPhys       = 0x%p\n", reinterpret_cast<void*>(m_IoSpaceStartPhys));
    NN_PCIE_LOG_VERBOSE("  m_PrefetchMemStart       = 0x%p\n", m_PrefetchMemStart);
    NN_PCIE_LOG_VERBOSE("  m_PrefetchMemStartPhys   = 0x%p\n", reinterpret_cast<void*>(m_PrefetchMemStartPhys));
    NN_PCIE_LOG_VERBOSE("  m_NoPrefetchMemStart     = 0x%p\n", m_NoPrefetchMemStart);
    NN_PCIE_LOG_VERBOSE("  m_NoPrefetchMemStartPhys = 0x%p\n", reinterpret_cast<void*>(m_NoPrefetchMemStartPhys));

#if !defined NN_PCI_SMMU_DISABLED
    NN_PCIE_ABORT_UPON_ERROR(nn::dd::CreateDeviceAddressSpace(&m_DeviceAddressSpace,
                                                              0x100000000ull));
    NN_PCIE_ABORT_UPON_ERROR(nn::dd::AttachDeviceAddressSpace(&m_DeviceAddressSpace,
                                                              nn::dd::DeviceName::DeviceName_Afi));
#endif

}


Result Tegra21xRootComplex::SetInterruptVectorEnable(bool isEnabled)
{
    Result result = ResultSuccess();

    do
    {
        if(isEnabled)
        {
            // MSI
            nn::os::InitializeInterruptEvent(&m_InterruptEvent[InterruptEvent_Msi],
                                             MsiInterruptName,
                                             nn::os::EventClearMode_ManualClear);
            NN_PCIE_BREAK_UPON_ERROR(m_pDriver->RegisterInterruptEvent(&m_InterruptEvent[InterruptEvent_Msi], IntVector_Msi));

            // INT
            nn::os::InitializeInterruptEvent(&m_InterruptEvent[InterruptEvent_Int],
                                             MainInterruptName,
                                             nn::os::EventClearMode_ManualClear);
            NN_PCIE_BREAK_UPON_ERROR(m_pDriver->RegisterInterruptEvent(&m_InterruptEvent[InterruptEvent_Int], IntVector_Int));

            // WAKE
            nn::os::InitializeInterruptEvent(&m_InterruptEvent[InterruptEvent_Wake],
                                             WakeInterruptName,
                                             nn::os::EventClearMode_ManualClear);
            NN_PCIE_BREAK_UPON_ERROR(m_pDriver->RegisterInterruptEvent(&m_InterruptEvent[InterruptEvent_Wake], IntVector_Wake));
        }
        else
        {
            // MSI
            NN_PCIE_BREAK_UPON_ERROR(m_pDriver->UnregisterInterruptEvent(IntVector_Msi));
            nn::os::FinalizeInterruptEvent(&m_InterruptEvent[InterruptEvent_Msi]);

            // INT
            NN_PCIE_BREAK_UPON_ERROR(m_pDriver->UnregisterInterruptEvent(IntVector_Int));
            nn::os::FinalizeInterruptEvent(&m_InterruptEvent[InterruptEvent_Int]);

            // WAKE
            NN_PCIE_BREAK_UPON_ERROR(m_pDriver->UnregisterInterruptEvent(IntVector_Wake));
            nn::os::FinalizeInterruptEvent(&m_InterruptEvent[InterruptEvent_Wake]);
        }
    }while (false);

    return result;
}

uint32_t Tegra21xRootComplex::GetEndpointFunctionEnableMask(int32_t baseResourceIndex)
{
    uint32_t enableMask = 0;
    for(int32_t i=0; i < nn::pcie::MaxIrqVectorsPerDeviceFunction; i++)
    {
        int32_t resourceIndex = baseResourceIndex + i;
        int32_t vecBit = 1 << (resourceIndex % AFI_MSI_VECTORS_PER_REGISTER);
        volatile uint32_t *pVecEnReg = &NN_PCI_AFI_MSI_EN_VEC0_REG + (resourceIndex >> AFI_MSI_VECTORS_PER_REGISTER_POWEROFTWO);
        uint32_t vectorEnMask = *pVecEnReg;
        if(vectorEnMask & vecBit) enableMask |= (1 << i);
    }
    return enableMask;
}

Result Tegra21xRootComplex::ComputeConfigSpaceAddress(volatile uint32_t** ppOut, BusNumber busNum, DeviceNumber devNum,
                                                      FunctionNumber funcNum, int32_t where)
{
    Result result = ResultInvalidRegisterOffset();
    volatile uint32_t *pDWordReg = NULL;

    // Configuration space offset limited to 4k, per PCIe specification
    if((where >= 0) && (where < 4096))
    {
        if (busNum == 0)
        {
            if (funcNum == 0)
            {
                switch (devNum)
                {
                case 0:
                    pDWordReg = (volatile uint32_t *)(m_PcieCtrl0CfgBase + (where & ~3));
                    result = ResultSuccess();
                    break;
                case 1:
                    pDWordReg = (volatile uint32_t *)(m_PcieCtrl1CfgBase + (where & ~3));
                    result = ResultSuccess();
                    break;
                default:
                    break;
                }
            }
        }
        else
        {
            //[27:24] extended register number
            //[23:16] bus number
            //[15:11] device number
            //[10: 8] function number
            //[ 7: 0] register number
            uint32_t offset = ((where & 0xf00) << 16) | (busNum << 16) | (devNum << 11) |
                ((funcNum & 7) << 8) | (where & 0xfc);
            pDWordReg = (volatile uint32_t *)(m_ConfigSpaceStart + offset);
            result = ResultSuccess();
        }
    }

    *ppOut = pDWordReg;
    return result;
}

Result Tegra21xRootComplex::ReadConfigSpace(BusNumber busNum, DeviceNumber devNum, FunctionNumber funcNum,
                                            int32_t where, uint32_t size, void *pVal)
{
    Result result;
    volatile uint32_t *pDWordReg = NULL;

    if ((result = ComputeConfigSpaceAddress(&pDWordReg, busNum, devNum, funcNum, where)).IsSuccess())
    {
        uint32_t dWordValue = *pDWordReg;
        switch (size)
        {
        case 1:
            *((uint8_t *)pVal) = (uint8_t)((dWordValue >> (8 * (where & 3))) & 0xff);
            break;
        case 2:
            *((uint16_t *)pVal) = (uint16_t)((dWordValue >> (8 * (where & 3))) & 0xffff);
            break;
        case 4:
            *((uint32_t *)pVal) = dWordValue;
            break;
        default:
            result = ResultInvalidRegisterSize();
            break;
        }
    }
    else
    {
        memset(pVal, 0, size);
    }

    return result;
}

Result Tegra21xRootComplex::WriteConfigSpace(BusNumber busNum, DeviceNumber devNum, FunctionNumber funcNum,
                                             int32_t where, uint32_t size, uint32_t val)
{
    Result result;
    volatile uint32_t *pDWordReg = NULL;

    if ((result = ComputeConfigSpaceAddress(&pDWordReg, busNum, devNum, funcNum, where)).IsSuccess())
    {
        uint32_t dWordValue;
        switch (size)
        {
        case 1:
            dWordValue = *pDWordReg & ~(0xff << ((where & 0x3) * 8));
            dWordValue |= (val << ((where & 0x3) * 8));
            *pDWordReg = dWordValue;
            break;
        case 2:
            dWordValue = *pDWordReg & ~(0xffff << ((where & 0x3) * 8));
            dWordValue |= (val << ((where & 0x3) * 8));
            *pDWordReg = dWordValue;
            break;
        case 4:
            *pDWordReg = val;
            break;
        default:
            result = ResultInvalidRegisterSize();
            break;
        }
    }

    return result;
}

void Tegra21xRootComplex::GetRegions(Region *pIo, Region *pPrefetchMem, Region *pNoPrefetchMem)
{
    pIo->resourceAddr            = m_IoSpaceStart;
    pIo->busAddr                 = m_IoSpaceStartPhys;
    pIo->size                    = Params_IoSpaceSize;

    pPrefetchMem->resourceAddr   = m_PrefetchMemStart;
    pPrefetchMem->busAddr        = m_PrefetchMemStartPhys;
    pPrefetchMem->size           = Params_PrefetchMemSize;

    pNoPrefetchMem->resourceAddr = m_NoPrefetchMemStart;
    pNoPrefetchMem->busAddr      = m_NoPrefetchMemStartPhys;
    pNoPrefetchMem->size         = Params_NoPrefetchMemSize;
}


Result Tegra21xRootComplex::AcquireIrq(EndpointFunction *pEp, IrqOptions *pOptions, IrqProfile *pRetProfile)
{
    Result result = ResultBadInternalArgument();
    IrqHandle irqHandle = -1;
    Bus *pBus = pEp->GetBus();
    BridgeFunction *pBf = pBus->GetRootBridgeFunction();
    DeviceNumber devNum = pBf->GetDevNum();
    if (pOptions->irqType == IrqType_Msi)
    {
        result = m_msiIrqMgr.Acquire(pEp, devNum, &irqHandle);
    }
    else if (pOptions->irqType == IrqType_IntX)
    {
        uint8_t pinIndex = pOptions->type.intx.intPin - 1;
        if ((devNum < m_Config.numPorts) &&
            (pinIndex < AFI_INT_TOTAL_VECTORS))
        {
            switch (devNum)
            {
            case 0:
                result = m_port0IntIrqMgrs[pinIndex].Acquire(pEp, devNum, &irqHandle);
                break;
            case 1:
                result = m_port1IntIrqMgrs[pinIndex].Acquire(pEp, devNum, &irqHandle);
                break;
            default:
                break;
            }
        }
    }
    if (result.IsSuccess())
    {
        GetIrqProfile(irqHandle, pRetProfile);
    }
    return result;
}

Result Tegra21xRootComplex::ReleaseIrq(IrqHandle irqHandle)
{
    Result result = ResultBadInternalArgument();
    int32_t poolId = m_msiIrqMgr.GetPoolId(irqHandle);
    if (poolId == msiIrqMgrPoolId)
    {
        for(int32_t index=0; index < nn::pcie::MaxIrqVectorsPerDeviceFunction; index++)
        {
            SetIrqEnable(irqHandle, index, false);
        }
        result = m_msiIrqMgr.Release(irqHandle);
    }
    else if ((poolId >= intPort0IrqMgrBasePoolId) &&
             (poolId < (intPort0IrqMgrBasePoolId + AFI_INT_TOTAL_VECTORS)))
    {
        int32_t index = poolId - intPort0IrqMgrBasePoolId;
        m_port0IntIrqMgrs[index].Release(irqHandle);
    }
    else if ((poolId >= intPort1IrqMgrBasePoolId) &&
             (poolId < (intPort1IrqMgrBasePoolId + AFI_INT_TOTAL_VECTORS)))
    {
        int32_t index = poolId - intPort1IrqMgrBasePoolId;
        m_port1IntIrqMgrs[index].Release(irqHandle);
    }
    return result;
}

Result Tegra21xRootComplex::SetIrqEnable(IrqHandle irqHandle, int32_t index, bool irqEnabled)
{
    Result result = ResultBadInternalArgument();
    int32_t poolId = m_msiIrqMgr.GetPoolId(irqHandle);
    if (poolId == msiIrqMgrPoolId)
    {
        int32_t resourceIndex;
        if ((result = m_msiIrqMgr.IrqHandleLookup(&resourceIndex, irqHandle)).IsSuccess())
        {
            resourceIndex += index;
            if ((resourceIndex < AFI_MSI_TOTAL_VECTORS) && (resourceIndex >= 0))
            {
                uint32_t vecMask = 1 << (resourceIndex % AFI_MSI_VECTORS_PER_REGISTER);
                volatile uint32_t *pVecEnReg = &NN_PCI_AFI_MSI_EN_VEC0_REG + (resourceIndex >> AFI_MSI_VECTORS_PER_REGISTER_POWEROFTWO);
                *pVecEnReg &= ~vecMask;
                *pVecEnReg |= ((irqEnabled) ? vecMask : 0);
                NN_PCIE_LOG_VERBOSE("Tegra21xRootComplex::SetIrqEnable(irqHandle=0x%x, index=%d, irqEnabled=%d): MSI_EN[%d] = 0x%08x\n",
                                    irqHandle, index, irqEnabled,
                                    (resourceIndex >> AFI_MSI_VECTORS_PER_REGISTER_POWEROFTWO),
                                    *pVecEnReg);
                result = ResultSuccess();
            }
        }
    }
    else if ((poolId >= intPort0IrqMgrBasePoolId) &&
             (poolId < (intPort0IrqMgrBasePoolId + AFI_INT_TOTAL_VECTORS)))
    {
        result = ResultSuccess();
    }
    else if ((poolId >= intPort1IrqMgrBasePoolId) &&
             (poolId < (intPort1IrqMgrBasePoolId + AFI_INT_TOTAL_VECTORS)))
    {
        result = ResultSuccess();
    }
    return result;
}

Result Tegra21xRootComplex::GetIrqProfile(IrqHandle irqHandle, IrqProfile *pRetProfile)
{
    Result result = ResultBadInternalArgument();
    int32_t poolId = m_msiIrqMgr.GetPoolId(irqHandle);
    if (poolId == msiIrqMgrPoolId)
    {
        int32_t resourceIndex;
        if ((result = m_msiIrqMgr.IrqHandleLookup(&resourceIndex, irqHandle)).IsSuccess())
        {
            if ((resourceIndex < AFI_MSI_TOTAL_VECTORS) && (resourceIndex >= 0))
            {
                pRetProfile->irqHandle  = irqHandle;
                pRetProfile->irqType    = IrqType_Msi;
                pRetProfile->enableMask = GetEndpointFunctionEnableMask(resourceIndex);
                pRetProfile->type.msi.messageAddress = (BusAddress)NN_PCI_AFI_MSI_AXI_BAR_ST_REG;
                pRetProfile->type.msi.messageData    = (uint16_t)resourceIndex;
                result = ResultSuccess();
            }
        }
    }
    else if ((poolId >= intPort0IrqMgrBasePoolId) &&
             (poolId < (intPort0IrqMgrBasePoolId + AFI_INT_TOTAL_VECTORS)))
    {
        pRetProfile->enableMask = 1;
        pRetProfile->irqHandle  = irqHandle;
        pRetProfile->irqType    = IrqType_IntX;
        result = ResultSuccess();
    }
    else if ((poolId >= intPort1IrqMgrBasePoolId) &&
             (poolId < (intPort1IrqMgrBasePoolId + AFI_INT_TOTAL_VECTORS)))
    {
        pRetProfile->enableMask = 1;
        pRetProfile->irqHandle  = irqHandle;
        pRetProfile->irqType    = IrqType_IntX;
        result = ResultSuccess();
    }
    return result;
}

void Tegra21xRootComplex::HandleInterrupt(IntVector intVector)
{
    bool isError = false;
    switch (intVector)
    {
    case IntVector_Wake:
        isError = true;
        IncrementSaturated(m_LoggedState.globalCounts.intVectorWakeCount);
        break;
    case IntVector_Msi:
        isError = HandleMSIInterrupt();
        IncrementSaturated(m_LoggedState.globalCounts.intVectorMsiCount);
        break;
    case IntVector_Int:
        isError = HandleINTInterrupt();
        IncrementSaturated(m_LoggedState.globalCounts.intVectorIntCount);
        break;
    default:
        break;
    }
    m_pDriver->RequestLoggedStateUpdate(isError);
}

void Tegra21xRootComplex::SetErrorReportingEnable(bool enable)
{
    if (enable)
    {
        NN_PCI_AFI_AFI_INTR_ENABLE_REG  |= AFI_INTR_EN_TGT_DECERR | AFI_INTR_EN_DFPCI_DECERR;
    }
    else
    {
        NN_PCI_AFI_AFI_INTR_ENABLE_REG  &= (~AFI_INTR_EN_TGT_DECERR | AFI_INTR_EN_DFPCI_DECERR);
    }
}

bool Tegra21xRootComplex::HandleINTInterrupt()
{
    bool isError = false;
    const char *debugIntCodeString = NULL;
    bool showFpciAddr = false;
    uint32_t code = NN_PCI_AFI_INTR_CODE_REG & AFI_INTR_CODE_MASK;
    uint32_t sigRegVal = NN_PCI_AFI_INTR_SIGNATURE_REG;
    NN_PCI_AFI_INTR_CODE_REG = 0;
    switch (code)
    {
    case  AFI_INTR_CODE_INI_SLVERR:
        debugIntCodeString = "AFI_INTR_CODE_INI_SLVERR";
        showFpciAddr = true;
        IncrementSaturated((NN_PCI_AFI_INTR_SIGNATURE_REG & AFI_INTR_SIGNATURE_DIR_READ_MASK) ?
            m_LoggedState.globalCounts.afiInitiatorSlaveReadErrorCount : m_LoggedState.globalCounts.afiInitiatorSlaveWriteErrorCount);
        break;
    case  AFI_INTR_CODE_INI_DECERR:
        debugIntCodeString = "AFI_INTR_CODE_INI_DECERR";
        showFpciAddr = true;
        IncrementSaturated((NN_PCI_AFI_INTR_SIGNATURE_REG & AFI_INTR_SIGNATURE_DIR_READ_MASK) ?
            m_LoggedState.globalCounts.afiInitiatorDecodeReadErrorCount : m_LoggedState.globalCounts.afiInitiatorDecodeWriteErrorCount);
        break;
    case  AFI_INTR_CODE_TGT_SLVERR:
        debugIntCodeString = "AFI_INTR_CODE_TGT_SLVERR";
        showFpciAddr = true;
        IncrementSaturated((NN_PCI_AFI_INTR_SIGNATURE_REG & AFI_INTR_SIGNATURE_DIR_READ_MASK) ?
            m_LoggedState.globalCounts.afiTargetSlaveReadErrorCount : m_LoggedState.globalCounts.afiTargetSlaveWriteErrorCount);
        break;
    case  AFI_INTR_CODE_TGT_DECERR:
        debugIntCodeString = "AFI_INTR_CODE_TGT_DECERR";
        showFpciAddr = true;
        IncrementSaturated((NN_PCI_AFI_INTR_SIGNATURE_REG & AFI_INTR_SIGNATURE_DIR_READ_MASK) ?
            m_LoggedState.globalCounts.afiTargetDecodeReadErrorCount : m_LoggedState.globalCounts.afiTargetDecodeWriteErrorCount);
        break;
    case  AFI_INTR_CODE_TGT_WRERR:
        debugIntCodeString = "AFI_INTR_CODE_TGT_WRERR";
        showFpciAddr = true;
        IncrementSaturated(m_LoggedState.globalCounts.afiTargetWriteErrorCount);
        break;
    case  AFI_INTR_CODE_SM_MSG:
        //debugIntCodeString = "AFI_INTR_CODE_SM_MSG";
        isError = HandleSidebandMessageInterrupt(sigRegVal);
        break;
    case  AFI_INTR_CODE_DFPCI_DECERR:
        debugIntCodeString = "AFI_INTR_CODE_DFPCI_DECERR";
        showFpciAddr = true;
        IncrementSaturated((NN_PCI_AFI_INTR_SIGNATURE_REG & AFI_INTR_SIGNATURE_DIR_READ_MASK) ?
            m_LoggedState.globalCounts.afiDfpciDecodeReadErrorCount : m_LoggedState.globalCounts.afiDfpciDecodeWriteErrorCount);
        break;
    case  AFI_INTR_CODE_FPCI_TIMEOUT:
        debugIntCodeString = "AFI_INTR_CODE_FPCI_TIMEOUT";
        showFpciAddr = true;
        IncrementSaturated((NN_PCI_AFI_INTR_SIGNATURE_REG & AFI_INTR_SIGNATURE_DIR_READ_MASK) ?
            m_LoggedState.globalCounts.afiFpciReadTimeoutCount : m_LoggedState.globalCounts.afiFpciWriteTimeoutCount);
        break;
    case  AFI_INTR_CODE_PE_PRSNT_SENSE:
        debugIntCodeString = "AFI_INTR_CODE_PE_PRSNT_SENSE";
        IncrementSaturated(m_LoggedState.globalCounts.afiPePresentSenseCount);
        break;
    case  AFI_INTR_CODE_PE_CLKREQ_SENSE:
        debugIntCodeString = "AFI_INTR_CODE_PE_CLKREQ_SENSE";
        IncrementSaturated(m_LoggedState.globalCounts.afiPeClockReqSenseCount);
        break;
    case  AFI_INTR_CODE_CLKCLAMP_SENSE:
        debugIntCodeString = "AFI_INTR_CODE_CLKCLAMP_SENSE";
        IncrementSaturated(m_LoggedState.globalCounts.afiClockClampSenseCount);
        break;
    case  AFI_INTR_CODE_RDY4PD_SENSE:
        debugIntCodeString = "AFI_INTR_CODE_RDY4PD_SENSE";
        IncrementSaturated(m_LoggedState.globalCounts.afiReadyForPowerDownSenseCount);
        break;
    case  AFI_INTR_P2P_ERROR:
        debugIntCodeString = "AFI_INTR_P2P_ERROR";
        showFpciAddr = true;
        IncrementSaturated(m_LoggedState.globalCounts.afiPeerToPeerErrorCount);
        break;
    default:
        break;
    }

    if (debugIntCodeString != NULL)
    {
        isError = true;
        NN_PCIE_LOG_INFO(" %s: AFI_INTR_SIGNATURE=0x%08x.\n",
                         debugIntCodeString, NN_PCI_AFI_INTR_SIGNATURE_REG);
        if(showFpciAddr)
        {
            NN_PCIE_LOG_INFO("   AFI_UPPER_FPCI_ADDR_REG = 0x%08x.\n",
                             NN_PCI_AFI_UPPER_FPCI_ADDR_REG);
        }
    }

    return isError;
}

bool Tegra21xRootComplex::DispatchIntx(PortIntxIrqMgrType *pMgr)
{
    bool isError = false;
    IrqManagerEntry entryList[MaxFunctionsPerIrq];
    int32_t numEntries = pMgr->GetAllAcquired(entryList);
    if(numEntries > 0)
    {
        IncrementSaturated(m_LoggedState.ports[entryList[0].deviceNumber].irqCount);
    }
    for (int32_t index = 0; index < numEntries; index++)
    {
        IrqManagerEntry *pE = entryList + index;
        EndpointFunction *pEp = (EndpointFunction *)pE->context;
        pEp->HandleIrq(IrqType_IntX, index);
    }
    return isError;
}

bool Tegra21xRootComplex::HandleSidebandMessageInterrupt(uint32_t sigRegVal)
{
    bool isError = false;
    uint32_t regVal32;
    uint32_t afiMsg = NN_PCI_AFI_MSG_REG;
    DeviceNumber devNum = (afiMsg & AFI_MSG_1_MASK) ? 1 : 0;

    NN_UNUSED(sigRegVal);

    if ((AFI_MSG_1_ENDPOINT_PME_MASK | AFI_MSG_0_ENDPOINT_PME_MASK) & afiMsg)
    {
        IncrementSaturated(m_LoggedState.ports[devNum].afiMsgPmeCount);

        // clear PME status
        ReadConfigSpace(0, devNum, 0, T_PCIE2_RP_RSR_OFFSET, sizeof(uint32_t), &regVal32);
        regVal32 |= RP_RSR_PMESTAT_MASK;
        WriteConfigSpace(0, devNum, 0, T_PCIE2_RP_RSR_OFFSET, sizeof(uint32_t), regVal32);
        isError = true;
    }

    // INTx dispatch
    if (AFI_MSG_0_PCIE_INTERRUPT_ALL_MASK & afiMsg)
    {
        IncrementSaturated(m_LoggedState.ports[devNum].afiMsgIntxCount);
        for (uint8_t intPinIndex = 0; intPinIndex < AFI_INT_TOTAL_VECTORS; intPinIndex++)
        {
            if ((AFI_MSG_0_PCIE_INTERRUPT_A_MASK << intPinIndex) & afiMsg)
            {
                isError |= DispatchIntx(m_port0IntIrqMgrs + intPinIndex);
            }
        }
    }
    else if (AFI_MSG_1_PCIE_INTERRUPT_ALL_MASK & afiMsg)
    {
        IncrementSaturated(m_LoggedState.ports[devNum].afiMsgIntxCount);
        for (uint8_t intPinIndex = 0; intPinIndex < AFI_INT_TOTAL_VECTORS; intPinIndex++)
        {
            if ((AFI_MSG_1_PCIE_INTERRUPT_A_MASK << intPinIndex) & afiMsg)
            {
                isError |= DispatchIntx(m_port1IntIrqMgrs + intPinIndex);
            }
        }
    }

    if ((AFI_MSG_1_PCIE_FATAL_ERROR_MASK | AFI_MSG_0_PCIE_FATAL_ERROR_MASK) & afiMsg)
    {
        IncrementSaturated(m_LoggedState.ports[devNum].afiMsgPcieFatalErrorCount);
        isError = true;
    }
    if ((AFI_MSG_1_PCIE_NONFATAL_ERROR_MASK | AFI_MSG_0_PCIE_NONFATAL_ERROR_MASK) & afiMsg)
    {
        IncrementSaturated(m_LoggedState.ports[devNum].afiMsgPcieNonFatalErrorCount);
        isError = true;
    }
    if ((AFI_MSG_1_PCIE_CORRECTABLE_ERROR_MASK | AFI_MSG_0_PCIE_CORRECTABLE_ERROR_MASK) & afiMsg)
    {
        IncrementSaturated(m_LoggedState.ports[devNum].afiMsgPcieCorrectableErrorCount);
        isError = true;
    }
    if ((AFI_MSG_1_PCIE_ROOT_PORT_INTERRUPT_MASK | AFI_MSG_0_PCIE_ROOT_PORT_INTERRUPT_MASK) & afiMsg)
    {
        IncrementSaturated(m_LoggedState.ports[devNum].afiMsgPcieRootPortIntCount);
        isError = true;
    }
    if ((AFI_MSG_1_HOTPLUG_MASK | AFI_MSG_0_HOTPLUG_MASK) & afiMsg)
    {
        IncrementSaturated(m_LoggedState.ports[devNum].afiMsgHotplugCount);
        isError = true;
    }
    return isError;
}

bool Tegra21xRootComplex::HandleMSIInterrupt()
{
    bool isError = false;

    // for all AFI_MSI_VECx_REG registers
    for (int32_t irqBaseNum = 0; irqBaseNum < AFI_MSI_TOTAL_VECTORS; irqBaseNum += AFI_MSI_VECTORS_PER_REGISTER)
    {
        volatile uint32_t *pVecReg = &NN_PCI_AFI_MSI_VEC0_REG + (irqBaseNum >> AFI_MSI_VECTORS_PER_REGISTER_POWEROFTWO);
        uint32_t vectorMask;
        while((vectorMask = *pVecReg) != 0)
        {
            // for all pending bits in the register
            for (int32_t bitNum = 0; (bitNum < AFI_MSI_VECTORS_PER_REGISTER) && (vectorMask != 0); bitNum++)
            {
                uint32_t bitMask = 1 << bitNum;
                if (bitMask & vectorMask)
                {
                    // clear pending
                    *pVecReg = bitMask;
                    vectorMask &= ~bitMask;

                    // dispatch
                    EndpointFunction *pEp = NULL;
                    int32_t offset = -1;
                    DeviceNumber deviceNumber = 0;
                    m_msiIrqMgr.ResourceIndexLookup(&pEp, &offset, &deviceNumber, irqBaseNum + bitNum);
                    if (pEp != NULL)
                    {
                        pEp->HandleIrq(IrqType_Msi, offset);
                        IncrementSaturated(m_LoggedState.ports[deviceNumber].irqCount);
                    }
                }
            }
        }
    }
    return isError;
}

Result Tegra21xRootComplex::Initialize()
{
    size_t size;
    int32_t port0LaneMap, port1LaneMap, pcieLaneMap;
    Result result = ResultSuccess();

    // Retrieve settings
    size = nn::settings::fwdbg::GetSettingsItemValue(
        &m_SocName, sizeof(m_SocName), "pcie", "soc_name"
    );
    NN_ABORT_UNLESS_EQUAL(size, static_cast<size_t>(5));
    //
    size = nn::settings::fwdbg::GetSettingsItemValue(
        &pcieLaneMap, sizeof(pcieLaneMap), "pcie", "pcie_lane_map"
    );
    NN_ABORT_UNLESS_EQUAL(size, sizeof(pcieLaneMap));
    //
    size = nn::settings::fwdbg::GetSettingsItemValue(
        &port0LaneMap, sizeof(port0LaneMap), "pcie", "pcie_port0_lane_map"
    );
    NN_ABORT_UNLESS_EQUAL(size, sizeof(port0LaneMap));
    //
    size = nn::settings::fwdbg::GetSettingsItemValue(
        &port1LaneMap, sizeof(port1LaneMap), "pcie", "pcie_port1_lane_map"
    );
    NN_ABORT_UNLESS_EQUAL(size, sizeof(port1LaneMap));
    //
    size = nn::settings::fwdbg::GetSettingsItemValue(
        &m_UsbLaneMap, sizeof(m_UsbLaneMap), "pcie", "usb_lane_map"
    );
    NN_ABORT_UNLESS_EQUAL(size, sizeof(m_UsbLaneMap));

    // Resolver derived settings
    m_PcieActiveLaneMap    = port0LaneMap | port1LaneMap;
    m_PcieInactiveLaneMap  = pcieLaneMap & ~m_PcieActiveLaneMap;
    if(nn::util::Strncmp(m_SocName, "T210", sizeof(m_SocName)) == 0)
    {
        m_Ldo1Voltage         = 1050000;
        m_RpClkClampThreshold = 0x0f;
        m_PadsRefClkCfg0      = 0x90b890b8;
    }
    else if(nn::util::Strncmp(m_SocName, "T214", sizeof(m_SocName)) == 0)
    {
        m_Ldo1Voltage         = 1000000;
        m_RpClkClampThreshold = 0x1f;
        m_PadsRefClkCfg0      = 0x80b880b8;
    }
    else
    {
        NN_PCIE_ABORT("Unsupport SOC Name: %d", m_SocName);
    }

    // Adjustments made for port enables
    if(!m_Config.ports[0].isEnabled)
    {
        NN_PCIE_LOG_INFO("Tegra21xRootComplex::Initialize() - port 0 (lane 0x%x) powered down.\n", port0LaneMap);
        m_PcieInactiveLaneMap |= port0LaneMap;
    }
    if(!m_Config.ports[1].isEnabled)
    {
        NN_PCIE_LOG_INFO("Tegra21xRootComplex::Initialize() - port 1 (lane 0x%x) powered down.\n", port1LaneMap);
        m_PcieInactiveLaneMap |= port1LaneMap;
    }

    NN_PCIE_LOG_VERBOSE("Summary of Settings:\n");
    NN_PCIE_LOG_VERBOSE("  m_SocName             = %s\n",   m_SocName);
    NN_PCIE_LOG_VERBOSE("  m_PcieActiveLaneMap   = 0x%x\n", m_PcieActiveLaneMap);
    NN_PCIE_LOG_VERBOSE("  m_PcieInactiveLaneMap = 0x%x\n", m_PcieInactiveLaneMap);
    NN_PCIE_LOG_VERBOSE("  port0LaneMap          = 0x%x\n", port0LaneMap);
    NN_PCIE_LOG_VERBOSE("  port1LaneMap          = 0x%x\n", port1LaneMap);
    NN_PCIE_LOG_VERBOSE("  m_UsbLaneMap          = 0x%x\n", m_UsbLaneMap);

    // Run base class initization first
    RootComplex::Initialize();

    // I/O VA Manager
    m_IoVaManager.Initialize(IoVaHeapBase, IoVaHeapSize);

    // Resolve root complex memory map
    ResolveMemMap();

    return result;
}

Result Tegra21xRootComplex::Finalize()
{
    Result result = ResultSuccess();

    // Finalize base class first, as this operation may cause hardware to be touched
    RootComplex::Finalize();


#if !defined NN_PCI_SMMU_DISABLED
    // sMMU teardown
    nn::dd::DetachDeviceAddressSpace(&m_DeviceAddressSpace,
                                     nn::dd::DeviceName::DeviceName_Afi);
    nn::dd::DestroyDeviceAddressSpace(&m_DeviceAddressSpace);
#endif

    m_IoVaManager.Finalize();

    return result;
}

Result Tegra21xRootComplex::Resume()
{
    Result result = ResultSuccess();

    // Re-initialize logged state
    m_LoggedState = InvalidRootComplexLoggedState;

    // Register local event pool
    m_pDriver->m_LocalEventManager.RegisterPool(this, &m_LocalEventPool, HandleLocalEventStatic,
                                                &m_LocalEventStorage[0], sizeof(m_LocalEventStorage[0]),
                                                NN_PCIE_ARRAY_SIZE(m_LocalEventStorage));

    // Turn on power rail and set to SOC specific voltage.
    // T210 needs 1.05V, T214 needs 1.00V
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetVoltageEnabled(nn::pcv::PowerDomain_Max77620_Ldo1, true));
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetVoltageValue(nn::pcv::PowerDomain_Max77620_Ldo1, m_Ldo1Voltage));

    // Put hardware into reset
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_PciExClk, true));
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_Pcie, true));
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_Afi, true));
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_PExUsbPhy, true));
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_XUsbPadCtl, true));

    // Enable PLLe
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetClockEnabled(nn::pcv::Module_Plle, true));

    // Restore AFI and PCIE power
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetPowerEnabled(nn::pcv::Module_Pcie, true));

    // Enable clocks to AFI and PCI
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetClockEnabled(nn::pcv::Module_Afi, true));
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetClockEnabled(nn::pcv::Module_Pcie, true));

    // Release PADCTL and PEXUSB from reset
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_XUsbPadCtl, false));
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_PExUsbPhy, false));

    // Enable UPHY
    NN_PCIE_RETURN_UPON_ERROR(InitPhy());

    // Enable HW power sequencer for PLLe
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetClockEnabled(nn::pcv::Module_PlleHwSeq, true));

    // Various in XUSB
    NN_PCIE_RETURN_UPON_ERROR(InitXusb());

    // Pin multiplexing
    NN_PCIE_RETURN_UPON_ERROR(InitPinMux());

    // Clear AFI reset
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_Afi, false));

    // Program AFI
    NN_PCIE_RETURN_UPON_ERROR(InitAFIRegisters());

    // Clear PCIe reset
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_Pcie, false));

    // Setup ports
    for (DeviceNumber portNum = 0; portNum < m_Config.numPorts; portNum++)
    {
        // Init logged state
        m_LoggedState.ports[portNum].speed             = BusSpeed_Invalid;
        m_LoggedState.ports[portNum].portResetTimeInUs = 0;
        m_LoggedState.ports[portNum].flagSet.Reset();

        // Init port monitor timers
        m_pDriver->m_LocalEventManager.InitStaticEvent(&m_LocalEventPool, m_PortMonitorTimers + portNum);
        m_PortMonitorTimers[portNum].data.eventId = LocalEventId_PortMonitorTimer;

        // Init function registers
        NN_PCIE_RETURN_UPON_ERROR(InitPCIeRegisters(portNum));
        NN_PCIE_RETURN_UPON_ERROR(InitPCIRegisters(portNum));
    }

    // Clear reset to PCIe devices
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_PciExClk, false));

    // Initialize interrupts
    SetInterruptVectorEnable(true);

    // AFI BARs (not the same as PCI device function BARs)
    NN_PCIE_ABORT_UPON_ERROR(InitAFIBars());

    // MSI
    NN_PCIE_ABORT_UPON_ERROR(InitMSI());

    // Setup interrupts
    NN_PCI_AFI_AFI_INTR_ENABLE_REG  = AFI_INTR_EN_INI_SLVERR | AFI_INTR_EN_INI_DECERR |
        AFI_INTR_EN_TGT_SLVERR | AFI_INTR_EN_TGT_WRERR  | AFI_INTR_EN_PE_PRSNT_SENSE |
        AFI_INTR_EN_RDY4PD_SENSE | AFI_INTR_EN_P2P_ERR;
    // AFI_INTR_EN_PE_CLKREQ_SENSE  - Enabling this would cause interrupt with each clock request!
    // AFI_INTR_EN_CLKCLAMP_SENSE   - Enabling this causes interrupt floods, particularly on T214.


    NN_PCI_AFI_SM_INTR_ENABLE_REG = 0xffffffff & ~AFI_SM_INTR_INTA_DEASSERT;
    NN_PCI_AFI_INTR_MASK_REG = AFI_INTR_MASK_INT_MASK | AFI_INTR_MASK_MSI_MASK;

    // No exceptions
    NN_PCI_AFI_FPCI_ERROR_MASKS_REG = 0;

    // Bring up port links
    for (DeviceNumber portNum = 0; portNum < m_Config.numPorts; portNum++)
    {
        if (m_Config.ports[portNum].isEnabled)
        {
            PortControl(portNum, PortCommand_CycleReset);
        }
        else
        {
            DoPortOperation(portNum, PORT_OP_DISABLE_CLOCK);
        }
    }

    RootComplex::Resume();

    return result;
}

Result Tegra21xRootComplex::Suspend()
{
    Result result = ResultSuccess();

    RootComplex::Suspend();

    // Disable interrupts
    NN_PCI_AFI_AFI_INTR_ENABLE_REG = 0;
    NN_PCI_AFI_SM_INTR_ENABLE_REG = 0;
    NN_PCI_AFI_INTR_MASK_REG = 0;
    NN_PCI_AFI_FPCI_ERROR_MASKS_REG = 0;

    // Unregister interrupts
    SetInterruptVectorEnable(false);

    // Override present map to indicate endpoint is absent
#if 0 // not applicable
    for (DeviceNumber portNum = 0; portNum < m_Config.numPorts; portNum++)
    {
        SetPresentMapOverride(portNum, false);
    }
#endif

    // PME_Turn_Off/PME_TO_Ack
    DoPmeTurnoff();

    // Top level PCIe disable
    NN_PCI_AFI_CONFIGURATION_REG &= ~AFI_CONFIGURATION_EN_FPCI;

    // Put IOPHY to IDDQ
    NN_PCI_XUSB_PADCTL_USB3_PAD_MUX_REG &= ~m_PadctrlMuxIddqMask;

    // Put AFI and PCIe blocks into reset
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_Afi, true));
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_Pcie, true));
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_PciExClk, true));

    // Gate AFI and PCIe blocks
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetClockEnabled(nn::pcv::Module_Afi, false));
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetClockEnabled(nn::pcv::Module_Pcie, false));

    // Remove AFI and PCIe power
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetPowerEnabled(nn::pcv::Module_Pcie, false));

    // Power down rail
    NN_PCIE_RETURN_UPON_ERROR(nn::pcv::SetVoltageEnabled(nn::pcv::PowerDomain_Max77620_Ldo1, false));

    // Teardown MSI related
    if (m_pMsiBuffer != NULL)
    {
#if !defined NN_PCI_SMMU_DISABLED
        nn::dd::
            UnmapDeviceAddressSpace(&m_DeviceAddressSpace,
                                    nn::dd::GetCurrentProcessHandle(),
                                    reinterpret_cast<uintptr_t>(m_pMsiBuffer),
                                    Params_MsiBufferSize,
                                    IoVaInternalBase);
#endif
        nn::pcie::detail::MemoryFree(m_pMsiBuffer, "Tegra21xRootComplex Msi buffer");
        m_pMsiBuffer = NULL;
    }

    // Unregister local event pool
    m_pDriver->m_LocalEventManager.UnRegisterPool(&m_LocalEventPool);

    return result;
}

void Tegra21xRootComplex::MonitorPortCounters(DeviceNumber portNum, bool isPortMonitorTimerArmed)
{
    uint32_t     L0sRXcnt, L0sTXcnt, L1cnt, L11cnt, L12cnt, L12aborts, RepLTRVal;
    L0sRXcnt = L0sTXcnt = L1cnt = L11cnt = L12cnt = L12aborts = RepLTRVal = 0;

    ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_LTR_REP_VAL_OFFSET, sizeof(RepLTRVal), &RepLTRVal);
    ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_PRIV_XP_RX_L0S_ENTRY_COUNT_OFFSET, sizeof(L0sRXcnt), &L0sRXcnt);
    ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_PRIV_XP_TX_L0S_ENTRY_COUNT_OFFSET, sizeof(L0sTXcnt), &L0sTXcnt);
    ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_PRIV_XP_L1_ENTRY_COUNT_OFFSET, sizeof(L1cnt), &L1cnt);
    ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_L1_1_ENTRY_COUNT_OFFSET, sizeof(L11cnt), &L11cnt);
    ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_L1_2_ENTRY_COUNT_OFFSET, sizeof(L12cnt), &L12cnt);
    ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_L1_2_ABORT_COUNT_OFFSET, sizeof(L12aborts), &L12aborts);

    NN_PCIE_LOG_INFO("Tegra21xRootComplex::MonitorLxCounters(port %d)\n", portNum);
    //NN_PCIE_LOG_NOPREFIX("    LTR reported value = 0x%04x:%04x\n", RepLTRVal>>16, RepLTRVal&0xffff);
    NN_PCIE_LOG_INFO("    L0s rx count = %u\n", L0sRXcnt);
    NN_PCIE_LOG_INFO("    L0s tx count = %u\n", L0sTXcnt);
    NN_PCIE_LOG_INFO("    L1 count     = %u\n", L1cnt);
    NN_PCIE_LOG_INFO("    L1.1 count   = %u\n", L11cnt);
    NN_PCIE_LOG_INFO("    L1.2 count   = %u\n", L12cnt);
    NN_PCIE_LOG_INFO("    L1.2 aborts  = %u\n", L12aborts);

    if(isPortMonitorTimerArmed)
    {
        m_pDriver->m_LocalEventManager.RestartTimedEvent(m_PortMonitorTimers + portNum, nn::TimeSpan::FromSeconds(10));
    }
    else
    {
        m_pDriver->m_LocalEventManager.StopTimedEvent(m_PortMonitorTimers + portNum);
    }
}

void Tegra21xRootComplex::HandleLocalEvent(LocalEventType *pEvent)
{
    switch(pEvent->data.eventId)
    {
        case LocalEventId_PortMonitorTimer:
        {
            DeviceNumber portNum = static_cast<DeviceNumber>(pEvent - m_PortMonitorTimers);
            MonitorPortCounters(portNum, true);
            break;
        }
        default:
            break;
    }
}


} // end of namespace detail
} // end of namespace driver
} // end of namespace pcie
} // end of namespace nn
