﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file    pcie_SafeHandle.h
 * @brief   PCI Driver: handle management
 */

namespace nn { namespace pcie { namespace driver { namespace detail {

template<class ManagedObjectType, typename HandleType, int32_t maxHandleCount, HandleType sequenceInit>
class SafeHandleManager
{
public:
    SafeHandleManager() : m_RollingHandleSequence(sequenceInit)
    {
        memset(m_HandleTable, 0, sizeof(m_HandleTable));
    }
    ~SafeHandleManager() { }

    Result Assign(ManagedObjectType *pObject, HandleType *pRetHandle)
    {
        Result result = ResultHandleAllocError();
        // find free handle
        for (int32_t index = 0; index < MaxHandleCount; index++)
        {
            if (m_HandleTable[index].pObject == NULL)
            {
                // put handle in global lookup table
                *pRetHandle = MakeHandle(m_RollingHandleSequence, index);
                m_HandleTable[index].pObject = pObject;
                m_HandleTable[index].handle = *pRetHandle;
                result = ResultSuccess();
                break;
            }
        }

        return result;
    }

    Result Assign(ManagedObjectType *pObject, HandleType reservedHandle)
    {
        Result result = ResultSuccess();
        int32_t index = ParseHandle(reservedHandle);
        if ((index < MaxHandleCount) && (index >=0 ))
        {
            if (m_HandleTable[index].pObject == NULL)
            {
                m_HandleTable[index].pObject = pObject;
                m_HandleTable[index].handle = reservedHandle;
            }
            else
            {
                result = ResultHandleAllocError();
            }
        }
        else
        {
            result = ResultInvalidHandle();
        }
        return result;
    }

    Result Release(HandleType handle)
    {
        Result result = ResultInvalidHandle();
        int32_t index = ParseHandle(handle);
        if ((index < MaxHandleCount) && (index >=0))
        {
            if (m_HandleTable[index].handle == handle)
            {
                m_HandleTable[index].handle  = InvalidHandleValue;
                m_HandleTable[index].pObject = NULL;
                result = ResultSuccess();
            }
        }
        return result;
    }

    ManagedObjectType* Get(HandleType handle)
    {
        ManagedObjectType *pObject = NULL;
        int32_t index = ParseHandle(handle);
        if ((index < MaxHandleCount) && (index >=0 ))
        {
            if (m_HandleTable[index].handle == handle)
            {
                pObject = m_HandleTable[index].pObject;
            }
        }
        return pObject;
    }

    int32_t GetNormalizedIndex(HandleType handle)
    {
        int32_t index = ParseHandle(handle);
        if ((index < MaxHandleCount) && (index >=0 ))
        {
            return index;
        }
        return -1;
    }

private:
    const int32_t    TotalHandleBitCount = sizeof(HandleType) * 8;
    const int32_t    SequenceNumberBitCount = TotalHandleBitCount / 2;
    const int32_t    IndexBitCount = TotalHandleBitCount - SequenceNumberBitCount;
    const int32_t    MaxHandleCount = maxHandleCount;
    const HandleType InvalidHandleValue = 0;

    struct HandleTableEntry
    {
        HandleType         handle;
        ManagedObjectType *pObject;
    };

    HandleType MakeHandle(HandleType& sequence, int32_t index)
    {
        sequence++;
        HandleType maskedSequence = static_cast<HandleType>((sequence & ((1<<(SequenceNumberBitCount - 1)) - 1)));
        HandleType createdHandle = InvalidHandleValue;

        if(maskedSequence == InvalidHandleValue)
        {
            sequence = 1;
            maskedSequence = sequence;
        }

        createdHandle = static_cast<HandleType>((maskedSequence << IndexBitCount) | (index & ((1<<(IndexBitCount - 1)) - 1)));

        return createdHandle;
    }

    int32_t ParseHandle(HandleType handle)
    {
        if(handle != InvalidHandleValue)
        {
            return (int32_t)(handle & ((1<<(IndexBitCount - 1)) - 1));
        }
        return -1;
    }

    HandleType       m_RollingHandleSequence;
    HandleTableEntry m_HandleTable[maxHandleCount];
};


} // namespace detail
} // namespace driver
} // namespace pcie
} // namespace nn
