﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "pcie_PrivateIncludes.h"

namespace nn { namespace pcie { namespace driver { namespace detail {

struct LocalEventPoolType;

struct LocalEventHeaderType
{
    nn::util::IntrusiveListNode listNode;
    LocalEventPoolType          *pPool;
    TimeSpan                    time;
    uint8_t                     freeAfterCallback : 1;
    uint8_t                     isAllocated       : 1;
    uint8_t                     isInEventQueue    : 1;
    uint8_t                     isInTimerList     : 1;
};

static const int32_t LOCAL_EVENT_MAX_ARGS = 3;
struct LocalEventDataType
{
    int32_t     eventId;
    int32_t     stateOfOrigin;
    Result      status;
    union
    {
        bool        dataBool;
        uint32_t    dataUint32;
        int32_t     dataInt32;
        void        *pData;
        uint32_t      size;
    }args[LOCAL_EVENT_MAX_ARGS];
};
#define LOCAL_EVENT_DATA_ZERO {0,0,ResultSuccess(),{{0},{0},{0}}}

struct LocalEventType
{
    LocalEventHeaderType    header;
    LocalEventDataType      data;
};

typedef nn::util::IntrusiveList
<LocalEventHeaderType,nn::util::IntrusiveListMemberNodeTraits<LocalEventHeaderType, &LocalEventHeaderType::listNode>> LocalEventListType;

typedef void (*LocalEventCallback)(void *pContext, LocalEventType *pEvent);

struct LocalEventPoolType
{
    LocalEventListType   freeList;
    LocalEventCallback   callback;
    void                 *pContext;
    LocalEventType       *pEventStorage;
    uint32_t               eventSize;
    int32_t              numEvents;
};


class LocalEventManager
{
public:

    LocalEventManager()
    {

    }

    ~LocalEventManager()
    {

    }


    /*
     * These API are called administratively by main thread owner
     */
    void Initialize(nn::os::MultiWaitHolderType *pTimerWaitHolder)
    {
        nn::os::InitializeTimerEvent(&m_TimerEvent, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeMultiWaitHolder(pTimerWaitHolder, &m_TimerEvent);
    }

    void DispatchEvents()
    {
        while (!m_EventQueue.empty())
        {
            LocalEventHeaderType *pEvtHdr = &m_EventQueue.front();
            CompleteEvent(reinterpret_cast<LocalEventType *>(pEvtHdr), true);
        }
    }

    void DispatchTimedEvents()
    {
        nn::os::ClearTimerEvent(&m_TimerEvent);
        m_CurrentTime = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        while (!m_TimerList.empty())
        {
            LocalEventHeaderType *pEvtHdr = &m_TimerList.front();

            // expired?
            if (m_CurrentTime > pEvtHdr->time)
            {
                CompleteEvent(reinterpret_cast<LocalEventType *>(pEvtHdr), true);
            }
            else
            {
                break;
            }
        }
        ScheduleTimed();
    }

    /*
     * These API are called by subordinate pool owners
     */
    void RegisterPool(void *pContext, LocalEventPoolType *pPool, LocalEventCallback callback,
                      LocalEventType *pEventStorage, uint32_t eventSize,
                      int32_t numEvents)
    {
        memset(pPool, 0, sizeof(LocalEventPoolType));
        memset(pEventStorage, 0, eventSize * numEvents);
        pPool->callback      = callback;
        pPool->pContext      = pContext;
        pPool->pEventStorage = pEventStorage;
        pPool->eventSize     = eventSize;
        pPool->numEvents     = numEvents;
        new(&pPool->freeList) LocalEventListType();
        for (int32_t i = 0; i < numEvents; i++)
        {
            LocalEventType *pEvent = (LocalEventType *)((uint8_t *)pEventStorage +
                                                        (eventSize * i));
            pEvent->header.pPool       = pPool;
            pEvent->header.isAllocated = false;
            new(&pEvent->header.listNode) nn::util::IntrusiveListNode();
            pPool->freeList.push_back(pEvent->header);
        }
    }

    void UnRegisterPool(LocalEventPoolType *pPool)
    {
        // unchain any events in queue
        for (int32_t i = 0; i < pPool->numEvents; i++)
        {
            LocalEventType *pEvent = (LocalEventType *)((uint8_t *)pPool->pEventStorage +
                                                        (pPool->eventSize * i));
            CompleteEvent(pEvent, false);
        }

        // unchain any active timer events
        if (!m_TimerList.empty())
        {
            LocalEventListType::iterator it = m_TimerList.begin();
            while (it != m_TimerList.end())
            {
                LocalEventHeaderType *pEvtHdr = &(*it);
                if (pEvtHdr->pPool == pPool)
                {
                    //LocalEventListType::iterator toBeErased = it;
                    ++it;
                    CompleteEvent(reinterpret_cast<LocalEventType *>(pEvtHdr), false);
                }
                else
                {
                    ++it;
                }
            }
        }
        pPool->callback = NULL;
        pPool->pContext  = NULL;
    }

    LocalEventType* AllocEvent(LocalEventPoolType *pPool, bool freeAfterCallback)
    {
        LocalEventType *pEvent = NULL;
        if (!pPool->freeList.empty())
        {
            pEvent = reinterpret_cast<LocalEventType *>(&pPool->freeList.front());
            pPool->freeList.pop_front();
            pEvent->header.isAllocated       = true;
            pEvent->header.isInEventQueue    = false;
            pEvent->header.isInTimerList     = false;
            pEvent->header.time              = 0;
            pEvent->header.freeAfterCallback = freeAfterCallback;
            memset(&pEvent->data, 0, sizeof(pEvent->data));
        }
        return pEvent;
    }

    void FreeEvent(LocalEventType *pEvent)
    {
        CompleteEvent(pEvent, false);
    }

    void InitStaticEvent(LocalEventPoolType *pPool, LocalEventType *pEvent)
    {
        new(&pEvent->header.listNode) nn::util::IntrusiveListNode();
        pEvent->header.isAllocated       = true;
        pEvent->header.isInEventQueue    = false;
        pEvent->header.isInTimerList     = false;
        pEvent->header.time              = 0;
        pEvent->header.freeAfterCallback = false;
        pEvent->header.pPool             = pPool;
    }

    void QueueEvent(LocalEventType *pEvent)
    {
        pEvent->header.isInEventQueue = true;
        m_EventQueue.push_back(pEvent->header);
    }

    Result QueueEvent(LocalEventPoolType *pPool, LocalEventDataType *pData)
    {
        Result result = ResultInternalEventError();
        LocalEventType *pEvent = AllocEvent(pPool, true);
        if (pEvent)
        {
            if (pData != NULL)
            {
                pEvent->data = *pData;

                // any additional for custom type
                if (pPool->eventSize >  sizeof(LocalEventType))
                {
                    uint32_t additionalSize = pPool->eventSize - sizeof(LocalEventType);
                    memcpy(pEvent + 1, pData + 1, additionalSize);
                }
            }
            else
            {
                memset(&pEvent->data, 0, sizeof(pEvent->data));
            }

            QueueEvent(pEvent);
            result = ResultSuccess();
        }
        return result;
    }

    void StartTimedEvent(LocalEventType *pEvent, TimeSpan relativeTime)
    {
        m_CurrentTime = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        if (pEvent->header.isInTimerList || pEvent->header.isInEventQueue)
        {
            NN_PCIE_ABORT("trying to start timer using active event");
        }
        pEvent->header.time = relativeTime + m_CurrentTime;
        if (!m_TimerList.empty())
        {
            for (LocalEventListType::iterator it = m_TimerList.begin(); it != m_TimerList.end(); it++)
            {
                if (it->time > pEvent->header.time)
                {
                    pEvent->header.isInTimerList = true;
                    m_TimerList.insert(it, pEvent->header);
                    break;
                }
            }
        }
        if (!pEvent->header.isInTimerList)
        {
            pEvent->header.isInTimerList = true;
            m_TimerList.push_back(pEvent->header);
        }
        if (&m_TimerList.front() == &pEvent->header)
        {
            ScheduleTimed();
        }
    }

    void StopTimedEvent(LocalEventType *pEvent)
    {
        CompleteEvent(pEvent, false);
    }

    void RestartTimedEvent(LocalEventType *pEvent, TimeSpan relativeTime)
    {
        StopTimedEvent(pEvent);
        StartTimedEvent(pEvent, relativeTime);
    }

    bool TimedEventHasExpired(LocalEventType *pEvent)
    {
        bool expired = false;
        if (pEvent)
        {
            expired = !(pEvent->header.isInTimerList && pEvent->header.isInEventQueue);
        }
        return expired;
    }

private:
    void CompleteEvent(LocalEventType *pEvent, bool doCallback)
    {
        bool eventWasActive = true;
        LocalEventPoolType *pPool = pEvent->header.pPool;
        if (pEvent->header.isInEventQueue)
        {
            pEvent->header.isInEventQueue = false;
            m_EventQueue.erase(m_EventQueue.iterator_to(pEvent->header));
        }
        else if (pEvent->header.isInTimerList)
        {
            pEvent->header.isInTimerList = false;
            m_TimerList.erase(m_TimerList.iterator_to(pEvent->header));
        }
        else
        {
            eventWasActive = false;
        }

        if (eventWasActive)
        {
            if (doCallback)
            {
                pPool->callback(pPool->pContext, pEvent);
            }
            if (pEvent->header.freeAfterCallback)
            {
                pEvent->header.isAllocated       = false;
                pEvent->header.freeAfterCallback = false;
                pPool->freeList.push_back(pEvent->header);
            }
        }
    }

    void ScheduleTimed()
    {
        if ( !m_TimerList.empty())
        {
            TimeSpan timeout = nn::TimeSpan::FromMicroSeconds(100);
            LocalEventHeaderType *pEvtHdr = &m_TimerList.front();
            m_CurrentTime = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());

            if (pEvtHdr->time > m_CurrentTime)
            {
                timeout = pEvtHdr->time - m_CurrentTime;
            }

            //NN_USB_LOG_INFO("ScheduleTimedEvents pEvent->time=0x%llx, m_CurrentTime=0x%llx, timeout=0x%llx\n",
            //                pEvtHdr->time, m_CurrentTime, timeout);
            nn::os::StartOneShotTimerEvent(&m_TimerEvent, timeout);
        }
    }
    nn::os::TimerEventType  m_TimerEvent;
    nn::TimeSpan            m_CurrentTime;
    LocalEventListType      m_EventQueue;
    LocalEventListType      m_TimerList;
};



} // end of namespace detail
} // end of namespace driver
} // end of namespace pcie
} // end of namespace nn

