﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief Irq manager
 *
 * @details Manage logical Irq allocations
 */

namespace nn { namespace pcie { namespace driver { namespace detail {

struct IrqManagerEntry
{
    IrqHandle    irqHandle;
    void*        context;
    DeviceNumber deviceNumber;
};

template<int32_t maxIrqResources, int32_t minimumBlockAllocationCount,  typename ClientContextType, int32_t defaultPoolId>
class IrqManager
{
public:
    IrqManager()
    {
        int32_t resourceIndex;
        m_PoolId = defaultPoolId;
        for (resourceIndex = 0; resourceIndex < maxIrqResources; resourceIndex++)
        {
            Resource *pR = m_ResourceTable + resourceIndex;
            pR->allocated   = false;
            pR->irqHandle   = -1;
        }
    }
    ~IrqManager() { }

    void SetPoolId(int32_t poolId)
    {
        m_PoolId = poolId;
    }

    static int32_t GetPoolId(IrqHandle irqHandle)
    {
        return static_cast<int32_t>(((irqHandle) & 0xff000000) >> 24);
    }

    Result Acquire(ClientContextType context, DeviceNumber deviceNumber, IrqHandle *pRetIrqNum)
    {
        IrqHandle irqHandle = -1;
        Result result = ResultMaximumExceeded();
        int32_t allocatedCount = 0;
        Resource* pResources[maxIrqResources];
        memset(pResources, 0, sizeof(pResources));
        for (int32_t resourceIndex = 0; resourceIndex < maxIrqResources; resourceIndex++)
        {
            Resource *pR = m_ResourceTable + resourceIndex;
            if ( !pR->allocated)
            {
                if(irqHandle < 0)
                { // base of IRQ group
                    irqHandle   = MakeIrqHandle(resourceIndex);
                    *pRetIrqNum = irqHandle;
                }
                pResources[allocatedCount++] = pR;
                pR->allocated    = true;
                pR->irqHandle    = irqHandle;
                pR->context      = context;
                pR->deviceNumber = deviceNumber;
                if(allocatedCount == minimumBlockAllocationCount)
                {
                    result = ResultSuccess();
                    break;
                }
            }
        }

        // Cleanup after allocation failure, although at a higher level this is likely fatal
        if(result.IsFailure())
        {
            for (int32_t resourceIndex = 0; resourceIndex < maxIrqResources; resourceIndex++)
            {
                if(pResources[resourceIndex] != NULL)
                {
                    pResources[resourceIndex]->allocated = false;
                    pResources[resourceIndex]->irqHandle = -1;
                    pResources[resourceIndex]->context   = NULL;
                }
            }
        }

        return result;
    }

    Result Release(IrqHandle irqHandle)
    {
        Result result = ResultBadInternalArgument();
        int32_t resourceIndex = ParseIrqHandle(irqHandle);
        if ((resourceIndex < maxIrqResources) && (resourceIndex >= 0))
        {
            Resource *pR = m_ResourceTable + resourceIndex;
            if (pR->allocated)
            {
                for(int32_t i = 0; i < minimumBlockAllocationCount; i++)
                {
                    pR[i].allocated = false;
                    pR[i].irqHandle = -1;
                    pR[i].context   = NULL;
                }
                result = ResultSuccess();
            }
        }
        return result;
    }

    int32_t GetAllAcquired(IrqManagerEntry *pRetEntries)
    {
        int32_t entries = 0;
        for (int32_t resourceIndex = 0; resourceIndex < maxIrqResources; resourceIndex++)
        {
            Resource *pR = m_ResourceTable + resourceIndex;
            if (pR->allocated)
            {
                pRetEntries[entries].irqHandle    = pR->irqHandle;
                pRetEntries[entries].context      = pR->context;
                pRetEntries[entries].deviceNumber = pR->deviceNumber;
                entries++;
            }
        }
        return entries;
    }

    Result IrqHandleLookup(int32_t *pRetResourceIndex, IrqHandle irqHandle)
    {
        Result result = ResultBadInternalArgument();
        int32_t resourceIndex = ParseIrqHandle(irqHandle);
        if ((resourceIndex < maxIrqResources) && (resourceIndex >= 0))
        {
            Resource *pR = m_ResourceTable + resourceIndex;
            if ((pR->allocated) && (pR->irqHandle == irqHandle))
            {
                *pRetResourceIndex = resourceIndex;
                result = ResultSuccess();
            }
        }
        return result;
    }

    Result ResourceIndexLookup(ClientContextType *pRetClientContext, int32_t* pRetOffset, DeviceNumber* pRetDeviceNumber, int32_t resourceIndex)
    {
        Result result = ResultBadInternalArgument();
        if ((resourceIndex < maxIrqResources) && (resourceIndex >= 0))
        {
            Resource *pR = m_ResourceTable + resourceIndex;
            if (pR->allocated)
            {
                *pRetOffset        = resourceIndex - ParseIrqHandle(pR->irqHandle);
                *pRetClientContext = pR->context;
                *pRetDeviceNumber  = pR->deviceNumber;
                result = ResultSuccess();
            }
        }
        return result;
    }


private:
    struct Resource
    {
        bool                allocated;
        int32_t             clientIndex;
        IrqHandle           irqHandle;
        ClientContextType   context;
        DeviceNumber        deviceNumber;
    };

    int32_t  m_PoolId;
    Resource m_ResourceTable[maxIrqResources];

    IrqHandle MakeIrqHandle(int32_t irqIndex)
    {
        IrqHandle irqHandle = (IrqHandle)(((m_PoolId) << 24) | ((irqIndex)&0xffffff));
        return  irqHandle;
    }

    int32_t ParseIrqHandle(IrqHandle irqHandle)
    {
        return (irqHandle & 0x00ffffff);
    }
};

} // namespace detail
} // namespace driver
} // namespace pcie
} // namespace nn

