﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief Function class
 *
 * @details Functions are the fundemental units of device
 *          functionality.
 *          - There can be one to many functions implemented by
 *            a PCI device.
 *          - There are different kinds of functions, such as
 *          endpoints, bridges, which inherit this base class.
 *          - This class is not to be directly instantiated.
 */
namespace nn { namespace pcie { namespace driver { namespace detail {

class Bus;
class Device;
class Driver;
class RootComplex;

class Function
{

public:
    struct Resource
    {
        nn::util::IntrusiveListNode listNode;
        int32_t         index;
        ResourceAddr    address;
        ResourceSize    size;
        RegionFlagMask  flags;
        Function*       pFunction;
    };
    struct Profile
    {
        FunctionNumber  funcNum;
        uint16_t        vendorId;
        uint16_t        deviceId;
        uint8_t         headerType;
        FullClassCode   classCode;
        uint8_t         revision;
        BusSpeed        speed;
        BusSpeed        capableSpeed;
        PmState         state;
    };
    enum FunctionType
    {
        FunctionType_Invalid,
        FunctionType_Base,
        FunctionType_Endpoint,
        FunctionType_Bridge
    };


    Function(Driver *pDriver, Device *pDevice, Profile *pProf);
    virtual ~Function();


    virtual Result Initialize();
    virtual Result Finalize();
    virtual void Display();
    Result ConfigureResource(Resource *pRes);
    FullClassCode GetClassCode();
    ClassCode Get16BitClassCode();
    BaseClassCode GetBaseClassCode();
    SubClassCode GetSubClassCode();
    uint8_t GetProgClassCode();
    uint16_t GetVendorId();
    uint16_t GetDeviceId();
    uint8_t GetRevision();
    PmState GetPmState();
    BusSpeed GetCurrentSpeed();
    BusSpeed GetCapableSpeed();
    Result ReadSpeed(BusSpeed *pRetCapability, BusSpeed *pRetCurrentSpeed);
    Bus* GetBus();
    bool IsPcie();
    uint8_t GetPCIeFlagsType();
    DeviceNumber GetDevNum();
    Device* GetDevice();
    void GetClassCodeDescription(char *buffer, size_t bufferSize);
    FunctionNumber GetFuncNum();
    FunctionType GetFunctionType();
    uint8_t GetHeaderType();
    void GetFunctionProfile(Profile *pProf);
    Result GetBarProfile(uint8_t bar, DriverBarProfile *pProfile);
    Result GetBarRegion(ResourceAddr *pReturnedBase, uint8_t bar, ResourceSize offset, ResourceSize size);
    int32_t FindCapability(CapabilityId cap);
    int32_t FindExtendedCapability(ExtendedCapabilityId extCap);
    Result ReadPCIeCapabilityWord(int32_t where, uint16_t *pWord);
    Result ReadPCIeCapabilityDWord(int32_t where, uint32_t *pDWord);
    Result WritePCIeCapabilityWord(int32_t where, uint16_t word);
    Result WritePCIeCapabilityDWord(int32_t where, uint32_t dWord, uint32_t writeMask=0xffffffff);
    Result ReadPCIeExtendedCapabilityDWord(int32_t where, uint32_t *dWord);
    Result WritePCIeExtendedCapabilityDWord(int32_t where, uint32_t dWord, uint32_t writeMask=0xffffffff);
    Result GetMaxPayloadSize(int32_t *pRetMaxPayloadSize);
    Result SetMaxPayloadSize(int32_t maxPayloadSize);
    Result GetMaxReadRequestSize(int32_t *pRetMaxSize);
    Result DoPcieFunctionLevelReset();
    Result DoAdvancedFeatureFunctionLevelReset();
    Result DoFunctionLevelReset();
    Result WaitForClearedWordMask(int32_t where, uint16_t mask, uint32_t maxWaitTimeInMs);
    void * operator new(size_t size) NN_NOEXCEPT;
    void operator delete(void *p, size_t size) NN_NOEXCEPT;

    static ClassCode Get16BitClassCode(FullClassCode classCode);
    static Result ReadProfile(Bus *pBus, DeviceNumber devNum, FunctionNumber funcNum,
                              Profile *pProf, uint32_t timeoutInMilliseconds);


    nn::util::IntrusiveListNode m_DevFuncListNode;


protected:
    void SetFunctionType(FunctionType functionType);


    Driver                 *m_pDriver;
    Bus                    *m_pBus;
    RootComplex            *m_pRootComplex;
    Device                 *m_pDevice;


private:
    RegionFlagMask DecodeFlags(uint32_t rawBarVal);
    void DetectResources();
    void DetectAttributes();
    Result InitSpeed();
    int32_t ReadExtConfigSpaceSize();
    int32_t ReadConfigSpaceSize();
    bool IsPCIeCapabilityImplemented(int32_t where);
    uint8_t FindCapabilityStart();
    int32_t FindNextCapability(uint8_t startingOffset, CapabilityId cap);
    int32_t FindNextExtendedCapability(int32_t start, ExtendedCapabilityId extCap);
    Result DoFunctionLevelResetWait();

    // constructor initialized
    uint8_t         m_HeaderType;
    FullClassCode   m_ClassCode;
    FunctionNumber  m_FuncNum;
    uint16_t        m_VendorId;
    uint16_t        m_DeviceId;
    uint8_t         m_Revision;
    PmState         m_PmState;
    FunctionType    m_FunctionType;
    BusSpeed        m_CapableSpeed;
    BusSpeed        m_CurrentSpeed;

    bool            m_IsPCIe;
    int32_t         m_ExpCapOffset;
    uint16_t        m_ExpFlags;
    uint8_t         m_DevCapPayload:3;
    int32_t         m_ConfigSpaceSize;
    int32_t         m_NumResources;
    Resource m_Resources[StandardConfigMaxBaseAddresses];
};



} // namespace detail
} // namespace driver
} // namespace pcie
} // namespace nn



